package com.inzyme.spatiotemporal.common.utils;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.data.redis.serializer.StringRedisSerializer;
import org.springframework.data.redis.support.atomic.RedisAtomicLong;
import org.springframework.stereotype.Component;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.concurrent.TimeUnit;

@Component
@Slf4j
/**
 * Redis缓存工具类
 *
 */

@Service
public class RedisUtil {
	
	 @Autowired
	 private RedisTemplate<String, Object> redisTemplate;
	 
	
 
	/**
	 * 指定缓存失效时间
	 * 
	 * @param key
	 *            键
	 * @param time
	 *            时间(秒)
	 * @return
	 */
	public  boolean expire(String key, long time) {
		try {
			if (time > 0) {
				redisTemplate.expire(key, time, TimeUnit.SECONDS);
			}
			return true;
		} catch (Exception e) {
			log.error("expire() is error : {}", e);
			return false;
		}
	}
 
	/**
	 * 根据key 获取过期时间
	 * 
	 * @param key
	 *            键 不能为null
	 * @return 时间(秒) 返回0代表为永久有效
	 */
	public  long getExpire(String key) {
		return redisTemplate.getExpire(key, TimeUnit.SECONDS);
	}
 
	/**
	 * 判断key是否存在
	 * 
	 * @param key
	 *            键
	 * @return true 存在 false不存在
	 */
	public  boolean hasKey(String key) {
		try {
			return redisTemplate.hasKey(key);
		} catch (Exception e) {
			log.error("hasKey() is error : {}", e);
			return false;
		}
	}
 
	/**
	 * 普通缓存获取
	 * 
	 * @param key
	 *            键
	 * @return 值
	 */
	public  Object get(String key) {

		return key == null ? null : redisTemplate.opsForValue().get(key);
	}
 
	/**
	 * 普通缓存放入
	 * 
	 * @param key
	 *            键
	 * @param value
	 *            值
	 * @return true成功 false失败
	 */
	public  boolean set(String key, Object value) {
		try {
			redisTemplate.opsForValue().set(key, value);
			return true;
		} catch (Exception e) {
			log.error("set() is error : {}", e);
			return false;
		}
 
	}
 
	/**
	 * 普通缓存放入并设置时间
	 * 
	 * @param key
	 *            键
	 * @param value
	 *            值
	 * @param time
	 *            时间(秒) time要大于0 如果time小于等于0 将设置无限期
	 * @return true成功 false 失败
	 */
	public  boolean set(String key, Object value, long time) {
		try {
			if (time > 0) {
				redisTemplate.opsForValue().set(key, value, time, TimeUnit.SECONDS);
			} else {
				set(key, value);
			}
			return true;
		} catch (Exception e) {
			log.error("set() is error : {}", e);
			return false;
		}
	}


	/**
	 * 删除数据
	 * @param key  键
	 */
	public  void remove(String key) {
				redisTemplate.delete(key);
	}
 
	/**
	 * HashGet
	 * 
	 * @param key
	 *            键 不能为null
	 * @param item
	 *            项 不能为null
	 * @return 值
	 */
	public  Object hget(String key, String item) {
		return redisTemplate.opsForHash().get(key, item);
	}
 
	/**
	 * 获取hashKey对应的所有键值
	 * 
	 * @param key
	 *            键
	 * @return 对应的多个键值
	 */
	public  Map<Object, Object> hmget(String key) {
		return redisTemplate.opsForHash().entries(key);
	}
 
	/**
	 * HashSet
	 * 
	 * @param key
	 *            键
	 * @param map
	 *            对应多个键值
	 * @return true 成功 false 失败
	 */
	public  boolean hmset(String key, Map<Object, Object> map) {
		try {
			redisTemplate.opsForHash().putAll(key, map);
			return true;
		} catch (Exception e) {
			log.error("hmset1 is error : {}", e);
			return false;
		}
	}
	

 
	/**
	 * HashSet 并设置时间
	 * 
	 * @param key
	 *            键
	 * @param map
	 *            对应多个键值
	 * @param time
	 *            时间(秒)
	 * @return true成功 false失败
	 */
	public  boolean hmset(String key, Map<String, Object> map, long time) {
		try {
			redisTemplate.opsForHash().putAll(key, map);
			if (time > 0) {
				expire(key, time);
			}
			return true;
		} catch (Exception e) {
			log.error("hmset22 is error : {}", e);
			return false;
		}
	}
 
	/**
	 * 向一张hash表中放入数据,如果不存在将创建
	 * 
	 * @param key
	 *            键
	 * @param item
	 *            项
	 * @param value
	 *            值
	 * @return true 成功 false失败
	 */
	public  boolean hset(String key, String item, Object value) {
		try {
			redisTemplate.opsForHash().put(key, item, value);
			return true;
		} catch (Exception e) {
			log.error("hset() is error : {}", e);
			return false;
		}
	}
 
	/**
	 * 向一张hash表中放入数据,如果不存在将创建
	 * 
	 * @param key
	 *            键
	 * @param item
	 *            项
	 * @param value
	 *            值
	 * @param time
	 *            时间(秒) 注意:如果已存在的hash表有时间,这里将会替换原有的时间
	 * @return true 成功 false失败
	 */
	public  boolean hset(String key, String item, Object value, long time) {
		try {
			redisTemplate.opsForHash().put(key, item, value);
			if (time > 0) {
				expire(key, time);
			}
			return true;
		} catch (Exception e) {
			log.error("hset() is error : {}", e);
			return false;
		}
	}
 
	/**
	 * 删除hash表中的值
	 * 
	 * @param key
	 *            键 不能为null
	 * @param item
	 *            项 可以使多个 不能为null
	 */
	public  void hdel(String key, Object... item) {
		redisTemplate.opsForHash().delete(key, item);
	}
 
	/**
	 * 判断hash表中是否有该项的值
	 * 
	 * @param key
	 *            键 不能为null
	 * @param item
	 *            项 不能为null
	 * @return true 存在 false不存在
	 */
	public  boolean hHasKey(String key, String item) {
		return redisTemplate.opsForHash().hasKey(key, item);
	}
 
	/**
	 * hash递增 如果不存在,就会创建一个 并把新增后的值返回
	 * 
	 * @param key
	 *            键
	 * @param item
	 *            项
	 * @param by
	 *            要增加几(大于0)
	 * @return
	 */
	public  double hincr(String key, String item, double by) {
		return redisTemplate.opsForHash().increment(key, item, by);
	}
 
	/**
	 * hash递减
	 * 
	 * @param key
	 *            键
	 * @param item
	 *            项
	 * @param by
	 *            要减少记(小于0)
	 * @return
	 */
	public  double hdecr(String key, String item, double by) {
		return redisTemplate.opsForHash().increment(key, item, -by);
	}
 
	/**
	 * 根据key获取Set中的所有值
	 * 
	 * @param key
	 *            键
	 * @return
	 */
	public  Set<Object> sGet(String key) {
		try {
			return redisTemplate.opsForSet().members(key);
		} catch (Exception e) {
			log.error("sGet() is error : {}", e);
			return new HashSet<>();
		}
	}
 
	/**
	 * 根据key获取Set中的所有值,特殊序列化问题处理
	 * 
	 * @param key
	 *            键
	 * @return
	 */
	public  Set<Object> sGetByChangeSerializer(String key) {
		try {
			RedisSerializer stringSerializer = new StringRedisSerializer();
			redisTemplate.setKeySerializer(stringSerializer);
			redisTemplate.setValueSerializer(stringSerializer);
			redisTemplate.setHashKeySerializer(stringSerializer);
			redisTemplate.setHashValueSerializer(stringSerializer);
			return redisTemplate.opsForSet().members(key);
		} catch (Exception e) {
			log.error("sGet() is error : {}", e);
			return new HashSet<>();
		}
	}
	
	/**
	 * 根据value从一个set中查询,是否存在
	 * 
	 * @param key
	 *            键
	 * @param value
	 *            值
	 * @return true 存在 false不存在
	 */
	public  boolean sHasKey(String key, Object value) {
		try {
			return redisTemplate.opsForSet().isMember(key, value);
		} catch (Exception e) {
			log.error("sHasKey() is error : {}", e);
			return false;
		}
	}
 
	/**
	 * 将数据放入set缓存
	 * 
	 * @param key
	 *            键
	 * @param values
	 *            值 可以是多个
	 * @return 成功个数
	 */
	public  long sSet(String key, Object... values) {
		try {
			return redisTemplate.opsForSet().add(key, values);
		} catch (Exception e) {
			log.error("sSet() is error : {}", e);
			return 0;
		}
	}
 
	/**
	 * 将set数据放入缓存
	 * 
	 * @param key
	 *            键
	 * @param time
	 *            时间(秒)
	 * @param values
	 *            值 可以是多个
	 * @return 成功个数
	 */
	public  long sSetAndTime(String key, long time, Object... values) {
		try {
			Long count = redisTemplate.opsForSet().add(key, values);
			if (time > 0) {
				expire(key, time);
			}
			return count;
		} catch (Exception e) {
			log.error("sSetAndTime() is error : {}", e);
			return 0;
		}
	}
 
	/**
	 * 获取set缓存的长度
	 * 
	 * @param key
	 *            键
	 * @return
	 */
	public  long sGetSetSize(String key) {
		try {
			return redisTemplate.opsForSet().size(key);
		} catch (Exception e) {
			log.error("sGetSetSize() is error : {}", e);
			return 0;
		}
	}
 
	/**
	 * 移除值为value的
	 * 
	 * @param key
	 *            键
	 * @param values
	 *            值 可以是多个
	 * @return 移除的个数
	 */
	public  long setRemove(String key, Object... values) {
		try {
			return redisTemplate.opsForSet().remove(key, values);
		} catch (Exception e) {
			log.error("setRemove() is error : {}", e);
			return 0;
		}
	}
 
	/**
	 * 获取list缓存的内容
	 * 
	 * @param key
	 *            键
	 * @param start
	 *            开始
	 * @param end
	 *            结束 0 到 -1代表所有值
	 * @return
	 */
	public  List<Object> lGet(String key, long start, long end) {
		try {
			return redisTemplate.opsForList().range(key, start, end);
		} catch (Exception e) {
			log.error("lGet() is error : {}", e);
			return new ArrayList<>();
		}
	}
 
	/**
	 * 获取list缓存的内容
	 * 
	 * @param key
	 *            键
	 * @return
	 */
	public  List<Object> lGet(String key) {
		return lGet(key, 0, -1);
	}
 
	/**
	 * 获取list缓存的长度
	 * 
	 * @param key
	 *            键
	 * @return
	 */
	public  long lGetListSize(String key) {
		try {
			return redisTemplate.opsForList().size(key);
		} catch (Exception e) {
			log.error("lGetListSize() is error : {}", e);
			return 0;
		}
	}
 
	/**
	 * 通过索引 获取list中的值
	 * 
	 * @param key
	 *            键
	 * @param index
	 *            索引 index>=0时， 0 表头，1 第二个元素，依次类推；index<0时，-1，表尾，-2倒数第二个元素，依次类推
	 * @return
	 */
	public  Object lGetIndex(String key, long index) {
		try {
			return redisTemplate.opsForList().index(key, index);
		} catch (Exception e) {
			log.error("lGetIndex() is error : {}", e);
			return null;
		}
	}
 
	/**
	 * 将list放入缓存
	 * 
	 * @param key
	 *            键
	 * @param value
	 *            值
	 * @param time
	 *            时间(秒)
	 * @return
	 */
	public  boolean lSet(String key, Object value) {
		try {
			redisTemplate.opsForList().rightPush(key, value);
			return true;
		} catch (Exception e) {
			log.error("lSet1 is error : {}", e);
			return false;
		}
	}
 
	/**
	 * 将list放入缓存
	 * 
	 * @param key
	 *            键
	 * @param value
	 *            值
	 * @param time
	 *            时间(秒)
	 * @return
	 */
	public  boolean lSet(String key, Object value, long time) {
		try {
			redisTemplate.opsForList().rightPush(key, value);
			if (time > 0) {
				expire(key, time);
			}
			return true;
		} catch (Exception e) {
			log.error("lSet2 is error : {}", e);
			return false;
		}
	}
 
	/**
	 * 将list放入缓存
	 * 
	 * @param key
	 *            键
	 * @param value
	 *            值
	 * @param time
	 *            时间(秒)
	 * @return
	 */
	public  boolean lSet(String key, List<Object> value) {
		try {
			redisTemplate.opsForList().rightPushAll(key, value);
			return true;
		} catch (Exception e) {
			log.error("lSet3 is error : {}", e);
			return false;
		}
	}
 
	/**
	 * 将list放入缓存
	 * 
	 * @param key
	 *            键
	 * @param value
	 *            值
	 * @param time
	 *            时间(秒)
	 * @return
	 */
	public  boolean lSet(String key, List<Object> value, long time) {
		try {
			redisTemplate.opsForList().rightPushAll(key, value);
			if (time > 0) {
				expire(key, time);
			}
			return true;
		} catch (Exception e) {
			log.error("lSet()4 is error : {}", e);
			return false;
		}
	}
 
	/**
	 * 根据索引修改list中的某条数据
	 * 
	 * @param key
	 *            键
	 * @param index
	 *            索引
	 * @param value
	 *            值
	 * @return
	 */
	public  boolean lUpdateIndex(String key, long index, Object value) {
		try {
			redisTemplate.opsForList().set(key, index, value);
			return true;
		} catch (Exception e) {
			log.error("lUpdateIndex() is error : {}", e);
			return false;
		}
	}
 
	/**
	 * 移除N个值为value
	 * 
	 * @param key
	 *            键
	 * @param count
	 *            移除多少个
	 * @param value
	 *            值
	 * @return 移除的个数
	 */
	public  long lRemove(String key, long count, Object value) {
		try {
			return redisTemplate.opsForList().remove(key, count, value);
		} catch (Exception e) {
			log.error("lRemove() is error : {}", e);
			return 0;
		}
	}
 
	/**
	 * 
	 * 获取对象
	 * 
	 * @param key
	 * @param clazz
	 * @return
	 */
	public  <T> T getObjectBean(String key, Class<T> clazz) {
 
		String value = (String) redisTemplate.opsForValue().get(key);
		return parseJson(value, clazz);
	}
 
	/**
	 * 存放对象
	 * 
	 * @param key
	 * @param obj
	 */
	public  <T> void setObjectBean(String key, T obj) {
		if (obj == null) {
			return;
		}
 
		String value = toJson(obj);
		redisTemplate.opsForValue().set(key, value);
	}
 
	/**
	 * 存放对象并返回该对象
	 * 
	 * @param key
	 * @param obj
	 */
	public  <T> T getAndSet(String key, T obj, Class<T> clazz) {
		if (obj == null) {
			return getObjectBean(key, clazz);
		}
		String value = (String) redisTemplate.opsForValue().getAndSet(key, toJson(obj));
		return parseJson(value, clazz);
	}
 
	/**
	 * 原子自增
	 * 
	 * @param key
	 * @return
	 */
	public  long generate(String key) {
		RedisAtomicLong counter = new RedisAtomicLong(key, redisTemplate.getConnectionFactory());
		return counter.incrementAndGet();
	}
 
	/**
	 * A原子自增
	 * 
	 * @param key
	 * @param expireTime
	 * @return
	 */
	public  long generate(String key, Date expireTime) {
		RedisAtomicLong counter = new RedisAtomicLong(key, redisTemplate.getConnectionFactory());
		counter.expireAt(expireTime);
		return counter.incrementAndGet();
	}
 
	/**
	 * 原子自增
	 * 
	 * @param key
	 * @param increment
	 * @return
	 */
	public  long generate(String key, int increment) {
		RedisAtomicLong counter = new RedisAtomicLong(key, redisTemplate.getConnectionFactory());
		return counter.addAndGet(increment);
	}
 
	/**
	 * 原子自增
	 * 
	 * @param key
	 * @param increment
	 * @param expireTime
	 * @return
	 */
	public  long generate(String key, int increment, Date expireTime) {
		RedisAtomicLong counter = new RedisAtomicLong(key, redisTemplate.getConnectionFactory());
		counter.expireAt(expireTime);
		return counter.addAndGet(increment);
	}
 
	/**
	 * 序列化对象为JSONString
	 * 
	 * @param obj
	 * @return
	 */
	public  String toJson(Object obj) {
		return JSON.toJSONString(obj, SerializerFeature.SortField);
	}
 
	/**
	 * 序列化JSON为ObjectBean
	 * 
	 * @param obj
	 * @return
	 */
	public  <T> T parseJson(String json, Class<T> clazz) {
		return JSON.parseObject(json, clazz);
	}
 
	/**
	 * 序列化List对象
	 * 
	 * @param key
	 * @param clazz
	 * @return
	 */
	public  <T> List<T> getList(String key, Class<T> clazz) {
		try {
			Object cache = get(key);
			if (null != cache && StringUtils.isNotBlank(cache.toString())) {
				return JSON.parseArray(JSON.parseArray(cache.toString()).toJSONString(), clazz);
			}
		} catch (Exception e) {
			log.error("redis查询异常，e:", e);
		}
		return new ArrayList<>();
	}


}
