package com.inzyme.spatiotemporal.common.utils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import com.inzyme.spatiotemporal.common.domain.dto.Point;

/**
 * @ClassName: MapAreaUtils
 * @Description: 围栏计算工具
 * @date 2023年3月8日 上午11:10:21
 * 
 * @author Q.JI
 * @version
 * @since JDK 1.8
 */
public class MapAreaUtils {
	/**
	 * 地球半径(米)
	 */
	private static final double EARTH_RADIUS = 6378137.0;

	private static double rad(double d) {
		return d * Math.PI / 180.0;
	}

	/**
	 * 计算是否在圆内
	 * 
	 * @param radius 半径（单位/米）
	 * @param p1     圆心坐标
	 * @param p2     判断点坐标
	 * @return: boolean true:在圆内,false:在圆外
	 * @date: 2021-11-08 09:44:54
	 */
	public static boolean isInCircle(double radius, Point p1, Point p2) {
		double radLat1 = rad(p1.getLat());
		double radLat2 = rad(p2.getLat());
		double a = radLat1 - radLat2;
		double b = rad(p1.getLng()) - rad(p2.getLng());
		double s = 2 * Math.asin(Math.sqrt(
				Math.pow(Math.sin(a / 2), 2) + Math.cos(radLat1) * Math.cos(radLat2) * Math.pow(Math.sin(b / 2), 2)));
		s = s * EARTH_RADIUS;
		s = Math.round(s * 10000) / 10000;
		return !(s > radius);
	}

	/**
	 * 是否在矩形区域内
	 * 
	 * @param lng    测试点经度
	 * @param lat    测试点纬度
	 * @param minLng 矩形四个点中最小经度
	 * @param maxLng 矩形四个点中最大经度
	 * @param minLat 矩形四个点中最小纬度
	 * @param maxLat 矩形四个点中最大纬度
	 * @return boolean true:在矩形内, false:在矩形外
	 * @Title: isInArea
	 */
	public static boolean isInRectangleArea(double lng, double lat, double minLng, double maxLng, double minLat,
			double maxLat) {
		if (isInRange(lat, minLat, maxLat)) {// 如果在纬度的范围内
			if (minLng * maxLng > 0) {
				return isInRange(lng, minLng, maxLng);
			} else {
				if (Math.abs(minLng) + Math.abs(maxLng) < 180) {
					return isInRange(lng, minLng, maxLng);
				} else {
					double left = Math.max(minLng, maxLng);
					double right = Math.min(minLng, maxLng);
					return isInRange(lng, left, 180) || isInRange(lng, right, -180);
				}
			}
		} else {
			return false;
		}
	}

	/**
	 * 是否在矩形区域内
	 * 
	 * @param point     测试点
	 * @param gpsPoints 矩形GPS四个坐标点
	 * @return boolean true:在矩形内, false:在矩形外
	 * @Title: isInArea
	 */
	public static boolean isInRectangleArea(Point point, Point[] gpsPoints) {
		if (gpsPoints.length != 4) {
			return false;
		}
		double[] lats = new double[4];
		double[] lngs = new double[4];
		for (int i = 0; i < gpsPoints.length; i++) {
			lats[i] = gpsPoints[i].getLat();
			lngs[i] = gpsPoints[i].getLng();
		}
		Arrays.sort(lats);
		Arrays.sort(lngs);
		return isInRectangleArea(point.getLat(), point.getLng(), lats[0], lats[3], lngs[0], lngs[3]);
	}

	/**
	 * 判断是否在经纬度范围内
	 * 
	 * @param point
	 * @param left
	 * @param right
	 * @return boolean
	 */
	public static boolean isInRange(double point, double left, double right) {
		return point >= Math.min(left, right) && point <= Math.max(left, right);
	}

	/**
	 * 判断点是否在多边形内
	 * 
	 * @param point 测试点
	 * @param pts   多边形的点
	 * @return boolean true:在多边形内, false:在多边形外
	 * @throws
	 * @Title: IsPointInPoly
	 */
	public static boolean isInPolygon(Point point, List<Point> pts) {

		int N = pts.size();
		boolean boundOrVertex = true;
		int intersectCount = 0;// 交叉点数量
		double precision = 2e-10; // 浮点类型计算时候与0比较时候的容差
		Point p1, p2;// 临近顶点
		Point p = point; // 当前点

		p1 = pts.get(0);
		for (int i = 1; i <= N; ++i) {
			if (p.equals(p1)) {
				return boundOrVertex;
			}

			p2 = pts.get(i % N);
			if (p.getLng() < Math.min(p1.getLng(), p2.getLng()) || p.getLng() > Math.max(p1.getLng(), p2.getLng())) {
				p1 = p2;
				continue;
			}

			// 射线穿过算法
			if (p.getLng() > Math.min(p1.getLng(), p2.getLng()) && p.getLng() < Math.max(p1.getLng(), p2.getLng())) {
				if (p.getLat() <= Math.max(p1.getLat(), p2.getLat())) {
					if (p1.getLng() == p2.getLng() && p.getLat() >= Math.min(p1.getLat(), p2.getLat())) {
						return boundOrVertex;
					}

					if (p1.getLat() == p2.getLat()) {
						if (p1.getLat() == p.getLat()) {
							return boundOrVertex;
						} else {
							++intersectCount;
						}
					} else {
						double xinters = (p.getLng() - p1.getLng()) * (p2.getLat() - p1.getLat())
								/ (p2.getLng() - p1.getLng()) + p1.getLat();
						if (Math.abs(p.getLat() - xinters) < precision) {
							return boundOrVertex;
						}

						if (p.getLat() < xinters) {
							++intersectCount;
						}
					}
				}
			} else {
				if (p.getLng() == p2.getLng() && p.getLat() <= p2.getLat()) {
					Point p3 = pts.get((i + 1) % N);
					if (p.getLng() >= Math.min(p1.getLng(), p3.getLng())
							&& p.getLng() <= Math.max(p1.getLng(), p3.getLng())) {
						++intersectCount;
					} else {
						intersectCount += 2;
					}
				}
			}
			p1 = p2;
		}
		return intersectCount % 2 != 0;
	}
	
	
	/**
	 * 
	 * @Title: pointToSegDist    
	 * @Description: 计算点到线段的距离
	 * @param x		判断点的横坐标
	 * @param y		判断点的纵坐标
	 * @param x1
	 * @param y1
	 * @param x2
	 * @param y2
	 * @return  
	 * double
	 */
	public static double pointToSegDist(double x, double y, double x1, double y1, double x2, double y2) {
		double cross = (x2 - x1) * (x - x1) + (y2 - y1) * (y - y1);
		if (cross <= 0) {
			return Math.sqrt((x - x1) * (x - x1) + (y - y1) * (y - y1));
		}

		double d2 = (x2 - x1) * (x2 - x1) + (y2 - y1) * (y2 - y1);
		if (cross >= d2) {
			return Math.sqrt((x - x2) * (x - x2) + (y - y2) * (y - y2));
		}

		double r = cross / d2;
		double px = x1 + (x2 - x1) * r;
		double py = y1 + (y2 - y1) * r;
		return Math.sqrt((x - px) * (x - px) + (py - y1) * (py - y1));
	}
	
	
	/**
	 * 
	 * @Title: isRouteOffset    
	 * @Description: 路线偏移方法
	 * @param points	路线数据集
	 * @param locationPoint		当前要判断的经纬度坐标
	 * @param distanceTolerate	判定偏移的距离
	 * @return  
	 * boolean	true 表示偏移，false 没有偏移
	 */	
	public static boolean isRouteOffset(List<Point> points, Point locationPoint, double distanceTolerate) {
		/**
		 * 经纬度距离换算 a）在纬度相等的情况下：
		 * 经度每隔0.00001度，距离相差约1米；每隔0.0001度，距离相差约10米；每隔0.001度，距离相差约100米；每隔0.01度，距离相差约1000米；
		 * 每隔0.1度，距离相差约10000米。 b）在经度相等的情况下：
		 * 纬度每隔0.00001度，距离相差约1.1米；每隔0.0001度，距离相差约11米；每隔0.001度，距离相差约111米；每隔0.01度，距离相差约1113米；
		 * 每隔0.1度，距离相差约11132米。
		 */		
		// 将距离转换为与经纬度的对应关系
		Double tolerate = distanceTolerate * 0.00001;
		
		Point leftPoint = null;
		Point rightPoint = null;
		if(null != points && !points.isEmpty()) {
			for(int i = 0; i < points.size() - 1; i++) {
				leftPoint = points.get(i);
				rightPoint = points.get(i + 1);
				
				if (Math.abs(locationPoint.getLng() - leftPoint.getLng()) <= tolerate && Math.abs(locationPoint.getLat() - leftPoint.getLat()) < distanceTolerate) {
					return false;
				} else if (Math.abs(locationPoint.getLng() - rightPoint.getLng()) <= tolerate && Math.abs(locationPoint.getLat() - rightPoint.getLat()) < distanceTolerate) {
					return false;
				} else if (pointToSegDist(locationPoint.getLng(), locationPoint.getLat(), leftPoint.getLng(), leftPoint.getLat(), rightPoint.getLng(), rightPoint.getLat()) < tolerate) {
					return false;
				}
			}
		}
		return true;
	}
	
	
	public static void main(String[] args) {
		List<Point> points = new ArrayList<Point>();
		points.add(new Point(121.366961, 31.190049));
		points.add(new Point(121.366951, 31.190039));
		points.add(new Point(121.366931, 31.190029));
		
		System.out.println("线路偏移与否:" + isRouteOffset(points, new Point(122.366941, 33.345678), 5));
	}
}
