package com.inzyme.spatiotemporal.common.utils;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFCellStyle;
import org.apache.poi.hssf.usermodel.HSSFDateUtil;
import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellType;
import org.apache.poi.ss.usermodel.FillPatternType;
import org.apache.poi.ss.usermodel.HorizontalAlignment;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.xssf.usermodel.XSSFCell;
import org.apache.poi.xssf.usermodel.XSSFCellStyle;
import org.apache.poi.xssf.usermodel.XSSFRow;
import org.apache.poi.xssf.usermodel.XSSFSheet;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.http.MediaType;
import org.springframework.web.multipart.MultipartFile;

import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.inzyme.spatiotemporal.common.annotation.ExcelField;
import com.inzyme.spatiotemporal.common.annotation.ExcelSheet;

import lombok.extern.slf4j.Slf4j;


@Slf4j
public class ExcelUtil {
    private static final Pattern P = Pattern.compile(".0$");

    /**
     * 导出Excel
     *
     * @param sheetName sheet名称
     * @param title     标题
     * @param values    内容
     * @param wb        HSSFWorkbook对象
     * @return
     */
    public static HSSFWorkbook getHSSFWorkbook(String sheetName, String[] title, String[][] values, HSSFWorkbook wb) {

        // 第一步，创建一个HSSFWorkbook，对应一个Excel文件
        if (wb == null) {
            wb = new HSSFWorkbook();
        }

        // 第二步，在workbook中添加一个sheet,对应Excel文件中的sheet
        HSSFSheet sheet = wb.createSheet(sheetName);

        // 第三步，在sheet中添加表头第0行,注意老版本poi对Excel的行数列数有限制
        HSSFRow row = sheet.createRow(0);

        // 第四步，创建单元格，并设置值表头 设置表头居中
        HSSFCellStyle style = wb.createCellStyle();
        style.setAlignment(HorizontalAlignment.CENTER); // 创建一个居中格式

        //声明列对象
        HSSFCell cell = null;

        //创建标题
        for (int i = 0; i < title.length; i++) {
            cell = row.createCell(i);
            cell.setCellValue(title[i]);
            cell.setCellStyle(style);
        }

        //创建内容
        for (int i = 0; i < values.length; i++) {
            row = sheet.createRow(i + 1);
            for (int j = 0; j < values[i].length; j++) {
                //将内容按顺序赋给对应的列对象
                row.createCell(j).setCellValue(values[i][j]);
            }
        }
        return wb;
    }

    /**
     * Excel表头对应Entity属性 解析封装javabean
     *
     * @param clazz      类
     * @param filePath   文件路径
     * @param <T>
     * @return
     * @throws Exception
     */
    public static <T> List<T> readExcelToEntity(Class<T> clazz, String filePath) {
        try {
            File file = new File(filePath);
            // 是否EXCEL文件
            checkFile(file.getName());
            // 兼容新老版本
            Workbook workbook = getWorkBoot(new FileInputStream(file), file.getName());
            // 解析Excel
            return readExcel(clazz, workbook);
        } catch (Exception e) {
            log.error("读取Excel异常：{}", e);
            return null;
        }
    }

    /**
     * Excel表头对应Entity属性 解析封装javabean
     * @param clazz
     * @param file
     * @param <T>
     * @return
     */
    public static <T> List<T> readExcelToEntity(Class<T> clazz, MultipartFile file) {
        try {
            // 是否EXCEL文件
            checkFile(file.getOriginalFilename());
            // 兼容新老版本
            Workbook workbook = getWorkBoot(file.getInputStream(), file.getOriginalFilename());
            // 解析Excel
            return readExcel(clazz, workbook);
        } catch (Exception e) {
            log.error("读取Excel异常：{}", e);
            return null;
        }
    }

    /**
     * 校验是否是Excel文件
     *
     * @param fileName
     * @throws Exception
     */
    private static void checkFile(String fileName) throws Exception {
        if (!StringUtils.isEmpty(fileName) && !(fileName.endsWith(".xlsx") || fileName.endsWith(".xls"))) {
            throw new Exception("不是Excel文件！");
        }
    }

    /**
     * 兼容新老版Excel
     *
     * @param in
     * @param fileName
     * @return
     * @throws IOException
     */
    private static Workbook getWorkBoot(InputStream in, String fileName) throws IOException {
        if (fileName.endsWith(".xlsx")) {
            return new XSSFWorkbook(in);
        } else {
            return new HSSFWorkbook(in);
        }
    }

    /**
     * 读取excel
     * @param clazz
     * @param workbook
     * @param <T>
     * @return
     * @throws IllegalAccessException
     * @throws InstantiationException
     * @throws NoSuchMethodException
     * @throws InvocationTargetException
     */
    private static <T> List<T> readExcel(Class<T> clazz,Workbook workbook) throws IllegalAccessException, InstantiationException, NoSuchMethodException, InvocationTargetException {
        List<T> beans = new ArrayList<T>();
        int sheetNum = workbook.getNumberOfSheets();
        //ExcelSheet里面存放了Sheet名
        ExcelSheet excelSheet= (ExcelSheet) clazz.getAnnotation(ExcelSheet.class);
        String excelSheetName = excelSheet.value();
        // 判断表格名与自定义sheet名是否相同
        for (int sheetIndex = 0; sheetIndex < sheetNum; sheetIndex++) {
            Sheet sheet = workbook.getSheetAt(sheetIndex);
            String sheetName = sheet.getSheetName();
            if (!StringUtils.equals(excelSheetName,sheetName)){
                // 如果不相同，不做导入操作
                continue;
            }
            int firstRowNum = sheet.getFirstRowNum();
            int lastRowNum = sheet.getLastRowNum();
            // 获取表头
            Row head = sheet.getRow(firstRowNum);
            if (head == null) {
                // 判断表头如果为空也不做读取操作
                continue;
            }
            // 获取没一行的开始结束下标，循环要用
            short firstCellNum = head.getFirstCellNum();
            short lastCellNum = head.getLastCellNum();
            // 获取要导入的实体类的字段
            Field[] fields = clazz.getDeclaredFields();
            for (int rowIndex = firstRowNum + 1; rowIndex <= lastRowNum; rowIndex++) {
                // 读取每一行的数据
                Row dataRow = sheet.getRow(rowIndex);
                if (dataRow==null){
                    continue;
                }
                T instance = clazz.newInstance();
                for (int cellIndex = firstCellNum; cellIndex < lastCellNum; cellIndex++) {
                    // 读取表头数据
                    Cell headCell = head.getCell(cellIndex);
                    if (headCell == null) {
                        continue;
                    }
                    // 读取单元格数据
                    Cell cell = dataRow.getCell(cellIndex);
                    headCell.setCellType(CellType.STRING);
                    // 读取数据内容
                    String headName = headCell.getStringCellValue().trim();
                    String data = getCellStringValue(cell);
                    if (StringUtils.isEmpty(headName)||StringUtils.isEmpty(data)) {
                        // 如果为空不做操作
                        continue;
                    }
                    for (Field field : fields) {
                        // 获取自定义属性名
                        ExcelField excelField=field.getAnnotation(ExcelField.class);
                        if (excelField!=null){
                            if (headName.equals(excelField.value())) {
                                String methodName = MethodUtils.setMethodName(field.getName());
                                Method method = clazz.getMethod(methodName, field.getType());
                                method.invoke(instance, convertType(field.getType(), data.trim()));
                                break;
                            }
                        }
                    }
                }
                beans.add(instance);
            }
        }
        return beans;
    }

    /**
     * 读取单元格数据为
     * @param cell
     * @return
     */
    private static String getCellStringValue(Cell cell) {
        if (cell == null) {
            return "";
        } else {
            //.CELL_TYPE_NUMERIC
            if (cell.getCellType() == CellType.NUMERIC) {
                if (HSSFDateUtil.isCellDateFormatted(cell)) {
                    Date date = cell.getDateCellValue();
                    return DateFormatUtils.format(date, "yyyy-MM-dd HH:mm:ss");
                } else {
                    return getRealStringValueOfDouble(cell.getNumericCellValue());
                }
            }
            cell.setCellType(CellType.NUMERIC);
            return cell.getStringCellValue().trim();
        }
    }

    private static String getRealStringValueOfDouble(Double d) {
        String doubleStr = d.toString();
        boolean b = doubleStr.contains("E");
        int indexOfPoint = doubleStr.indexOf('.');
        if (b) {
            int indexOfE = doubleStr.indexOf('E');
            BigInteger xs = new BigInteger(doubleStr.substring(indexOfPoint
                    + BigInteger.ONE.intValue(), indexOfE));
            int pow = Integer.valueOf(doubleStr.substring(indexOfE
                    + BigInteger.ONE.intValue()));
            int xsLen = xs.toByteArray().length;
            int scale = xsLen - pow > 0 ? xsLen - pow : 0;
            doubleStr = String.format("%." + scale + "f", d);
        } else {
            Matcher m = P.matcher(doubleStr);
            if (m.find()) {
                doubleStr = doubleStr.replace(".0", "");
            }
        }
        return doubleStr;
    }


    /**
     * 类型转换
     *
     * @param clazz
     * @param value
     * @return
     */
    private static Object convertType(Class clazz, String value) {
        if (Integer.class == clazz || int.class == clazz) {
            return Integer.valueOf(value);
        }
        if (Short.class == clazz || short.class == clazz) {
            return Short.valueOf(value);
        }
        if (Byte.class == clazz || byte.class == clazz) {
            return Byte.valueOf(value);
        }
        if (Character.class == clazz || char.class == clazz) {
            return value.charAt(0);
        }
        if (Long.class == clazz || long.class == clazz) {
            return Long.valueOf(value);
        }
        if (Float.class == clazz || float.class == clazz) {
            return Float.valueOf(value);
        }
        if (Double.class == clazz || double.class == clazz) {
            return Double.valueOf(value);
        }
        if (Boolean.class == clazz || boolean.class == clazz) {
            return Boolean.valueOf(value.toLowerCase());
        }
        if (BigDecimal.class == clazz) {
            return new BigDecimal(value);
        }
        return value;
    }

    public static Workbook getWorkbook(List list,String type) throws ClassNotFoundException {
        if (StringUtils.equals(ObmsConstants.XLSX,type)){
            return getXSSFWorkbook(list);
        }else if (StringUtils.equals(ObmsConstants.SHEETS,type)){
            // 多sheet表
            return getSheets(list);
        }else {
            return getHSSFWorkbook(list);
        }
    }

    /**
     * 创建excel表格
     * @param list
     * @return
     * @throws ClassNotFoundException
     */
    private static XSSFWorkbook getSheets(List list) throws ClassNotFoundException {
        //创建一个Excel
        XSSFWorkbook wb = new XSSFWorkbook();
        if (CollectionUtils.isEmpty(list)){
            wb.createSheet();
            return wb;
        }
        for (int n=0;n<list.size();n++){
            List l=null;
            Object o = list.get(n);
            if (o!=null){
                 l = (List) o;
            }
            if (CollectionUtils.isEmpty(l)){
                wb.createSheet();
                continue;
            }
            Class c = l.get(0).getClass();
            //ExcelSheet里面存放了Sheet名
            ExcelSheet excelSheet= (ExcelSheet) c.getAnnotation(ExcelSheet.class);
            String tagging = excelSheet.tagging();
            //excelSheet.value()获取sheet名然后设置sheet名
            XSSFSheet sheet =  wb.createSheet(excelSheet.value());
            //index标记表头列号
            int index=0;
            //标记表的行号
            int indexRow=0;
            XSSFRow xssfRow=null;
            XSSFCell xssfCell = null;
            List<Class> classList=new ArrayList();//定义一个存放类的栈
            List<Integer> classListHeadIndex=new ArrayList();//定义一个存放每个类开始位置
            classList.add(c);
            classListHeadIndex.add(0);
            //创建表头
            while (classList.size()!=0){
                int indexTemp=classList.size();//有多少向下扩展的属性
                xssfRow=sheet.createRow(indexRow++);
                //class c里面放有不需要动态增加的表头信息
                for (int k = 0; k<indexTemp ; k++) {
                    Field[] fields = classList.remove(0).getDeclaredFields();
                    int indexCell=classListHeadIndex.remove(0);
                    for (int i = 0; i < fields.length; i++) {
                        ExcelField excelField=fields[i].getAnnotation(ExcelField.class);
                        if (excelField!=null){
                            xssfCell=xssfRow.createCell(indexCell++);
                            //设置单元格内容
                            xssfCell.setCellValue(excelField.value());
                        }
                    }
                    index=indexCell;
                }
            }

            //表内容
            XSSFRow xssfRowData=null;
            for (int i = 0; i <l.size() ; i++) {
                xssfRowData=sheet.createRow(indexRow++);
                Object object=l.get(i);
                Field[] fieldDatas=object.getClass().getDeclaredFields();
                int temp=0;//标记数据项列号
                XSSFCellStyle style = wb.createCellStyle();
                for (int j = 0; j <fieldDatas.length ; j++) {
                    /**
                     * 添加数据项
                     */
                    ExcelField excelField=fieldDatas[j].getAnnotation(ExcelField.class);
                    if (excelField!=null){
                        xssfCell=xssfRowData.createCell(temp++);
                        Object value= getFieldValueByFieldName(fieldDatas[j].getName(),object);
                        setValue(xssfCell,value);//设置值
                        xssfCell.setCellStyle(style);
                        // 设置颜色
                        if (StringUtils.equals(tagging,fieldDatas[j].getName())){
                            if (ObmsConstants.COLOR_MAP.containsKey(value)){
                                //设置预定义填充颜色
                                style.setFillPattern(FillPatternType.SOLID_FOREGROUND);
                                style.setFillForegroundColor(ObmsConstants.COLOR_MAP.get(value));
                            }
                        }
                    }
                }
            }
            setAutoSizeColumn(sheet,index);
        }
        return wb;
    }


    /**
     * 创建excel表格
     * @param list
     * @return
     * @throws ClassNotFoundException
     */
    private static HSSFWorkbook getHSSFWorkbook(List list) throws ClassNotFoundException {
        if (CollectionUtils.isEmpty(list)){
            HSSFWorkbook wb = new HSSFWorkbook();
            wb.createSheet();
            return wb;
        }
        Class c = list.get(0).getClass();
        /**
         * 记录需要动态增加列的列号和列标题
         */
        Map cglab=new HashMap();
        //创建一个Excel
        HSSFWorkbook wb = new HSSFWorkbook();

        //ExcelSheet里面存放了Sheet名
        ExcelSheet excelSheet= (ExcelSheet) c.getAnnotation(ExcelSheet.class);
        //excelSheet.value()获取sheet名然后设置sheet名
        HSSFSheet sheet = wb.createSheet(excelSheet.value());

        //index标记表头列号
        int index=0;
        //标记表的行号
        int indexRow=0;
        HSSFRow hssfRow=null;
        HSSFCell headCell=null;
        List<Class> classList=new ArrayList();//定义一个存放类的栈
        List<Integer> classListHeadIndex=new ArrayList();//定义一个存放每个类开始位置
        classList.add(c);
        classListHeadIndex.add(0);
        //创建表头
        while (classList.size()!=0){
            int indexTemp=classList.size();//有多少向下扩展的属性
            hssfRow=sheet.createRow(indexRow++);
            //class c里面放有不需要动态增加的表头信息
            for (int k = 0; k<indexTemp ; k++) {
                Field[] fields = classList.remove(0).getDeclaredFields();
                int indexCell=classListHeadIndex.remove(0);
                for (int i = 0; i < fields.length; i++) {
                    ExcelField excelField=fields[i].getAnnotation(ExcelField.class);
                    if (excelField!=null){
                        headCell=hssfRow.createCell(indexCell++);
                        //设置单元格内容
                        headCell.setCellValue(excelField.value());
                        if (!excelField.className().equals("")){
                            Class classTemp=Class.forName(excelField.className());
                            classList.add(classTemp);//向栈中添加一个类
                            classListHeadIndex.add(indexCell-1);
                            int count=countLeaf(classTemp);
                            //创建单元格合并
                            CellRangeAddress cellRangeAddress=new CellRangeAddress(indexRow-1,indexRow-1,indexCell-1,indexCell+count-2);
                            sheet.addMergedRegion(cellRangeAddress);
                            indexCell=indexCell+count-1;
                        }
                    }
                }
                index=indexCell;
            }
        }

        //表内容
        HSSFRow hssfRowData=null;
        for (int i = 0; i <list.size() ; i++) {
            hssfRowData=sheet.createRow(indexRow++);
            Object object=list.get(i);
            Field[] fieldDatas=object.getClass().getDeclaredFields();
            int temp=0;//标记数据项列号
            HSSFCell hssfCell=null;
            for (int j = 0; j <fieldDatas.length ; j++) {
                /**
                 * 添加数据项
                 */
                ExcelField excelField=fieldDatas[j].getAnnotation(ExcelField.class);
                if (excelField!=null){
                    if (!excelField.className().equals("")){
                        List valueList=new ArrayList();
                        Object o=getFieldValueByFieldName(fieldDatas[j].getName(),object);
                        if(o==null){
                            for (int k = 0; k < countLeaf(fieldDatas[j].getType().getClass()); k++) {
                                valueList.add("");
                            }
                        }else {
                            valueLeaf(valueList,o);
                        }
                        for (int k = 0; k < valueList.size(); k++) {
                            hssfCell=hssfRowData.createCell(temp++);
                            Object value=valueList.get(k);
                            setValue(hssfCell,value);//设置值
                        }
                    }else {
                        hssfCell=hssfRowData.createCell(temp++);
                        Object value= getFieldValueByFieldName(fieldDatas[j].getName(),object);
                        setValue(hssfCell,value);//设置值
                    }
                }
            }

        }
        setAutoSizeColumn(sheet,index);
        return wb;
    }


    /**
     * 创建excel表格. xlsx格式
     * @param list
     * @return
     * @throws ClassNotFoundException
     */
    private static XSSFWorkbook getXSSFWorkbook(List list) throws ClassNotFoundException {
        if (CollectionUtils.isEmpty(list)){
            XSSFWorkbook wb = new XSSFWorkbook();
            wb.createSheet();
            return wb;
        }
        Class c = list.get(0).getClass();
        /**
         * 记录需要动态增加列的列号和列标题
         */
        Map cglab=new HashMap();
        //创建一个Excel
        XSSFWorkbook wb = new XSSFWorkbook();

        //ExcelSheet里面存放了Sheet名
       ExcelSheet excelSheet= (ExcelSheet) c.getAnnotation(ExcelSheet.class);
        String tagging = excelSheet.tagging();
        //excelSheet.value()获取sheet名然后设置sheet名
       XSSFSheet sheet =  wb.createSheet(excelSheet.value());
        //index标记表头列号
        int index=0;
        //标记表的行号
        int indexRow=0;
        XSSFRow xssfRow=null;
        XSSFCell xssfCell = null;
        List<Class> classList=new ArrayList();//定义一个存放类的栈
        List<Integer> classListHeadIndex=new ArrayList();//定义一个存放每个类开始位置
        classList.add(c);
        classListHeadIndex.add(0);
        //创建表头
        while (classList.size()!=0){
            int indexTemp=classList.size();//有多少向下扩展的属性
            xssfRow=sheet.createRow(indexRow++);
            //class c里面放有不需要动态增加的表头信息
            for (int k = 0; k<indexTemp ; k++) {
                Field[] fields = classList.remove(0).getDeclaredFields();
                int indexCell=classListHeadIndex.remove(0);
                for (int i = 0; i < fields.length; i++) {
                    ExcelField excelField=fields[i].getAnnotation(ExcelField.class);
                    if (excelField!=null){
                        xssfCell=xssfRow.createCell(indexCell++);
                        //设置单元格内容
                        xssfCell.setCellValue(excelField.value());
                        if (!excelField.className().equals("")){
                            Class classTemp=Class.forName(excelField.className());
                            classList.add(classTemp);//向栈中添加一个类
                            classListHeadIndex.add(indexCell-1);
                            int count=countLeaf(classTemp);
                            //创建单元格合并
                            CellRangeAddress cellRangeAddress=new CellRangeAddress(indexRow-1,indexRow-1,indexCell-1,indexCell+count-2);
                            sheet.addMergedRegion(cellRangeAddress);
                            indexCell=indexCell+count-1;
                        }
                    }
                }
                index=indexCell;
            }
        }

        //表内容
        XSSFRow xssfRowData=null;
        for (int i = 0; i <list.size() ; i++) {
            xssfRowData=sheet.createRow(indexRow++);
            Object object=list.get(i);
            Field[] fieldDatas=object.getClass().getDeclaredFields();
            int temp=0;//标记数据项列号
            XSSFCellStyle style = wb.createCellStyle();
            for (int j = 0; j <fieldDatas.length ; j++) {
                /**
                 * 添加数据项
                 */
                ExcelField excelField=fieldDatas[j].getAnnotation(ExcelField.class);
                if (excelField!=null){
                    if (!excelField.className().equals("")){
                        List valueList=new ArrayList();
                        Object o=getFieldValueByFieldName(fieldDatas[j].getName(),object);
                        if(o==null){
                            for (int k = 0; k < countLeaf(fieldDatas[j].getType().getClass()); k++) {
                                valueList.add("");
                            }
                        }else {
                            valueLeaf(valueList,o);
                        }
                        for (int k = 0; k < valueList.size(); k++) {
                            xssfCell=xssfRowData.createCell(temp++);
                            Object value=valueList.get(k);
                            setValue(xssfCell,value);//设置值
                        }
                    }else {
                        xssfCell=xssfRowData.createCell(temp++);
                        Object value= getFieldValueByFieldName(fieldDatas[j].getName(),object);
                        /*if(value != null && !value.equals(""))
                            value.toString();*/
                        setValue(xssfCell,value);//设置值
                        xssfCell.setCellStyle(style);
                        // 设置颜色
                        /*if (StringUtils.equals(tagging,fieldDatas[j].getName())){
                            if (ObmsConstants.COLOR_MAP.containsKey(value)){
                                //设置预定义填充颜色
                                style.setFillPattern(FillPatternType.SOLID_FOREGROUND);
                                style.setFillForegroundColor(ObmsConstants.COLOR_MAP.get(value));
                            }
                        }*/
                    }
                }
            }
        }
        setAutoSizeColumn(sheet,index);
        return wb;
    }


    /**
     * 设置表格内容的值
     * @param hssfCell 单元格对象
     * @param value 单元格的值
     */
    private static void setValue(HSSFCell hssfCell,Object value){
        if (value instanceof String) {
            hssfCell.setCellValue(value.toString());
        }else if (value instanceof Integer) {
            hssfCell.setCellValue((Integer) value);
        } else if (value instanceof Double) {
            hssfCell.setCellValue((Double) value);
        } else if (value instanceof Boolean) {
            hssfCell.setCellValue((Boolean) value);
        } else if (value instanceof Float) {
            hssfCell.setCellValue((Float) value);
        } else if (value instanceof Short) {
            hssfCell.setCellValue((Short) value);
        }else if (value instanceof Long) {
            hssfCell.setCellValue(value.toString());
        } else if (value instanceof Character) {
            hssfCell.setCellValue((Character) value);
        }
    }
    /**
     * 设置表格内容的值
     * @param xssfCell 单元格对象
     * @param value 单元格的值
     */
    private static void setValue(XSSFCell xssfCell,Object value){
        if (value instanceof String) {
            xssfCell.setCellValue(value.toString());
        }else if (value instanceof Integer) {
            xssfCell.setCellValue((Integer) value);
        } else if (value instanceof Double) {
            xssfCell.setCellValue((Double) value);
        } else if (value instanceof Boolean) {
            xssfCell.setCellValue((Boolean) value);
        } else if (value instanceof Float) {
            xssfCell.setCellValue((Float) value);
        } else if (value instanceof Short) {
            xssfCell.setCellValue((Short) value);
        }else if (value instanceof Long) {
            xssfCell.setCellValue((Long) value);
        } else if (value instanceof Character) {
            xssfCell.setCellValue((Character) value);
        }
    }

    /**
     * 设置自动列宽
     * @param hssfSheet
     * @param size
     */
    private static void setAutoSizeColumn(HSSFSheet hssfSheet,int size){
        for (int i = 0; i <size ; i++) {
            hssfSheet.autoSizeColumn(i, true);
        }
    }

    /**
     * 设置自动列宽
     * @param xssfSheet
     * @param size
     */
    private static void setAutoSizeColumn(XSSFSheet xssfSheet,int size){
        for (int i = 0; i <size ; i++) {
            xssfSheet.autoSizeColumn(i, true);
        }
    }

    /**
     * 获取这个类下带ExcelField注解的子节点个数
     * @param c
     * @return
     * @throws ClassNotFoundException
     */
    private static int countLeaf(Class c) throws ClassNotFoundException {
        Field fields[]=c.getDeclaredFields();
        int ans=0;
        for (int i = 0; i <fields.length ; i++) {
            ExcelField excelField=fields[i].getAnnotation(ExcelField.class);
            if(excelField!=null){
                if (!excelField.className().equals("")){
                    ans+=countLeaf(Class.forName(excelField.className()));
                }else {
                    ans++;
                }
            }

        }
        return ans;
    }

    /**
     * 获取叶子节点的值
     * @param list 传入的list
     * @param o 传入的object
     */
    private static void valueLeaf(List list,Object o){
        Class c=o.getClass();
        Field fields[]=c.getDeclaredFields();
        for (int i = 0; i <fields.length ; i++) {
            ExcelField excelField =fields[i].getAnnotation(ExcelField.class);
            if (excelField!=null){
                if (!excelField.className().equals("")){
                    valueLeaf(list,getFieldValueByFieldName(fields[i].getName(), o));
                }else {
                    list.add(getFieldValueByFieldName(fields[i].getName(), o));
                }

            }
        }
    }
    /**
     * 获取map
     * @param list
     * @return
     */
    public static Map getMap(List list){
        Map map =new LinkedHashMap();
        String obExcelField;
        String obFieldValue;
        List obExcelFieldList=new ArrayList();
        List obobFieldValueList=new ArrayList();
        for (int i = 0; i < list.size(); i++) {
            Object ob=list.get(i);
            Class c=ob.getClass();
            Field[] fields=c.getDeclaredFields();

            for (int j = 0; j < fields.length; j++) {
                String tempName=fields[j].getName();
                if (tempName.equals("obExcelField")){
                    obExcelFieldList.add((String) getFieldValueByFieldName(fields[j].getName(),ob));
                }
                if (tempName.equals("obFieldValue")){
                    obobFieldValueList.add((String) getFieldValueByFieldName(fields[j].getName(),ob));
                }
            }
        }
        map.put("obExcelField",obExcelFieldList);
        map.put("obFieldValue",obobFieldValueList);
        return map;
    }
    /**
     * 通过属性名字，调用相应的Get方法获取属性值
     * @param object
     * @param fieldName 属性名字
     * @return
     */
    private static Object getFieldValueByFieldName(String fieldName, Object object) {
        Class c=object.getClass();
        try {
            Field field = object.getClass().getDeclaredField(fieldName);
            //设置对象的访问权限，保证对private的属性的访问
            String s=fieldName.substring(0,1).toUpperCase()+fieldName.substring(1);
            Method method=c.getMethod("get"+s);
            return method.invoke(object);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 浏览器下载excel
     * @param fileName
     * @param wb
     * @param response
     */

    public static  void  buildExcelDocument(String fileName, Workbook wb, HttpServletResponse response){
        try {
            response.setContentType(MediaType.APPLICATION_OCTET_STREAM_VALUE);
            response.setHeader("Content-Disposition", "attachment;filename="+ URLEncoder.encode(fileName, "utf-8"));
            response.flushBuffer();
            wb.write(response.getOutputStream());
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

}
