package com.inzyme.spatiotemporal.common.utils;

import java.math.BigDecimal;
import java.util.Calendar;
import java.util.Date;

/**
 * @ClassName: CalcUtils
 * @Description: 计算工具类
 * @date 2023年2月15日 上午11:20:43
 * 
 * @author Q.JI
 * @version
 * @since JDK 1.8
 */
public class CalcUtils {

	/**
	 * @param obj
	 * @return int
	 * @Title: toInt
	 * @Description: toInt
	 */
	public static int toInt(Object obj) {
		Integer rtn = 0;
		try {
			if (null != obj) {
				rtn = Integer.valueOf(obj.toString());
			}
		} catch (Exception ex) {
			rtn = 0;
		}
		return rtn;
	}

	/**
	 * @param obj
	 * @return double
	 * @Title: toDouble
	 * @Description: toDouble
	 */
	public static double toDouble(Object obj) {
		double rtn = 0d;
		try {
			if (null != obj) {
				rtn = Double.valueOf(obj.toString());
			}
		} catch (Exception ex) {
		}
		return rtn;
	}

	/**
	 * @param str
	 * @return long
	 * @Title: toLong
	 * @Description: toLong
	 */
	public static long toLong(Object str) {
		Long rtn = 0l;
		try {
			rtn = Long.valueOf(str.toString());
		} catch (Exception ex) {
		}
		return rtn;
	}

	/**
	 * @param str
	 * @return Float
	 * @Title: toFloat
	 * @Description: toFloat
	 */
	public static Float toFloat(Object str) {
		Float rtn = 0f;
		try {
			rtn = Float.valueOf(str.toString());
		} catch (Exception ex) {
		}
		return rtn;
	}

	/**
	 * 
	 * add:(提供精确的加法运算). <br/>
	 * 
	 * @param v1
	 * @param v2
	 * @return double 两个参数的和
	 * @since 1.0
	 */
	public static double add(double v1, double v2) {
		BigDecimal b1 = new BigDecimal(Double.toString(v1));
		BigDecimal b2 = new BigDecimal(Double.toString(v2));
		return b1.add(b2).doubleValue();
	}

	/**
	 * 
	 * subtract:(提供精确的减法运算). <br/>
	 * 
	 * @param v1
	 * @param v2
	 * @return double
	 * @since 1.0
	 */
	public static double subtract(double v1, double v2) {
		BigDecimal b1 = new BigDecimal(Double.toString(v1));
		BigDecimal b2 = new BigDecimal(Double.toString(v2));
		return b1.subtract(b2).doubleValue();
	}

	/**
	 * 
	 * mul:(提供精确的乘法运算，默认保留2位小数). <br/>
	 * 
	 * @param v1
	 * @param v2
	 * @return double 两个参数的积
	 * @since 1.0
	 */
	public static double mul(double v1, double v2) {
		return mul(v1, v2, 2);
	}

	/**
	 * 
	 * @Title: mul @Description: (提供精确的乘法运算，保留n位小数). @param v1 @param v2 @param
	 *         n @return @return: double @throws
	 */
	public static double mul(double v1, double v2, int n) {
		BigDecimal b1 = new BigDecimal(Double.toString(v1));
		BigDecimal b2 = new BigDecimal(Double.toString(v2));
		return round(b1.multiply(b2).doubleValue(), n);
	}

	/**
	 * 
	 * div:(精确的除法运算，当发生除不尽的情况时，默认保留两位小数). <br/>
	 * 
	 * @param v1
	 * @param v2
	 * @return double 两个参数的商
	 * @since 1.0
	 */
	public static double div(double v1, double v2) {
		return div(v1, v2, 2);
	}

	/**
	 * 
	 * div:(精确的除法运算。当发生除不尽的情况时，由scale参数指 定精度，以后的数字四舍五入). <br/>
	 * 
	 * @param v1
	 * @param v2
	 * @param scale 表示表示需要精确到小数点以后几位。
	 * @return double
	 * @since 1.0
	 */
	public static double div(double v1, double v2, int scale) {
		if (scale < 0) {
			throw new IllegalArgumentException("The scale must be a positive integer or zero");
		}
		if (v2 == 0) {
			return 0;
		}
		BigDecimal b1 = new BigDecimal(Double.toString(v1));
		BigDecimal b2 = new BigDecimal(Double.toString(v2));
		return b1.divide(b2, scale, BigDecimal.ROUND_HALF_UP).doubleValue();
	}

	/**
	 * 
	 * @Title: round2
	 * @Description: 四舍五入取整
	 * @param value
	 * @param inum
	 * @return double
	 */
	public static int trunc(double value) {
		BigDecimal big = new BigDecimal(Double.toString(value)).setScale(0, BigDecimal.ROUND_HALF_UP);
		return big.intValue();
	}

	/**
	 * 
	 * round:(保留小数的四舍五入). <br/>
	 * 
	 * @param value
	 * @param inum
	 * @return float
	 * @since 1.0
	 */
	public static float round(float value, int inum) {
		double num = Math.pow(10, inum);
		return (float) (Math.round(value * num) / num);
	}

	/**
	 * 
	 * round:(保留小数的四舍五入). <br/>
	 * 
	 * @param value
	 * @param inum
	 * @return double
	 * @since 1.0
	 */
	public static double round(double value, int inum) {
		double num = Math.pow(10, inum);
		return (double) (Math.round(value * num) / num);
	}

	/**
	 * 
	 * @Title: between
	 * @Description: 是否在区间内 （下上限闭区间）
	 * @param lower
	 * @param upper
	 * @param value
	 * @return boolean
	 */
	public static boolean between(double lower, double upper, double value) {
		if (lower <= value && value <= upper) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * 
	 * @Title: between1
	 * @Description: 是否在区间内 （下限闭、上限开区间）
	 * @param lower
	 * @param upper
	 * @param value
	 * @return boolean
	 */
	public static boolean between1(double lower, double upper, double value) {
		if (lower <= value && value < upper) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * 
	 * @Title: alarmLayer
	 * @Description: getAlarmLayer
	 * @param aStatus
	 * @param bStatus
	 * @param cStatus
	 * @return String
	 */
	public static String alarmLayer(Boolean aStatus, Boolean bStatus, Boolean cStatus) {
		return aStatus ? "A" : (bStatus ? "B" : "C");
	}

	/**
	 * 
	 * @Title: isAlarmUpgrade
	 * @Description: 判断是否告警升级
	 * @param newLayer
	 * @param oldLayer
	 * @return int -1:告警降级；0：告警同级；1：告警升级
	 */
	public static int isAlarmUpgrade(String newLayer, String oldLayer) {
		int flg = -1;
		if (StringUtils.isEmpty(newLayer) || StringUtils.isEmpty(oldLayer)
				|| newLayer.codePointCount(0, newLayer.length()) != 1
				|| oldLayer.codePointCount(0, oldLayer.length()) != 1) {
			return flg;
		} else {
			int nl = (int) newLayer.charAt(0);
			int ol = (int) oldLayer.charAt(0);
			flg = (100 - nl > 100 - ol) ? 1 : (100 - nl == 100 - ol) ? 0 : -1;
		}
		return flg;
	}

	/**
	 * 
	 * @Title: calcHoursBetween
	 * @Description: 计算两个DATE间的小时数，去掉每天1:00:00~07:59:59间的时长
	 * @param redisDate
	 * @param msgDate
	 * @return double
	 */
	private static double calcHoursBetween(int cfgStartHour, int cfgEndHour, Date redisDate, Date msgDate) {
		double hoursBetweenDates = 0;

		try {
			if (redisDate.before(msgDate)) {
				Calendar redisCalendar = Calendar.getInstance();
				redisCalendar.setTime(redisDate);
				int redisHour = redisCalendar.get(Calendar.HOUR_OF_DAY);
				int redisYearDay = redisCalendar.get(Calendar.DAY_OF_YEAR);

				Calendar msgCalendar = Calendar.getInstance();
				msgCalendar.setTime(msgDate);
				int msgHour = msgCalendar.get(Calendar.HOUR_OF_DAY);
				int msgYearDay = msgCalendar.get(Calendar.DAY_OF_YEAR);

				long diffInMilliseconds = Math.abs(msgDate.getTime() - redisDate.getTime());
				double realHours = CalcUtils.div(diffInMilliseconds, (60 * 60 * 1000), 2);
				int days = (int) realHours / 24;

				boolean redisRange = false;
				boolean msgRange = false;

				if (cfgStartHour <= redisHour && redisHour < cfgEndHour)
					redisRange = true;
				if (cfgStartHour <= msgHour && msgHour < cfgEndHour)
					msgRange = true;

				if (redisRange) {
					redisCalendar.set(Calendar.HOUR_OF_DAY, cfgEndHour);
					redisCalendar.set(Calendar.MINUTE, 0);
					redisCalendar.set(Calendar.SECOND, 0);
//            		System.out.println("fix redis date:" + redisCalendar.getTime());
				}
				if (msgRange) {
					msgCalendar.set(Calendar.HOUR_OF_DAY, cfgStartHour);
					msgCalendar.set(Calendar.MINUTE, 0);
					msgCalendar.set(Calendar.SECOND, 0);
//            		System.out.println("fix message date:" + msgCalendar.getTime());
				}

				if (redisYearDay == msgYearDay) {
					if (redisRange && msgRange) {
						return hoursBetweenDates;
					}
					if (redisHour < cfgStartHour && msgHour >= cfgEndHour) {
						days = days + 1;
					}
				} else {
					if ((msgYearDay - redisYearDay) == days) {
						if (redisRange && msgRange) {
							days = days - 1;
						}
						if (redisHour < cfgStartHour && msgHour >= cfgEndHour) {
							days = days + 1;
						}
					} else {
						if (redisHour >= cfgEndHour && msgHour >= cfgEndHour) {
							days = days + 1;
						}
						if (redisHour < cfgStartHour && msgHour < cfgStartHour) {
							days = days + 1;
						}
					}
				}

				long fixDiffInMilliseconds = Math.abs(msgCalendar.getTimeInMillis() - redisCalendar.getTimeInMillis());
				double fixRealHours = CalcUtils.div(fixDiffInMilliseconds, (60 * 60 * 1000), 2);
				hoursBetweenDates = fixRealHours - days * (cfgEndHour - 1);
			}
		} catch (Exception ex) {
			ex.printStackTrace();
		}
		return hoursBetweenDates;
	}

	/**
	 * 
	 * @Title: calcHoursBetween
	 * @Description: 计算两个DATE间的小时数，去掉每天20:00:00~07:59:59间的时长（配置跨天）
	 * @param redisDate
	 * @param msgDate
	 * @return double
	 */
	private static double calcHoursBetweenJump(int cfgJumpStartHour, int cfgJumpEndHour, Date redisDate, Date msgDate) {
		double hoursBetweenDates = 0;

		try {
			if (redisDate.before(msgDate)) {
				Calendar redisCalendar = Calendar.getInstance();
				redisCalendar.setTime(redisDate);
				int redisHour = redisCalendar.get(Calendar.HOUR_OF_DAY);
				int redisYearDay = redisCalendar.get(Calendar.DAY_OF_YEAR);

				Calendar msgCalendar = Calendar.getInstance();
				msgCalendar.setTime(msgDate);
				int msgHour = msgCalendar.get(Calendar.HOUR_OF_DAY);
				int msgYearDay = msgCalendar.get(Calendar.DAY_OF_YEAR);

				int interval = 24 - cfgJumpStartHour + cfgJumpEndHour;

				if (redisYearDay == msgYearDay) {
					if (redisHour >= 0 && redisHour < cfgJumpEndHour) {
						if (msgHour >= 0 && msgHour < cfgJumpEndHour) {
							return hoursBetweenDates;
						}
						redisCalendar.set(Calendar.HOUR_OF_DAY, cfgJumpEndHour);
						redisCalendar.set(Calendar.MINUTE, 0);
						redisCalendar.set(Calendar.SECOND, 0);
//                		System.out.println("fix redis start date: " + DateUtil.format(redisCalendar.getTime(), "yyyy-MM-dd HH:mm:ss"));
					}
					if (msgHour >= cfgJumpStartHour && msgHour < 24) {
						if (redisHour >= cfgJumpStartHour && redisHour < 24) {
							return hoursBetweenDates;
						}
						msgCalendar.set(Calendar.HOUR_OF_DAY, cfgJumpStartHour);
						msgCalendar.set(Calendar.MINUTE, 0);
						msgCalendar.set(Calendar.SECOND, 0);
//                		System.out.println("fix message end date: " + DateUtil.format(msgCalendar.getTime(), "yyyy-MM-dd HH:mm:ss"));
					}

					long fixDiffInMilliseconds = Math
							.abs(msgCalendar.getTimeInMillis() - redisCalendar.getTimeInMillis());
					hoursBetweenDates = CalcUtils.div(fixDiffInMilliseconds, (60 * 60 * 1000), 2);
				} else {
					double calcHour = 0;
					Calendar tmpCalendar = Calendar.getInstance();
					for (int i = redisYearDay; i <= msgYearDay; i++) {
						if (i == redisYearDay) {
							if (redisHour >= 0 && redisHour < cfgJumpEndHour) {
								calcHour = calcHour + interval;
							} else if (redisHour >= cfgJumpEndHour && redisHour < cfgJumpStartHour) {
								tmpCalendar.setTime(redisDate);
								tmpCalendar.set(Calendar.HOUR_OF_DAY, cfgJumpStartHour);
								tmpCalendar.set(Calendar.MINUTE, 0);
								tmpCalendar.set(Calendar.SECOND, 0);

								long fixDiffInMilliseconds = Math
										.abs(tmpCalendar.getTimeInMillis() - redisCalendar.getTimeInMillis());
								calcHour = calcHour + CalcUtils.div(fixDiffInMilliseconds, (60 * 60 * 1000), 2);
							} else {
							}
						} else if (i == msgYearDay) {
							if (msgHour >= 0 && msgHour < cfgJumpEndHour) {
							} else if (msgHour >= cfgJumpEndHour && msgHour < cfgJumpStartHour) {
								tmpCalendar.setTime(msgDate);
								tmpCalendar.set(Calendar.HOUR_OF_DAY, cfgJumpEndHour);
								tmpCalendar.set(Calendar.MINUTE, 0);
								tmpCalendar.set(Calendar.SECOND, 0);

								long fixDiffInMilliseconds = Math
										.abs(msgCalendar.getTimeInMillis() - tmpCalendar.getTimeInMillis());
								calcHour = calcHour + CalcUtils.div(fixDiffInMilliseconds, (60 * 60 * 1000), 2);
							} else {
								calcHour = calcHour + interval;
							}
						} else {
							calcHour = calcHour + interval;
						}
					}
					hoursBetweenDates = calcHour;
				}
			}
		} catch (Exception ex) {
			ex.printStackTrace();
		}
		return hoursBetweenDates;
	}

	/**
	 * 
	 * @Title: calcHours4Exclude
	 * @Description: 计算两个date之间的小时数，同时排除掉配置区间类的小时数
	 * @param configStartHour
	 * @param configEndHour
	 * @param startDate
	 * @param messageDate
	 * @return double
	 */
	public static double calcHours4Exclude(int configStartHour, int configEndHour, Date startDate, Date messageDate) {
		if (configStartHour < configEndHour) { // 配置时段不跨天，开始区间 < 结束区间，如 01:00~08:00
			return round(calcHoursBetween(configStartHour, configEndHour, startDate, messageDate), 2);
		} else { // 配置时段跨天，开始区间 > 结束区间，如 20:00~08:00
			return round(calcHoursBetweenJump(configStartHour, configEndHour, startDate, messageDate), 2);
		}
	}
}
