package com.inzyme.spatiotemporal.common.utils;

import java.io.UnsupportedEncodingException;
import java.lang.reflect.Field;
import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import org.apache.commons.collections4.CollectionUtils;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;

import lombok.extern.slf4j.Slf4j;
@Slf4j
public class BeanUtil {
	
	static public Map<String ,Object> toMap(Object obj){
		return toMap(obj, null);
	}
	
	static public Map<String ,Object> toMap(String key ,Object obj){
		Map<String ,Object> empty = new HashMap<String, Object>();
		empty.put(key, obj);
		return empty;
	}
	
	static public Map<String ,Object> toMap(Object obj ,String ...excludeFields){
		Map<String ,Object> empty = new HashMap<String, Object>();
		
		if(Objects.isNull(obj)) {
			return empty;
		}
		
		return beanToMap(obj ,excludeFields);
	}
	
	/**
	 *  将 target目标的feild属性值赋值给original对象相同feild属性，前提是该属性值不为空
	 * @param original
	 * @param target
	 * @return
	 */
	static public <T> T addFeildValueInOrginal(T original ,T target) {
		if(Objects.isNull(original) || Objects.isNull(target) ) {
			return original;
		}
		
		return addFeildValueInOrginal0(original, target ,false);
	}
	
	/**
	 *  将 target目标的feild属性值替换掉original对象相同feild属性，前提是该属性值不为空
	 * @param original
	 * @param target
	 * @return
	 */
	static public <T> T switchFeildValueInOrginal(T original ,T target) {
		if(Objects.isNull(original) || Objects.isNull(target) ) {
			return original;
		}
		
		return addFeildValueInOrginal0(original, target ,true);
	}
	
	/**
	 * 将给定的对象转换为json字符串，转换失败返回null
	 * @param obj
	 * @return
	 */
	static public String toJson(Object obj) {
		try {
			return JSON.toJSONString(obj);
		} catch (Exception e) {
			log.warn("---- > switch object to json string is failed ,and that object is : " + obj.toString() ,e);
		}
		return  null;
	}
	
	static public JSONObject toJosnObject(String json) {
		return JSON.parseObject(json);
	}
	
	public static byte[] toJsonAsBytes(Object obj) {
		String json = toJson(obj);
		return Objects.isNull(json) ? new byte[0] : json.getBytes();
	}

	public static JSONArray toJosnArray(String json) {
		return JSON.parseArray(json);
	}
	
	/**
	 *	 从给定的json字符中，遍历查找第一个给定字段的值
	 * @param json
	 * @param field
	 * @return
	 */
	static public String deepExtractData(String json ,String field) {
		JSONObject jo = toJosnObject(json);
		Set<String> keys = jo.keySet();
		if(keys.contains(field)) {
			String result = jo.get(field).toString();
			//除去前后字符 引号；
			return  result.startsWith("\"") && result.endsWith("\"") ? result.substring(1,result.length() -1) : result;
		}
		for(String key : keys) {
			try {
				String s = deepExtractData(jo.get(key).toString(), field);
				if(s != null) {
					return s;
				}
			} catch (Exception e) {
			}
		}
		return null;
	}
	
	/**
	 * 	将给定的json字符串转换为给定的class对象，失败返回null
	 * @param json
	 * @param cls
	 * @return
	 */
	static public <T> T toBean(String json, Class<? extends T> cls) {
		try {
			return JSON.parseObject(json, cls);
		} catch (Exception e) {
			log.warn("---- > switch json string to bean is failed ,and that object class is : " + cls + " ,and that json string is : " + json,e);
		}
		return null;
	}
	
	static public <T> T toBean(byte[] json ,Class<? extends T> cls) {
		return toBean(new String(json), cls);
	}
	
	static public <T> T toBean(byte[] json ,Charset charset ,Class<? extends T> cls) {
		try {
			return toBean(new String(json ,charset.name()), cls);
		} catch (UnsupportedEncodingException e) {
			log.warn("---- > switch json string to bean is failed ,and that object class is : " + cls + " ,and that json string is : " + json
					+ " ;cause is is no given character : " + charset,e);
			return null;
		}
	}

	static private <T> T addFeildValueInOrginal0(T original, T target ,boolean isSwitch) {
		Field[] originalFields = getDeclaredFields(original);
		Map<String ,Object> targetFields = beanToMap(target, null);
		
		if(Objects.isNull(originalFields)) {
			return original;
		}
		
		for(int i = 0 ,len = originalFields.length ;i < len ;++i) {
			Field f = originalFields[i];
			f.setAccessible(true);
			Object originalValue = null;
			try {
				originalValue = f.get(original);
			} catch (IllegalArgumentException e) {
			} catch (IllegalAccessException e) {
			}
			
			if(!isSwitch && !Objects.isNull(originalValue)) {
				continue;
			}
			
			String name = f.getName();
			Class<?> fieldType = f.getType();
			Object val = targetFields.get(name);
			if(val != null && (fieldType.isAssignableFrom(val.getClass()) || fieldType.getName().equals(val.getClass().getName()))) {
				try {
					f.set(original, val);
				} catch (IllegalArgumentException | IllegalAccessException e) {
					// 
				}
			}
		}
		return original;
	}

	private static Field[] getDeclaredFields(Object original) {
		return original.getClass().getDeclaredFields();
	}

	private static Map<String, Object> beanToMap(Object obj, String[] excludeFields) {
		Map<String ,Object> result = new HashMap<String, Object>();
		
		Class<?> cls = obj.getClass();
		Field[] fields = cls.getDeclaredFields();
		
		if(Objects.isNull(fields) || fields.length == 0) {
			return result;
		}
		
		for(int i = 0 ,len = fields.length ;i < len ;++i) {
			Field field = fields[i];
			//打开私有访问
            field.setAccessible(true);
            //获取属性
            String name = field.getName();
            //排除属性
            if(CollectionUtils.containsAny(Arrays.asList(excludeFields), name)) {
            	continue;
            }
            //获取属性值
            Object value = null;
            try {
				value = field.get(obj);
			} catch (IllegalArgumentException e) {
			} catch (IllegalAccessException e) {
			}
            
            result.put(name, value);
		}
		return result;
	}

	/**
	 *  	给定对象所有字段是否为空
	 * @param obj
	 * @return
	 */
	static public boolean isFeildNull(Object obj) {
		return isFeildNull(obj,"");
	}
	
	static public boolean isFeildNull(Object obj ,String excludeField) {
		return isFeildNull(obj, new String[] {excludeField} );
	}
	
	/**
	 *	 给定的对象中，排除给定字段名称，其它字段是否都为空
	 * @param obj
	 * @param excludeFields
	 * @return
	 */
	static public boolean isFeildNull(Object obj ,String[] excludeFields) {
		if(obj != null) {
			Map<String ,Object> fields = beanToMap(obj, null);
			for(Map.Entry<String, Object> entry : fields.entrySet()) {
				if(!CollectionUtils.containsAny(Arrays.asList(excludeFields), entry.getKey()) && entry.getValue() != null) {
					return false;
				}
			}
		}
		return true;
	}
}
