package com.inzyme.spatiotemporal.common.utils;


import java.time.DayOfWeek;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.YearMonth;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.time.temporal.TemporalAdjusters;
import java.time.temporal.WeekFields;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author： 谢辉
 * @date： 2021/4/14
 * @email: xieh_mail@163.com
 * @description： 获取每个月有几周，并且每周的开始日期和结束日期
 * 国内都是以周一开始，注：即使第一天是周末，也算一周
 * @modifiedBy：
 * @version: 1.0
 */
public class AdDateUtils {

    /**
     * 获取指定月份有几周，每周的开始日期和结束日期
     *
     * @param yearMonth
     * @return
     */
    public static Map<Integer, WeekData> weeks(YearMonth yearMonth) {
        // 获取指定月份的第一天
        LocalDate start = LocalDate.now().with(yearMonth).with(TemporalAdjusters.firstDayOfMonth());
        // 获取指定月份的最后一天
        LocalDate end = LocalDate.now().with(yearMonth).with(TemporalAdjusters.lastDayOfMonth());

        Map<Integer, WeekData> map = Stream.iterate(start, localDate -> localDate.plusDays(1l))
                .limit(ChronoUnit.DAYS.between(start, end) + 1)
                // DayOfWeek.MONDAY 这里是指定周一（MONDAY）为一周的开始
                .collect(Collectors.groupingBy(localDate -> localDate.get(WeekFields.of(DayOfWeek.MONDAY, 1).weekOfMonth()),
                        Collectors.collectingAndThen(Collectors.toList(), WeekData::new)));
        return map;

    }

    /**
     * 获取指定时期的后面月份的第一天开始时间
     *
     * @param startDate 开始时间
     * @param month     第几个月
     * @return
     */
    public static Date getMonthDate(Date startDate, int month) {
        LocalDateTime localDateTime = startDate.toInstant()
                .atZone(ZoneId.systemDefault())
                .toLocalDateTime().plusMonths(month);
        Date date = Date.from(localDateTime.atZone(ZoneId.systemDefault()).toInstant());
        return date;
    }


    public static class WeekData {
        // 一周的开始时间
        public LocalDate start;
        // 一周的结束时间
        public LocalDate end;

        public WeekData(List<LocalDate> localDates) {
            this.start = localDates.get(0);
            this.end = localDates.get(localDates.size() - 1);
        }

        @Override
        public String toString() {
            return this.start + "   " + this.end;
        }
    }

    /**
     * 上一个月
     * @param localDate
     * @return
     */
    public static Map<Integer, WeekData> weeksNow(LocalDate localDate) {
        Map<Integer ,WeekData> weekDataMap =  weeks(YearMonth.of(localDate.getYear(), localDate.getMonth()));
        Map<Integer ,WeekData> map = new HashMap<>();
        int now = 0;
        for (Integer integer : weekDataMap.keySet()) {
                if (weekDataMap.get(integer).start.compareTo(localDate)<=0){
                    now=integer-1;
                }
        }
        if (weekDataMap.get(now)!=null){
            map.put(now,weekDataMap.get(now));
        }else {



        if (map.get(0)==null){
            weekDataMap = AdDateUtils.weeks(YearMonth.of(localDate.plusMonths(-1).getYear(), localDate.plusMonths(-1).getMonth()));
        }

        if (weekDataMap.get(5)!=null){
            map.put(5,weekDataMap.get(5));
        }else {
            map.put(4,weekDataMap.get(4));
        }
        }
        return map ;

    }

    public static Map<Integer, WeekData> weeksNowDq(LocalDate localDate) {
        Map<Integer ,WeekData> weekDataMap =  weeks(YearMonth.of(localDate.getYear(), localDate.getMonth()));
        Map<Integer ,WeekData> map = new HashMap<>();
        int now = 0;
        for (Integer integer : weekDataMap.keySet()) {
            if (weekDataMap.get(integer).start.compareTo(localDate)<=0){
                now=integer;
            }
        }
        map.put(now,weekDataMap.get(now));
        return map ;
    }

    public static void main(String[] args) throws Exception {
        Map<Integer ,WeekData> weekData =  weeksNowDq(LocalDate.now().minusDays(7));
//        Map<Integer ,WeekData> weekData =  weeksNowDq(LocalDate.of(2023, 7, 2));
        System.out.println(weekData);
        
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyyMMdd");
        LocalDate date = LocalDate.of(2021, 3, 14);
        LocalDate firstDay = date.with(TemporalAdjusters.firstDayOfMonth()); 
        LocalDate lastDay = date.with(TemporalAdjusters.lastDayOfMonth()); 
        System.out.println(Integer.valueOf(firstDay.format(formatter)));
        System.out.println(Integer.valueOf(lastDay.format(formatter)));
    }
}