package org.immutables.value.processor.encode;

import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.common.primitives.Booleans;
import java.lang.annotation.Annotation;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import javax.lang.model.element.*;
import javax.lang.model.type.*;
import javax.lang.model.util.*;

/**
 * ImplMirror used to parse data of AnnotationMirror for original annotation {@code org.immutables.encode.Encoding.Impl}
 * during annotation processing. Interface is being described using {@link org.immutables.value.processor.encode.Mirrors.Impl} annotation,
 * which should be structurally compatible to the annotation being modelled.
 * @see #find(Iterable)
 * @see #from(AnnotationMirror)
 */
@SuppressWarnings("all")
public class ImplMirror implements Mirrors.Impl {
  public static final String QUALIFIED_NAME = "org.immutables.encode.Encoding.Impl";
  public static final String MIRROR_QUALIFIED_NAME = "org.immutables.value.processor.encode.Mirrors.Impl";

  public static String mirrorQualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String qualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String simpleName() {
    return "Impl";
  }

  public static boolean isPresent(Element annotatedElement) {
    for (AnnotationMirror mirror : annotatedElement.getAnnotationMirrors()) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Finds first annotation of this type on the element.
   * @param element annotated element
   * @return optional {@code ImplMirror}, present if this annotation found
   */
  public static Optional<ImplMirror> find(Element element) {
    return find(element.getAnnotationMirrors());
  }

  /**
   * Finds first annotation of this type in an iterable of annotation mirrors.
   * @param mirrors annotation mirrors
   * @return optional {@code ImplMirror}, present if this annotation found
   */
  public static Optional<ImplMirror> find(Iterable<? extends AnnotationMirror> mirrors) {
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return Optional.of(new ImplMirror(mirror));
      }
    }
    return Optional.absent();
  }

  /**
   * Converts iterable of annotation mirrors where all annotation are of this type. Otherwise it fails
   * @param mirrors of this annotation type.
   * @return list of converted {@code ImplMirror}s
   */
  public static ImmutableList<ImplMirror> fromAll(Iterable<? extends AnnotationMirror> mirrors) {
    ImmutableList.Builder<ImplMirror> builder = ImmutableList.builder();
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      Preconditions.checkState(element.getQualifiedName().contentEquals(QUALIFIED_NAME),
          "Supplied mirrors should all be of this annotation type");
      builder.add(new ImplMirror(mirror));
    }
    return builder.build();
  }

  /**
   * Creates mirror with default values using annotation element (i.e. declaration, not usage).
   * @param element annotation type element
   * @return {@code ImplMirror}
   */
  public static ImplMirror from(TypeElement element) {
    return new ImplMirror(element);
  }

  /**
   * Tries to convert annotation mirror to this annotation type.
   * @param mirror annotation mirror
   * @return optional {@code ImplMirror}, present if mirror matched this annotation type
   */
  public static Optional<ImplMirror> from(AnnotationMirror mirror) {
    return find(Collections.singleton(mirror));
  }

  private final AnnotationMirror annotationMirror;
  private final boolean virtual;

  private ImplMirror(TypeElement defaultAnnotationElement) {
    Preconditions.checkArgument(defaultAnnotationElement.getQualifiedName().contentEquals(QUALIFIED_NAME)
        || defaultAnnotationElement.getQualifiedName().contentEquals(MIRROR_QUALIFIED_NAME));
    this.annotationMirror = null;

    // TBD TODO BIG

    boolean virtual = false;

    for (ExecutableElement attributeElement$
        : ElementFilter.methodsIn(defaultAnnotationElement.getEnclosedElements())) {
      String name$ = attributeElement$.getSimpleName().toString();
      if ("virtual".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @Impl");
        }
        VirtualExtractor virtualExtractor$ = new VirtualExtractor();
        annotationValue$.accept(virtualExtractor$, null);

        virtual = virtualExtractor$.get();
        continue;
      }
    }
    this.virtual = virtual;
  }

  private ImplMirror(AnnotationMirror annotationMirror) {
    this.annotationMirror = annotationMirror;

    boolean virtual = false;

    Map<? extends ExecutableElement, ? extends AnnotationValue> attributeValues$ = annotationMirror.getElementValues();
    for (ExecutableElement attributeElement$
        : ElementFilter.methodsIn(annotationMirror.getAnnotationType().asElement().getEnclosedElements())) {
      String name$ = attributeElement$.getSimpleName().toString();
      if ("virtual".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'virtual' attribute of @Impl");
        }
        VirtualExtractor virtualExtractor$ = new VirtualExtractor();
        annotationValue$.accept(virtualExtractor$, null);

        virtual = virtualExtractor$.get();
        continue;
      }
    }
    this.virtual = virtual;
  }

  /**
   * @return value of attribute {@code virtual}
   */
  @Override
  public boolean virtual() {
    return virtual;
  }

  /**
   * @return underlying annotation mirror
   */
  public AnnotationMirror getAnnotationMirror() {
    Preconditions.checkState(annotationMirror != null, "this is default mirror without originating AnnotationMirror");
    return annotationMirror;
  }

  /**
   * @return {@code Impl.class}
   */
  @Override
  public Class<? extends Annotation> annotationType() {
    return Mirrors.Impl.class;
  }

  @Override
  public int hashCode() {
    int h = 0;
    h += 127 * "virtual".hashCode() ^ Booleans.hashCode(virtual);
    return h;
  }

  @Override
  public boolean equals(Object other) {
    if (other instanceof ImplMirror) {
      ImplMirror otherMirror = (ImplMirror) other;
      return virtual == otherMirror.virtual;
    }
    return false;
  }

  @Override
  public String toString() {
    return "ImplMirror:" + annotationMirror;
  }

  private static class VirtualExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    boolean value;

    @Override
    public Void visitBoolean(boolean value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    boolean get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'virtual' in @" + QUALIFIED_NAME);
    }
  }
}
