/*
 * Decompiled with CFR 0.152.
 */
package org.geotools.data.postgis;

import java.io.IOException;
import java.util.Arrays;
import org.geotools.data.postgis.VarintDataInStream;
import org.locationtech.jts.geom.CoordinateSequence;
import org.locationtech.jts.geom.CoordinateSequenceFactory;
import org.locationtech.jts.geom.CoordinateSequences;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryCollection;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.LinearRing;
import org.locationtech.jts.geom.MultiLineString;
import org.locationtech.jts.geom.MultiPoint;
import org.locationtech.jts.geom.MultiPolygon;
import org.locationtech.jts.geom.Point;
import org.locationtech.jts.geom.Polygon;
import org.locationtech.jts.io.ByteArrayInStream;
import org.locationtech.jts.io.InStream;
import org.locationtech.jts.io.ParseException;

public class TWKBReader {
    static final int twkbPoint = 1;
    static final int twkbLineString = 2;
    static final int twkbPolygon = 3;
    static final int twkbMultiPoint = 4;
    static final int twkbMultiLineString = 5;
    static final int twkbMultiPolygon = 6;
    static final int twkbGeometryCollection = 7;
    protected CoordinateSequenceFactory csFactory;
    private GeometryFactory factory;
    TWKBMetadata metadata = new TWKBMetadata();
    VarintDataInStream dis = new VarintDataInStream();

    public TWKBReader(GeometryFactory geometryFactory) {
        this.factory = geometryFactory;
        this.csFactory = this.factory.getCoordinateSequenceFactory();
    }

    public Geometry read(byte[] bytes) throws ParseException, IOException {
        return this.read((InStream)new ByteArrayInStream(bytes));
    }

    public Geometry read(InStream is) throws IOException, ParseException {
        this.dis.setInStream(is);
        return this.readGeometry();
    }

    private Geometry readGeometry() throws IOException, ParseException {
        TWKBMetadata metadata = this.readMetadata();
        int geometryType = metadata.getType();
        switch (geometryType) {
            case 1: {
                return this.readPoint(metadata);
            }
            case 2: {
                return this.readLineString(metadata);
            }
            case 3: {
                return this.readPolygon(metadata);
            }
            case 4: {
                return this.readMultiPoint(metadata);
            }
            case 5: {
                return this.readMultiLineString(metadata);
            }
            case 6: {
                return this.readMultiPolygon(metadata);
            }
            case 7: {
                return this.readGeometryCollection(metadata);
            }
        }
        throw new ParseException("Unknown TWKB type " + geometryType);
    }

    private Point readPoint(TWKBMetadata metadata) throws IOException {
        if (!metadata.isEmpty()) {
            CoordinateSequence cs = this.readCoordinateSequence(1, metadata);
            return this.factory.createPoint(cs);
        }
        return this.factory.createPoint();
    }

    private LineString readLineString(TWKBMetadata metadata) throws IOException {
        if (!metadata.isEmpty()) {
            int size = this.dis.readUnsignedInt();
            CoordinateSequence cs = this.readCoordinateSequence(size, metadata);
            return this.factory.createLineString(cs);
        }
        return this.factory.createLineString();
    }

    private Polygon readPolygon(TWKBMetadata metadata) throws IOException {
        if (!metadata.isEmpty()) {
            int numRings = this.dis.readUnsignedInt();
            LinearRing[] holes = null;
            if (numRings > 1) {
                holes = new LinearRing[numRings - 1];
            }
            LinearRing shell = this.readLinearRing(metadata);
            for (int i = 0; i < numRings - 1; ++i) {
                holes[i] = this.readLinearRing(metadata);
            }
            return this.factory.createPolygon(shell, holes);
        }
        return this.factory.createPolygon();
    }

    private MultiPoint readMultiPoint(TWKBMetadata metadata) throws IOException {
        if (!metadata.isEmpty()) {
            int numGeom = this.dis.readUnsignedInt();
            Point[] geoms = new Point[numGeom];
            for (int i = 0; i < numGeom; ++i) {
                geoms[i] = this.readPoint(metadata);
            }
            return this.factory.createMultiPoint(geoms);
        }
        return this.factory.createMultiPoint();
    }

    private MultiLineString readMultiLineString(TWKBMetadata metadata) throws IOException, ParseException {
        if (!metadata.isEmpty()) {
            int numGeom = this.dis.readUnsignedInt();
            LineString[] geoms = new LineString[numGeom];
            for (int i = 0; i < numGeom; ++i) {
                geoms[i] = this.readLineString(metadata);
            }
            return this.factory.createMultiLineString(geoms);
        }
        return this.factory.createMultiLineString();
    }

    private MultiPolygon readMultiPolygon(TWKBMetadata metadata) throws IOException, ParseException {
        if (!metadata.isEmpty()) {
            int numGeom = this.dis.readUnsignedInt();
            Polygon[] geoms = new Polygon[numGeom];
            for (int i = 0; i < numGeom; ++i) {
                geoms[i] = this.readPolygon(metadata);
            }
            return this.factory.createMultiPolygon(geoms);
        }
        return this.factory.createMultiPolygon();
    }

    private GeometryCollection readGeometryCollection(TWKBMetadata metadata) throws IOException, ParseException {
        if (!metadata.isEmpty()) {
            int numGeom = this.dis.readUnsignedInt();
            Geometry[] geoms = new Geometry[numGeom];
            for (int i = 0; i < numGeom; ++i) {
                if (i > 0) {
                    Arrays.fill(metadata.valueArray, 0.0);
                }
                geoms[i] = this.readGeometry();
            }
            return this.factory.createGeometryCollection(geoms);
        }
        return this.factory.createGeometryCollection();
    }

    private LinearRing readLinearRing(TWKBMetadata metadata) throws IOException {
        int size = this.dis.readUnsignedInt();
        CoordinateSequence pts = this.readCoordinateSequenceRing(size, metadata);
        return this.factory.createLinearRing(pts);
    }

    private CoordinateSequence readCoordinateSequenceRing(int size, TWKBMetadata metadata) throws IOException {
        CoordinateSequence seq = this.readCoordinateSequence(size, metadata);
        return CoordinateSequences.ensureValidRing((CoordinateSequenceFactory)this.csFactory, (CoordinateSequence)seq);
    }

    private int zigzagDecode(int input) {
        return input >>> 1 ^ -(input & 1);
    }

    private TWKBMetadata readMetadata() throws IOException {
        byte geometryTypeAndPrecision = this.dis.readByte();
        int geometryType = geometryTypeAndPrecision & 0xF;
        int precision = this.zigzagDecode((geometryTypeAndPrecision & 0xF0) >>> 4);
        this.metadata.setType(geometryType);
        this.metadata.setPrecision(precision);
        byte header = this.dis.readByte();
        this.metadata.setHeader(header);
        int dims = 2;
        if (this.metadata.hasExtendedDims()) {
            byte dimensions = this.dis.readByte();
            if ((dimensions & 1) > 0) {
                ++dims;
                this.metadata.setHasM(true);
                this.metadata.setMprecision((dimensions & 0xE0) >> 5);
            }
            if ((dimensions & 2) > 0) {
                ++dims;
                this.metadata.setHasZ(true);
                this.metadata.setZprecision((dimensions & 0x1C) >> 2);
            }
        }
        this.metadata.setDims(dims);
        if (this.metadata.hasSize()) {
            this.metadata.setSize(this.dis.readSignedInt());
        } else {
            this.metadata.setSize(1);
        }
        if (this.metadata.hasBBOX()) {
            CoordinateSequence bbox = this.csFactory.create(2, dims);
            for (int i = 0; i < dims; ++i) {
                double min = this.readNextDouble(this.metadata.getScale(0));
                double delta = this.readNextDouble(this.metadata.getScale(0));
                bbox.setOrdinate(0, i, min);
                bbox.setOrdinate(1, i, min + delta);
            }
            this.metadata.setEnvelope(bbox);
        }
        return this.metadata;
    }

    protected CoordinateSequence readCoordinateSequence(int numPts, TWKBMetadata metadata) throws IOException {
        int i;
        int dims = metadata.getDims();
        CoordinateSequence seq = this.csFactory.create(numPts, dims);
        double[] scales = new double[dims];
        for (i = 0; i < scales.length; ++i) {
            scales[i] = metadata.getScale(i);
        }
        for (i = 0; i < numPts; ++i) {
            for (int j = 0; j < dims; ++j) {
                double value;
                double ordinateDelta = this.readNextDouble(scales[j]);
                metadata.valueArray[j] = value = metadata.valueArray[j] + ordinateDelta;
                seq.setOrdinate(i, j, value);
            }
        }
        return seq;
    }

    protected double readNextDouble(double scale) throws IOException {
        long value = this.dis.readSignedLong();
        return (double)value / scale;
    }

    protected static class TWKBMetadata {
        int zprecision;
        int mprecision;
        byte header;
        int size;
        int dims;
        double[] valueArray;
        int precision;
        double scale = 1.0;
        double scaleZ = 1.0;
        double scaleM = 1.0;
        boolean hasZ;
        boolean hasM;
        int type;
        CoordinateSequence envelope;

        public int getSize() {
            return this.size;
        }

        public void setSize(int size) {
            this.size = size;
        }

        public int getDims() {
            return this.dims;
        }

        public void setDims(int dims) {
            this.dims = dims;
            if (this.valueArray == null || this.valueArray.length != dims) {
                this.valueArray = new double[this.dims];
            } else {
                Arrays.fill(this.valueArray, 0.0);
            }
        }

        public byte getHeader() {
            return this.header;
        }

        public void setHeader(byte header) {
            this.header = header;
        }

        public int getType() {
            return this.type;
        }

        public void setType(int type) {
            this.type = type;
        }

        public int getPrecision() {
            return this.precision;
        }

        public double getScale(int i) {
            switch (i) {
                case 0: 
                case 1: {
                    return this.scale;
                }
                case 2: {
                    if (this.hasZ) {
                        return this.scaleZ;
                    }
                    if (this.hasM) {
                        return this.scaleM;
                    }
                    throw new IllegalArgumentException("Geometry only has XY dimensions.");
                }
                case 3: {
                    if (this.hasZ && this.hasM) {
                        return this.scaleM;
                    }
                    throw new IllegalArgumentException("Mismatch with the number of dimensions.");
                }
            }
            throw new IllegalArgumentException("Mismatch with the number of dimensions.");
        }

        public void setPrecision(int precision) {
            this.precision = precision;
            this.scale = Math.pow(10.0, precision);
        }

        public int getZprecision() {
            return this.zprecision;
        }

        public void setZprecision(int zprecision) {
            this.zprecision = zprecision;
            this.scaleZ = Math.pow(10.0, zprecision);
        }

        public int getMprecision() {
            return this.mprecision;
        }

        public void setMprecision(int mprecision) {
            this.mprecision = mprecision;
            this.scaleM = Math.pow(10.0, mprecision);
        }

        public boolean isHasZ() {
            return this.hasZ;
        }

        public void setHasZ(boolean hasZ) {
            this.hasZ = hasZ;
        }

        public boolean isHasM() {
            return this.hasM;
        }

        public void setHasM(boolean hasM) {
            this.hasM = hasM;
        }

        public CoordinateSequence getEnvelope() {
            return this.envelope;
        }

        public void setEnvelope(CoordinateSequence envelope) {
            this.envelope = envelope;
        }

        boolean hasBBOX() {
            return (this.header & 1) > 0;
        }

        boolean hasSize() {
            return (this.header & 2) > 0;
        }

        boolean hasIdList() {
            return (this.header & 4) > 0;
        }

        boolean hasExtendedDims() {
            return (this.header & 8) > 0;
        }

        boolean isEmpty() {
            return (this.header & 0x10) > 0;
        }
    }
}

