/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.client;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.hudi.avro.model.HoodieArchivedMetaEntry;
import org.apache.hudi.avro.model.HoodieMergeArchiveFilePlan;
import org.apache.hudi.client.transaction.TransactionManager;
import org.apache.hudi.client.utils.ArchivalUtils;
import org.apache.hudi.client.utils.MetadataConversionUtils;
import org.apache.hudi.common.engine.HoodieEngineContext;
import org.apache.hudi.common.fs.FSUtils;
import org.apache.hudi.common.model.HoodieAvroIndexedRecord;
import org.apache.hudi.common.model.HoodieAvroPayload;
import org.apache.hudi.common.model.HoodieLogFile;
import org.apache.hudi.common.model.HoodieRecord;
import org.apache.hudi.common.model.HoodieTableType;
import org.apache.hudi.common.table.HoodieTableMetaClient;
import org.apache.hudi.common.table.log.HoodieLogFormat;
import org.apache.hudi.common.table.log.block.HoodieAvroDataBlock;
import org.apache.hudi.common.table.log.block.HoodieLogBlock;
import org.apache.hudi.common.table.timeline.HoodieActiveTimeline;
import org.apache.hudi.common.table.timeline.HoodieArchivedTimeline;
import org.apache.hudi.common.table.timeline.HoodieInstant;
import org.apache.hudi.common.table.timeline.HoodieTimeline;
import org.apache.hudi.common.table.timeline.TimelineMetadataUtils;
import org.apache.hudi.common.table.timeline.TimelineUtils;
import org.apache.hudi.common.util.ClusteringUtils;
import org.apache.hudi.common.util.CollectionUtils;
import org.apache.hudi.common.util.CompactionUtils;
import org.apache.hudi.common.util.FileIOUtils;
import org.apache.hudi.common.util.Option;
import org.apache.hudi.common.util.collection.Pair;
import org.apache.hudi.config.HoodieWriteConfig;
import org.apache.hudi.exception.HoodieCommitException;
import org.apache.hudi.exception.HoodieException;
import org.apache.hudi.exception.HoodieIOException;
import org.apache.hudi.metadata.HoodieTableMetadata;
import org.apache.hudi.org.apache.avro.Schema;
import org.apache.hudi.org.apache.avro.generic.IndexedRecord;
import org.apache.hudi.storage.HoodieStorage;
import org.apache.hudi.storage.StoragePath;
import org.apache.hudi.storage.StoragePathInfo;
import org.apache.hudi.storage.StorageSchemes;
import org.apache.hudi.table.HoodieTable;
import org.apache.hudi.table.action.compact.CompactionTriggerStrategy;
import org.apache.hudi.table.marker.WriteMarkers;
import org.apache.hudi.table.marker.WriteMarkersFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HoodieTimelineArchiver<T extends HoodieAvroPayload, I, K, O> {
    private static final Logger LOG = LoggerFactory.getLogger(HoodieTimelineArchiver.class);
    private final StoragePath archiveFilePath;
    private final HoodieWriteConfig config;
    private HoodieLogFormat.Writer writer;
    private final int maxInstantsToKeep;
    private final int minInstantsToKeep;
    private final HoodieTable<T, I, K, O> table;
    private final HoodieTableMetaClient metaClient;
    private final TransactionManager txnManager;

    public HoodieTimelineArchiver(HoodieWriteConfig config, HoodieTable<T, I, K, O> table) {
        this.config = config;
        this.table = table;
        this.metaClient = table.getMetaClient();
        this.archiveFilePath = HoodieArchivedTimeline.getArchiveLogPath(this.metaClient.getArchivePath());
        this.txnManager = new TransactionManager(config, table.getMetaClient().getStorage());
        Pair<Integer, Integer> minAndMaxInstants = ArchivalUtils.getMinAndMaxInstantsToKeep(table, this.metaClient);
        this.minInstantsToKeep = minAndMaxInstants.getLeft();
        this.maxInstantsToKeep = minAndMaxInstants.getRight();
    }

    private HoodieLogFormat.Writer openWriter() {
        try {
            if (this.writer == null) {
                return HoodieLogFormat.newWriterBuilder().onParentPath(this.archiveFilePath.getParent()).withFileId(this.archiveFilePath.getName()).withFileExtension(".archive").withStorage(this.metaClient.getStorage()).overBaseCommit("").build();
            }
            return this.writer;
        }
        catch (IOException e) {
            throw new HoodieException("Unable to initialize HoodieLogFormat writer", e);
        }
    }

    public HoodieLogFormat.Writer reOpenWriter() {
        try {
            if (this.writer != null) {
                this.writer.close();
                this.writer = null;
            }
            this.writer = this.openWriter();
            return this.writer;
        }
        catch (IOException e) {
            throw new HoodieException("Unable to initialize HoodieLogFormat writer", e);
        }
    }

    private void close() {
        try {
            if (this.writer != null) {
                this.writer.close();
            }
        }
        catch (IOException e) {
            throw new HoodieException("Unable to close HoodieLogFormat writer", e);
        }
    }

    public boolean archiveIfRequired(HoodieEngineContext context) throws IOException {
        return this.archiveIfRequired(context, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean archiveIfRequired(HoodieEngineContext context, boolean acquireLock) throws IOException {
        try {
            if (acquireLock) {
                this.txnManager.beginTransaction(Option.empty(), Option.empty());
            }
            List<HoodieInstant> instantsToArchive = this.getInstantsToArchive().collect(Collectors.toList());
            this.verifyLastMergeArchiveFilesIfNecessary(context);
            boolean success = true;
            if (!instantsToArchive.isEmpty()) {
                this.writer = this.openWriter();
                LOG.info("Archiving instants " + instantsToArchive);
                this.archive(context, instantsToArchive);
                LOG.info("Deleting archived instants " + instantsToArchive);
                success = this.deleteArchivedInstants(instantsToArchive, context);
            } else {
                LOG.info("No Instants to archive");
            }
            if (this.shouldMergeSmallArchiveFiles()) {
                this.mergeArchiveFilesIfNecessary(context);
            }
            boolean bl = success;
            return bl;
        }
        finally {
            this.close();
            if (acquireLock) {
                this.txnManager.endTransaction(Option.empty());
            }
        }
    }

    public boolean shouldMergeSmallArchiveFiles() {
        return this.config.getArchiveMergeEnable() && !StorageSchemes.isAppendSupported(this.metaClient.getStorage().getScheme());
    }

    private void mergeArchiveFilesIfNecessary(HoodieEngineContext context) throws IOException {
        StoragePath planPath = new StoragePath(this.metaClient.getArchivePath(), "mergeArchivePlan");
        this.reOpenWriter();
        List<StoragePathInfo> entryList = this.metaClient.getStorage().globEntries(new StoragePath(this.metaClient.getArchivePath() + "/.commits_.archive*"));
        entryList.sort(new HoodieArchivedTimeline.ArchiveFileVersionComparator());
        int archiveMergeFilesBatchSize = this.config.getArchiveMergeFilesBatchSize();
        long smallFileLimitBytes = this.config.getArchiveMergeSmallFileLimitBytes();
        List<StoragePathInfo> mergeCandidate = this.getMergeCandidates(smallFileLimitBytes, entryList);
        if (mergeCandidate.size() >= archiveMergeFilesBatchSize) {
            List<String> candidateFiles = mergeCandidate.stream().map(fs -> fs.getPath().toString()).collect(Collectors.toList());
            String logFileName = this.computeLogFileName();
            this.buildArchiveMergePlan(candidateFiles, planPath, logFileName);
            this.mergeArchiveFiles(mergeCandidate);
            this.deleteFilesParallelize(this.metaClient, candidateFiles, context, true);
            LOG.info("Success to delete replaced small archive files.");
            this.metaClient.getStorage().deleteFile(planPath);
            LOG.info("Success to merge small archive files.");
        }
    }

    private List<StoragePathInfo> getMergeCandidates(long smallFileLimitBytes, List<StoragePathInfo> entryList) {
        int index;
        for (index = 0; index < entryList.size() && entryList.get(index).getLength() <= smallFileLimitBytes; ++index) {
        }
        return entryList.stream().limit(index).collect(Collectors.toList());
    }

    private String computeLogFileName() throws IOException {
        String logWriteToken = this.writer.getLogFile().getLogWriteToken();
        HoodieLogFile hoodieLogFile = this.writer.getLogFile().rollOver(this.metaClient.getStorage(), logWriteToken);
        return hoodieLogFile.getFileName();
    }

    private void verifyLastMergeArchiveFilesIfNecessary(HoodieEngineContext context) throws IOException {
        if (this.shouldMergeSmallArchiveFiles()) {
            StoragePath planPath = new StoragePath(this.metaClient.getArchivePath(), "mergeArchivePlan");
            HoodieStorage storage = this.metaClient.getStorage();
            if (storage.exists(planPath)) {
                HoodieMergeArchiveFilePlan plan = null;
                try {
                    plan = TimelineMetadataUtils.deserializeAvroMetadata(FileIOUtils.readDataFromPath(storage, planPath).get(), HoodieMergeArchiveFilePlan.class);
                }
                catch (IOException e) {
                    LOG.warn("Parsing merge archive plan failed.", (Throwable)e);
                    storage.deleteFile(planPath);
                    return;
                }
                StoragePath mergedArchiveFile = new StoragePath(this.metaClient.getArchivePath(), plan.getMergedArchiveFileName());
                List<StoragePath> candidates = plan.getCandidate().stream().map(StoragePath::new).collect(Collectors.toList());
                if (this.candidateAllExists(candidates)) {
                    if (storage.exists(mergedArchiveFile)) {
                        storage.deleteFile(mergedArchiveFile);
                    }
                } else if (storage.exists(mergedArchiveFile)) {
                    this.deleteFilesParallelize(this.metaClient, plan.getCandidate(), context, true);
                }
                storage.deleteFile(planPath);
            }
        }
    }

    private boolean candidateAllExists(List<StoragePath> candidates) throws IOException {
        for (StoragePath archiveFile : candidates) {
            if (this.metaClient.getStorage().exists(archiveFile)) continue;
            return false;
        }
        return true;
    }

    public void buildArchiveMergePlan(List<String> compactCandidate, StoragePath planPath, String compactedArchiveFileName) throws IOException {
        LOG.info("Start to build archive merge plan.");
        HoodieMergeArchiveFilePlan plan = HoodieMergeArchiveFilePlan.newBuilder().setCandidate(compactCandidate).setMergedArchiveFileName(compactedArchiveFileName).build();
        Option<byte[]> content = TimelineMetadataUtils.serializeAvroMetadata(plan, HoodieMergeArchiveFilePlan.class);
        FileIOUtils.createFileInPath(this.metaClient.getStorage(), planPath, content);
        LOG.info("Success to build archive merge plan");
    }

    public void mergeArchiveFiles(List<StoragePathInfo> compactCandidate) throws IOException {
        LOG.info("Starting to merge small archive files.");
        Schema wrapperSchema = HoodieArchivedMetaEntry.getClassSchema();
        try {
            ArrayList<IndexedRecord> records = new ArrayList<IndexedRecord>();
            for (StoragePathInfo fs : compactCandidate) {
                HoodieLogFormat.Reader reader = HoodieLogFormat.newReader(this.metaClient.getStorage(), new HoodieLogFile(fs.getPath()), HoodieArchivedMetaEntry.getClassSchema());
                Throwable throwable = null;
                try {
                    while (reader.hasNext()) {
                        HoodieAvroDataBlock blk = (HoodieAvroDataBlock)reader.next();
                        blk.getRecordIterator(HoodieRecord.HoodieRecordType.AVRO).forEachRemaining(r -> records.add((IndexedRecord)r.getData()));
                        if (records.size() < this.config.getCommitArchivalBatchSize()) continue;
                        this.writeToFile(wrapperSchema, records);
                    }
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (reader == null) continue;
                    if (throwable != null) {
                        try {
                            reader.close();
                        }
                        catch (Throwable throwable3) {
                            throwable.addSuppressed(throwable3);
                        }
                        continue;
                    }
                    reader.close();
                }
            }
            this.writeToFile(wrapperSchema, records);
        }
        catch (Exception e) {
            throw new HoodieCommitException("Failed to merge small archive files", e);
        }
        finally {
            this.writer.close();
        }
        LOG.info("Success to merge small archive files.");
    }

    private Map<String, Boolean> deleteFilesParallelize(HoodieTableMetaClient metaClient, List<String> paths, HoodieEngineContext context, boolean ignoreFailed) {
        return FSUtils.parallelizeFilesProcess(context, metaClient.getStorage(), this.config.getArchiveDeleteParallelism(), pairOfSubPathAndConf -> {
            StoragePath file = new StoragePath((String)pairOfSubPathAndConf.getKey());
            try {
                HoodieStorage storage = metaClient.getStorage();
                if (storage.exists(file)) {
                    return storage.deleteFile(file);
                }
                return true;
            }
            catch (IOException e) {
                if (!ignoreFailed) {
                    throw new HoodieIOException("Failed to delete : " + file, e);
                }
                LOG.warn("Ignore failed deleting : " + file);
                return true;
            }
        }, paths);
    }

    private Stream<HoodieInstant> getCleanInstantsToArchive() {
        HoodieTimeline cleanAndRollbackTimeline = this.table.getActiveTimeline().getTimelineOfActions(CollectionUtils.createSet("clean", "rollback")).filterCompletedInstants();
        return cleanAndRollbackTimeline.getInstantsAsStream().collect(Collectors.groupingBy(HoodieInstant::getAction)).values().stream().map(hoodieInstants -> {
            if (hoodieInstants.size() > this.maxInstantsToKeep) {
                return hoodieInstants.subList(0, hoodieInstants.size() - this.minInstantsToKeep);
            }
            return new ArrayList();
        }).flatMap(Collection::stream);
    }

    private Stream<HoodieInstant> getCommitInstantsToArchive() throws IOException {
        Option<HoodieInstant> completedCommitBeforeOldestPendingInstant;
        HoodieTimeline commitTimeline = this.table.getCompletedCommitsTimeline();
        Option<HoodieInstant> oldestPendingInstant = this.table.getActiveTimeline().getWriteTimeline().filter(instant -> !instant.isCompleted()).firstInstant();
        Option<Object> oldestCommitToRetain = oldestPendingInstant.isPresent() ? (!(completedCommitBeforeOldestPendingInstant = Option.fromJavaOptional(commitTimeline.getReverseOrderedInstants().filter(instant -> HoodieTimeline.compareTimestamps(instant.getTimestamp(), HoodieTimeline.LESSER_THAN, ((HoodieInstant)oldestPendingInstant.get()).getTimestamp())).findFirst())).isPresent() || HoodieTimeline.compareTimestamps(oldestPendingInstant.get().getTimestamp(), HoodieTimeline.LESSER_THAN, completedCommitBeforeOldestPendingInstant.get().getTimestamp()) ? oldestPendingInstant : completedCommitBeforeOldestPendingInstant) : Option.empty();
        Option<HoodieInstant> firstSavepoint = this.table.getCompletedSavepointTimeline().firstInstant();
        Set<String> savepointTimestamps = this.table.getSavepointTimestamps();
        if (!commitTimeline.empty() && commitTimeline.countInstants() > this.maxInstantsToKeep) {
            Option oldestInstantToRetainForCompaction = this.metaClient.getTableType() == HoodieTableType.MERGE_ON_READ && (this.config.getInlineCompactTriggerStrategy() == CompactionTriggerStrategy.NUM_COMMITS || this.config.getInlineCompactTriggerStrategy() == CompactionTriggerStrategy.NUM_AND_TIME) ? CompactionUtils.getOldestInstantToRetainForCompaction(this.table.getActiveTimeline(), this.config.getInlineCompactDeltaCommitMax()) : Option.empty();
            Option<HoodieInstant> oldestInstantToRetainForClustering = ClusteringUtils.getOldestInstantToRetainForClustering(this.table.getActiveTimeline(), this.table.getMetaClient());
            Stream<HoodieInstant> instantToArchiveStream = commitTimeline.getInstantsAsStream().filter(s -> {
                if (this.config.shouldArchiveBeyondSavepoint()) {
                    return !savepointTimestamps.contains(s.getTimestamp());
                }
                return !firstSavepoint.isPresent() || !HoodieTimeline.compareTimestamps(((HoodieInstant)firstSavepoint.get()).getTimestamp(), HoodieTimeline.LESSER_THAN_OR_EQUALS, s.getTimestamp());
            }).filter(s -> oldestCommitToRetain.map(instant -> HoodieTimeline.compareTimestamps(instant.getTimestamp(), HoodieTimeline.GREATER_THAN, s.getTimestamp())).orElse(true)).filter(s -> oldestInstantToRetainForCompaction.map(instantToRetain -> HoodieTimeline.compareTimestamps(s.getTimestamp(), HoodieTimeline.LESSER_THAN, instantToRetain.getTimestamp())).orElse(true)).filter(s -> oldestInstantToRetainForClustering.map(instantToRetain -> HoodieTimeline.compareTimestamps(s.getTimestamp(), HoodieTimeline.LESSER_THAN, instantToRetain.getTimestamp())).orElse(true));
            return instantToArchiveStream.limit(commitTimeline.countInstants() - this.minInstantsToKeep);
        }
        return Stream.empty();
    }

    private Stream<HoodieInstant> getInstantsToArchive() throws IOException {
        HoodieTableMetaClient dataMetaClient;
        Option<HoodieInstant> qualifiedEarliestInstant;
        Stream<HoodieInstant> instants = Stream.concat(this.getCleanInstantsToArchive(), this.getCommitInstantsToArchive());
        if (this.config.isMetaserverEnabled()) {
            return Stream.empty();
        }
        HoodieActiveTimeline rawActiveTimeline = new HoodieActiveTimeline(this.metaClient, false);
        Map<Pair, List<HoodieInstant>> groupByTsAction = rawActiveTimeline.getInstantsAsStream().collect(Collectors.groupingBy(i -> Pair.of(i.getTimestamp(), HoodieInstant.getComparableAction(i.getAction()))));
        if (this.table.getMetaClient().getTableConfig().isMetadataTableAvailable()) {
            try (HoodieTableMetadata tableMetadata = HoodieTableMetadata.create(this.table.getContext(), this.table.getStorage(), this.config.getMetadataConfig(), this.config.getBasePath());){
                Option<String> latestCompactionTime = tableMetadata.getLatestCompactionTime();
                if (!latestCompactionTime.isPresent()) {
                    LOG.info("Not archiving as there is no compaction yet on the metadata table");
                    instants = Stream.empty();
                } else {
                    LOG.info("Limiting archiving of instants to latest compaction on metadata table at " + latestCompactionTime.get());
                    instants = instants.filter(instant -> HoodieTimeline.compareTimestamps(instant.getTimestamp(), HoodieTimeline.LESSER_THAN, (String)latestCompactionTime.get()));
                }
            }
            catch (Exception e) {
                throw new HoodieException("Error limiting instant archival based on metadata table", e);
            }
        }
        if (this.table.isMetadataTable() && (qualifiedEarliestInstant = TimelineUtils.getEarliestInstantForMetadataArchival((dataMetaClient = HoodieTableMetaClient.builder().setBasePath(HoodieTableMetadata.getDatasetBasePath(this.config.getBasePath())).setConf(this.metaClient.getStorageConf()).build()).getActiveTimeline(), this.config.shouldArchiveBeyondSavepoint())).isPresent()) {
            instants = instants.filter(instant -> HoodieTimeline.compareTimestamps(instant.getTimestamp(), HoodieTimeline.LESSER_THAN, ((HoodieInstant)qualifiedEarliestInstant.get()).getTimestamp()));
        }
        return instants.flatMap(hoodieInstant -> {
            List instantsToStream = (List)groupByTsAction.get(Pair.of(hoodieInstant.getTimestamp(), HoodieInstant.getComparableAction(hoodieInstant.getAction())));
            if (instantsToStream != null) {
                return instantsToStream.stream();
            }
            return Stream.empty();
        });
    }

    private boolean deleteArchivedInstants(List<HoodieInstant> archivedInstants, HoodieEngineContext context) throws IOException {
        LOG.info("Deleting instants " + archivedInstants);
        ArrayList<HoodieInstant> pendingInstants = new ArrayList<HoodieInstant>();
        ArrayList<HoodieInstant> completedInstants = new ArrayList<HoodieInstant>();
        for (HoodieInstant instant2 : archivedInstants) {
            if (instant2.isCompleted()) {
                completedInstants.add(instant2);
                continue;
            }
            pendingInstants.add(instant2);
        }
        context.setJobStatus(this.getClass().getSimpleName(), "Delete archived instants: " + this.config.getTableName());
        HoodieActiveTimeline activeTimeline = this.metaClient.getActiveTimeline();
        if (!pendingInstants.isEmpty()) {
            context.foreach(pendingInstants, instant -> activeTimeline.deleteInstantFileIfExists((HoodieInstant)instant), Math.min(pendingInstants.size(), this.config.getArchiveDeleteParallelism()));
        }
        if (!completedInstants.isEmpty()) {
            completedInstants.stream().forEach(instant -> activeTimeline.deleteInstantFileIfExists((HoodieInstant)instant));
        }
        return true;
    }

    public void archive(HoodieEngineContext context, List<HoodieInstant> instants) throws HoodieCommitException {
        try {
            Schema wrapperSchema = HoodieArchivedMetaEntry.getClassSchema();
            LOG.info("Wrapper schema " + wrapperSchema.toString());
            ArrayList<IndexedRecord> records = new ArrayList<IndexedRecord>();
            for (HoodieInstant hoodieInstant : instants) {
                try {
                    this.deleteAnyLeftOverMarkers(context, hoodieInstant);
                    records.add(this.convertToAvroRecord(hoodieInstant));
                    if (records.size() < this.config.getCommitArchivalBatchSize()) continue;
                    this.writeToFile(wrapperSchema, records);
                }
                catch (Exception e) {
                    LOG.error("Failed to archive commits, .commit file: " + hoodieInstant.getFileName(), (Throwable)e);
                    if (!this.config.isFailOnTimelineArchivingEnabled()) continue;
                    throw e;
                }
            }
            this.writeToFile(wrapperSchema, records);
        }
        catch (Exception e) {
            throw new HoodieCommitException("Failed to archive commits", e);
        }
    }

    private void deleteAnyLeftOverMarkers(HoodieEngineContext context, HoodieInstant instant) {
        WriteMarkers writeMarkers = WriteMarkersFactory.get(this.config.getMarkersType(), this.table, instant.getTimestamp());
        if (writeMarkers.deleteMarkerDir(context, this.config.getMarkersDeleteParallelism())) {
            LOG.info("Cleaned up left over marker directory for instant :" + instant);
        }
    }

    private void writeToFile(Schema wrapperSchema, List<IndexedRecord> records) throws Exception {
        if (records.size() > 0) {
            HashMap<HoodieLogBlock.HeaderMetadataType, String> header = new HashMap<HoodieLogBlock.HeaderMetadataType, String>();
            header.put(HoodieLogBlock.HeaderMetadataType.SCHEMA, wrapperSchema.toString());
            String keyField = this.table.getMetaClient().getTableConfig().getRecordKeyFieldProp();
            List<HoodieRecord> indexRecords = records.stream().map(HoodieAvroIndexedRecord::new).collect(Collectors.toList());
            HoodieAvroDataBlock block = new HoodieAvroDataBlock(indexRecords, header, keyField);
            this.writer.appendBlock(block);
            records.clear();
        }
    }

    private IndexedRecord convertToAvroRecord(HoodieInstant hoodieInstant) throws IOException {
        return MetadataConversionUtils.createMetaWrapper(hoodieInstant, this.metaClient);
    }
}

