/*
 * Decompiled with CFR 0.152.
 */
package io.tileverse.rangereader.gcs;

import com.google.auth.Credentials;
import com.google.auth.oauth2.GoogleCredentials;
import com.google.cloud.ReadChannel;
import com.google.cloud.storage.Blob;
import com.google.cloud.storage.BlobId;
import com.google.cloud.storage.Storage;
import com.google.cloud.storage.StorageException;
import com.google.cloud.storage.StorageOptions;
import io.tileverse.rangereader.AbstractRangeReader;
import io.tileverse.rangereader.RangeReader;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URLDecoder;
import java.nio.ByteBuffer;
import java.time.Duration;
import java.util.Objects;
import java.util.OptionalLong;
import java.util.logging.Level;
import java.util.logging.Logger;

public class GoogleCloudStorageRangeReader
extends AbstractRangeReader
implements RangeReader {
    private static final Logger LOGGER = Logger.getLogger(GoogleCloudStorageRangeReader.class.getName());
    private final Storage storage;
    private final String bucket;
    private final String objectName;
    private Blob blob;

    GoogleCloudStorageRangeReader(Storage storage, String bucket, String objectName) throws IOException {
        this.storage = Objects.requireNonNull(storage, "Storage client cannot be null");
        this.bucket = Objects.requireNonNull(bucket, "Bucket name cannot be null");
        this.objectName = Objects.requireNonNull(objectName, "Object name cannot be null");
        BlobId blobId = BlobId.of((String)bucket, (String)objectName);
        this.blob = storage.get(blobId);
        if (this.blob == null || !this.blob.exists(new Blob.BlobSourceOption[0])) {
            throw new IOException("GCS object not found: gs://" + bucket + "/" + objectName);
        }
    }

    protected int readRangeNoFlip(long offset, int actualLength, ByteBuffer target) throws IOException {
        int n;
        block10: {
            long start = System.nanoTime();
            ReadChannel reader = this.blob.reader(new Blob.BlobSourceOption[0]);
            try {
                int totalBytesRead;
                int bytesRead;
                reader.seek(offset);
                reader.limit(offset + (long)actualLength);
                for (totalBytesRead = 0; totalBytesRead < actualLength && (bytesRead = reader.read(target)) != -1; totalBytesRead += bytesRead) {
                }
                if (LOGGER.isLoggable(Level.FINE)) {
                    long end = System.nanoTime();
                    long millis = Duration.ofNanos(end - start).toMillis();
                    LOGGER.fine("range:[%,d +%,d], time: %,dms]".formatted(offset, actualLength, millis));
                }
                n = totalBytesRead;
                if (reader == null) break block10;
            }
            catch (Throwable throwable) {
                try {
                    if (reader != null) {
                        try {
                            reader.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (StorageException e) {
                    throw new IOException("Failed to read range from GCS: " + e.getMessage(), e);
                }
            }
            reader.close();
        }
        return n;
    }

    public OptionalLong size() throws IOException {
        if (this.blob == null || !this.blob.exists(new Blob.BlobSourceOption[0])) {
            throw new IOException("GCS object not found: gs://" + this.bucket + "/" + this.objectName);
        }
        Long size = this.blob.getSize();
        return size == null ? OptionalLong.empty() : OptionalLong.of(size);
    }

    public String getSourceIdentifier() {
        return "gs://" + this.bucket + "/" + this.objectName;
    }

    public void close() {
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private Storage storage;
        private String projectId;
        private String bucket;
        private String objectName;
        private String host;
        private Credentials credentials;
        private boolean defaultCredentialsChain;
        private String quotaProjectId;

        private Builder() {
        }

        public Builder storage(Storage storage) {
            this.storage = Objects.requireNonNull(storage, "Storage cannot be null");
            return this;
        }

        public Builder projectId(String projectId) {
            this.projectId = Objects.requireNonNull(projectId, "Project ID cannot be null");
            return this;
        }

        public Builder bucket(String bucket) {
            this.bucket = Objects.requireNonNull(bucket, "Bucket cannot be null");
            return this;
        }

        public Builder objectName(String objectName) {
            this.objectName = Objects.requireNonNull(objectName, "Object name cannot be null");
            return this;
        }

        public Builder quotaProjectId(String quotaProjectId) {
            this.quotaProjectId = Objects.requireNonNull(quotaProjectId);
            return this;
        }

        public Builder defaultCredentialsChain(boolean defaultCredentialsChain) {
            this.defaultCredentialsChain = defaultCredentialsChain;
            return this;
        }

        public Builder uri(URI uri) {
            Objects.requireNonNull(uri, "URI cannot be null");
            String scheme = uri.getScheme();
            if (!("gs".equalsIgnoreCase(scheme) || "http".equalsIgnoreCase(scheme) || "https".equalsIgnoreCase(scheme))) {
                throw new IllegalArgumentException("URI must have gs or http(s) scheme: " + String.valueOf(uri));
            }
            if ("gs".equalsIgnoreCase(scheme)) {
                String bucketName = uri.getAuthority();
                if (bucketName == null || bucketName.isEmpty()) {
                    throw new IllegalArgumentException("GCS URI must have a bucket: " + String.valueOf(uri));
                }
                String pathStr = uri.getPath();
                if (pathStr == null || pathStr.isEmpty() || pathStr.equals("/")) {
                    throw new IllegalArgumentException("GCS URI must have an object name: " + String.valueOf(uri));
                }
                String object = pathStr.startsWith("/") ? pathStr.substring(1) : pathStr;
                this.bucket = bucketName;
                this.objectName = object;
            } else {
                String pathStr = uri.getPath();
                if (pathStr != null && pathStr.contains("/storage/v1/b/") && pathStr.contains("/o/")) {
                    this.parseGcsApiUrl(uri);
                    this.host = uri.getScheme() + "://" + uri.getAuthority();
                } else {
                    this.parsePublicGcsUrl(uri);
                    if (!"storage.googleapis.com".equals(uri.getHost())) {
                        this.host = uri.getScheme() + "://" + uri.getAuthority();
                    }
                }
            }
            return this;
        }

        private void parseGcsApiUrl(URI uri) {
            String pathStr = uri.getPath();
            String bucketMarker = "/storage/v1/b/";
            String objectMarker = "/o/";
            int bucketStart = pathStr.indexOf(bucketMarker);
            if (bucketStart == -1) {
                throw new IllegalArgumentException("Invalid GCS API URL format: " + String.valueOf(uri));
            }
            int bucketEnd = pathStr.indexOf(objectMarker, bucketStart += bucketMarker.length());
            if (bucketEnd == -1) {
                throw new IllegalArgumentException("Invalid GCS API URL format: " + String.valueOf(uri));
            }
            int objectStart = bucketEnd + objectMarker.length();
            this.bucket = pathStr.substring(bucketStart, bucketEnd);
            String objectPart = pathStr.substring(objectStart);
            int queryStart = objectPart.indexOf(63);
            if (queryStart != -1) {
                objectPart = objectPart.substring(0, queryStart);
            }
            try {
                this.objectName = URLDecoder.decode(objectPart, "UTF-8");
            }
            catch (UnsupportedEncodingException e) {
                this.objectName = objectPart;
            }
        }

        private void parsePublicGcsUrl(URI uri) {
            String pathStr = uri.getPath();
            if (pathStr == null || pathStr.isEmpty() || pathStr.equals("/")) {
                throw new IllegalArgumentException("GCS URI must have an object name: " + String.valueOf(uri));
            }
            String pathWithoutLeadingSlash = pathStr.startsWith("/") ? pathStr.substring(1) : pathStr;
            int firstSlash = pathWithoutLeadingSlash.indexOf(47);
            if (firstSlash == -1) {
                throw new IllegalArgumentException("GCS URI must have an object name: " + String.valueOf(uri));
            }
            String bucketName = pathWithoutLeadingSlash.substring(0, firstSlash);
            String objectName = pathWithoutLeadingSlash.substring(firstSlash + 1);
            if (bucketName.isEmpty() || objectName.isEmpty()) {
                throw new IllegalArgumentException("Invalid GCS URI format - bucket and object cannot be empty: " + String.valueOf(uri));
            }
            this.bucket = bucketName;
            this.objectName = objectName;
        }

        public GoogleCloudStorageRangeReader build() throws IOException {
            Storage storageClient;
            if (this.bucket == null || this.objectName == null) {
                throw new IllegalStateException("Bucket and object name must be set");
            }
            Credentials credentials = this.credentials;
            if (credentials == null && this.defaultCredentialsChain) {
                credentials = GoogleCredentials.getApplicationDefault();
            }
            if ((storageClient = this.storage) == null) {
                StorageOptions.Builder builder = StorageOptions.getDefaultInstance().toBuilder();
                if (this.projectId != null) {
                    builder.setProjectId(this.projectId);
                }
                if (this.quotaProjectId != null) {
                    builder.setQuotaProjectId(this.quotaProjectId);
                }
                if (this.host != null) {
                    builder.setHost(this.host);
                }
                if (credentials != null) {
                    builder.setCredentials(credentials);
                }
                storageClient = (Storage)builder.build().getService();
            }
            return new GoogleCloudStorageRangeReader(storageClient, this.bucket, this.objectName);
        }
    }
}

