/*
 * Decompiled with CFR 0.152.
 */
package io.tileverse.rangereader.azure;

import com.azure.core.credential.TokenCredential;
import com.azure.core.util.Context;
import com.azure.core.util.logging.ClientLogger;
import com.azure.identity.DefaultAzureCredential;
import com.azure.identity.DefaultAzureCredentialBuilder;
import com.azure.storage.blob.BlobClient;
import com.azure.storage.blob.BlobClientBuilder;
import com.azure.storage.blob.BlobUrlParts;
import com.azure.storage.blob.models.BlobDownloadResponse;
import com.azure.storage.blob.models.BlobErrorCode;
import com.azure.storage.blob.models.BlobRange;
import com.azure.storage.blob.models.BlobRequestConditions;
import com.azure.storage.blob.models.BlobStorageException;
import com.azure.storage.blob.models.DownloadRetryOptions;
import com.azure.storage.common.StorageSharedKeyCredential;
import com.azure.storage.common.implementation.connectionstring.StorageConnectionString;
import io.tileverse.rangereader.AbstractRangeReader;
import io.tileverse.rangereader.RangeReader;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URI;
import java.nio.ByteBuffer;
import java.time.Duration;
import java.util.Objects;
import java.util.OptionalLong;
import java.util.logging.Level;
import java.util.logging.Logger;

public class AzureBlobRangeReader
extends AbstractRangeReader
implements RangeReader {
    private static final Logger LOGGER = Logger.getLogger(AzureBlobRangeReader.class.getName());
    private final BlobClient blobClient;
    private long contentLength = -1L;

    AzureBlobRangeReader(BlobClient blobClient) throws IOException {
        this.blobClient = Objects.requireNonNull(blobClient, "BlobClient cannot be null");
        try {
            if (!blobClient.exists().booleanValue()) {
                throw new IOException("Blob does not exist: " + blobClient.getBlobUrl());
            }
            this.contentLength = blobClient.getProperties().getBlobSize();
        }
        catch (BlobStorageException e) {
            BlobErrorCode errorCode = e.getErrorCode();
            int statusCode = e.getStatusCode();
            throw new IOException("%s (%d): failure to access %s".formatted(errorCode, statusCode, blobClient.getBlobUrl()), e);
        }
        catch (RuntimeException e) {
            throw new IOException("failure to access %s: %s".formatted(blobClient.getBlobUrl(), e.getMessage()), e);
        }
    }

    protected int readRangeNoFlip(long offset, int actualLength, ByteBuffer target) throws IOException {
        try {
            long start = System.nanoTime();
            BlobRange range = new BlobRange(offset, Long.valueOf(actualLength));
            DownloadRetryOptions options = new DownloadRetryOptions().setMaxRetryRequests(3);
            ByteArrayOutputStream outputStream = new ByteArrayOutputStream(actualLength);
            BlobDownloadResponse response = this.blobClient.downloadStreamWithResponse((OutputStream)outputStream, range, options, new BlobRequestConditions(), false, Duration.ofSeconds(60L), Context.NONE);
            if (LOGGER.isLoggable(Level.FINE)) {
                long end = System.nanoTime();
                long millis = Duration.ofNanos(end - start).toMillis();
                LOGGER.fine("range:[%,d +%,d], time: %,dms]".formatted(offset, actualLength, millis));
            }
            if (response.getStatusCode() < 200 || response.getStatusCode() >= 300) {
                throw new IOException("Failed to download blob range, status code: " + response.getStatusCode());
            }
            byte[] data = outputStream.toByteArray();
            target.put(data);
            return data.length;
        }
        catch (Exception e) {
            throw new IOException("Failed to read range from blob: " + e.getMessage(), e);
        }
    }

    public OptionalLong size() throws IOException {
        if (this.contentLength < 0L) {
            try {
                this.contentLength = this.blobClient.getProperties().getBlobSize();
            }
            catch (Exception e) {
                throw new IOException("Failed to get blob size: " + e.getMessage(), e);
            }
        }
        return OptionalLong.of(this.contentLength);
    }

    public String getSourceIdentifier() {
        return this.blobClient.getBlobUrl();
    }

    public void close() {
    }

    public static AzureBlobRangeReader of(BlobClient blobClient) throws IOException {
        return new AzureBlobRangeReader(blobClient);
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private TokenCredential tokenCredential;
        private String accountName;
        private String accountKey;
        private String connectionString;
        private String containerName;
        private String blobName;
        private String sasToken;
        private URI endpoint;

        private Builder() {
        }

        public Builder tokenCredential(TokenCredential tokenCredential) {
            this.tokenCredential = Objects.requireNonNull(tokenCredential, "Token credential cannot be null");
            return this;
        }

        public Builder accountName(String accountName) {
            this.accountName = Objects.requireNonNull(accountName, "Account name cannot be null");
            return this;
        }

        public Builder accountKey(String accountKey) {
            this.accountKey = Objects.requireNonNull(accountKey, "Account key cannot be null");
            return this;
        }

        public Builder accountCredentials(String accountName, String accountKey) {
            this.accountName(accountName);
            this.accountKey(accountKey);
            return this;
        }

        public Builder connectionString(String connectionString) {
            this.connectionString = Objects.requireNonNull(connectionString, "Connection string cannot be null");
            return this;
        }

        public Builder containerName(String containerName) {
            this.containerName = Objects.requireNonNull(containerName, "Container name cannot be null");
            return this;
        }

        public Builder blobName(String blobName) {
            this.blobName = Objects.requireNonNull(blobName, "Blob path cannot be null");
            return this;
        }

        public Builder sasToken(String sasToken) {
            this.sasToken = Objects.requireNonNull(sasToken, "SAS token cannot be null");
            return this;
        }

        public Builder endpoint(URI uri) {
            Objects.requireNonNull(uri, "URI cannot be null");
            String scheme = uri.getScheme().toLowerCase();
            if (!"https".equals(scheme) && !"http".equals(scheme)) {
                throw new IllegalArgumentException("URI must have azure, https, or blob scheme: " + String.valueOf(uri));
            }
            this.endpoint = uri;
            return this;
        }

        public AzureBlobRangeReader build() throws IOException {
            String endpointUrl;
            BlobClientBuilder blobClientBuilder = new BlobClientBuilder();
            String accountName = this.accountName;
            String accountKey = this.accountKey;
            if (this.endpoint != null) {
                endpointUrl = this.endpoint.toString();
                blobClientBuilder.endpoint(endpointUrl);
                if (accountName == null) {
                    BlobUrlParts parts = BlobUrlParts.parse((String)endpointUrl);
                    accountName = parts.getAccountName();
                }
            } else if (this.connectionString != null) {
                if (this.containerName == null || this.blobName == null) {
                    throw new IllegalStateException("Container name and blob path are required with connection string");
                }
                blobClientBuilder.connectionString(this.connectionString).containerName(this.containerName).blobName(this.blobName);
                StorageConnectionString storageConnectionString = StorageConnectionString.create((String)this.connectionString, (ClientLogger)new ClientLogger(AzureBlobRangeReader.class));
                accountName = storageConnectionString.getAccountName();
                endpointUrl = storageConnectionString.getBlobEndpoint().getPrimaryUri();
            } else {
                throw new IllegalStateException("Either provide the endpoint URI or connectionString, containerName, and blobName");
            }
            if (accountName != null && accountKey != null) {
                StorageSharedKeyCredential credential = new StorageSharedKeyCredential(accountName, accountKey);
                blobClientBuilder.credential(credential);
            } else if (accountName != null && this.sasToken != null) {
                Object sasTokenWithQuestion = this.sasToken.startsWith("?") ? this.sasToken : "?" + this.sasToken;
                blobClientBuilder.sasToken((String)sasTokenWithQuestion);
            } else if (this.tokenCredential != null) {
                blobClientBuilder.credential(this.tokenCredential);
            } else if (endpointUrl.startsWith("https://%s.blob.core.windows.net".formatted(accountName))) {
                DefaultAzureCredential defaultAzureCredential = new DefaultAzureCredentialBuilder().build();
                blobClientBuilder.credential((TokenCredential)defaultAzureCredential);
            }
            BlobClient client = blobClientBuilder.buildClient();
            return new AzureBlobRangeReader(client);
        }
    }
}

