/*
 * Decompiled with CFR 0.152.
 */
package io.tileverse.pmtiles;

import io.tileverse.pmtiles.CompressionUtil;
import io.tileverse.pmtiles.DirectoryUtil;
import io.tileverse.pmtiles.HilbertCurve;
import io.tileverse.pmtiles.PMTilesEntry;
import io.tileverse.pmtiles.PMTilesHeader;
import io.tileverse.pmtiles.PMTilesWriter;
import io.tileverse.pmtiles.UnsupportedCompressionException;
import io.tileverse.tiling.pyramid.TileIndex;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.TreeMap;

class PMTilesWriterImpl
implements PMTilesWriter {
    private static final int MAX_ROOT_DIR_SIZE = 16384;
    private final Path outputPath;
    private final int minZoom;
    private final int maxZoom;
    private final byte tileCompression;
    private final byte internalCompression;
    private final byte tileType;
    private final int minLonE7;
    private final int minLatE7;
    private final int maxLonE7;
    private final int maxLatE7;
    private final byte centerZoom;
    private final int centerLonE7;
    private final int centerLatE7;
    private final TileRegistry tileRegistry;
    private byte[] compressedMetadata;
    private PMTilesWriter.ProgressListener progressListener;
    private boolean completed;
    private boolean closed;

    private PMTilesWriterImpl(Path outputPath, byte minZoom, byte maxZoom, byte tileCompression, byte internalCompression, byte tileType, int minLonE7, int minLatE7, int maxLonE7, int maxLatE7, byte centerZoom, int centerLonE7, int centerLatE7) throws IOException {
        this.outputPath = Objects.requireNonNull(outputPath, "Output path cannot be null");
        this.minZoom = minZoom;
        this.maxZoom = maxZoom;
        this.tileCompression = tileCompression;
        this.internalCompression = internalCompression;
        this.tileType = tileType;
        this.minLonE7 = minLonE7;
        this.minLatE7 = minLatE7;
        this.maxLonE7 = maxLonE7;
        this.maxLatE7 = maxLatE7;
        this.centerZoom = centerZoom;
        this.centerLonE7 = centerLonE7;
        this.centerLatE7 = centerLatE7;
        this.tileRegistry = new TileRegistry();
        this.setMetadata("{}");
        Files.createDirectories(outputPath.getParent(), new FileAttribute[0]);
    }

    @Override
    public void addTile(TileIndex tileIndex, byte[] data) throws IOException {
        this.checkNotCompletedOrClosed();
        Objects.requireNonNull(tileIndex, "Tile coordinates cannot be null");
        Objects.requireNonNull(data, "Tile data cannot be null");
        byte[] processedData = data;
        if (this.tileCompression != 1) {
            try {
                processedData = CompressionUtil.compress(data, this.tileCompression);
            }
            catch (UnsupportedCompressionException e) {
                throw new IOException("Failed to compress tile data", e);
            }
        }
        this.tileRegistry.addTile(tileIndex, processedData);
    }

    @Override
    public void setMetadata(String metadata) throws IOException {
        this.checkNotCompletedOrClosed();
        Objects.requireNonNull(metadata, "Metadata cannot be null");
        byte[] metadataBytes = metadata.getBytes(StandardCharsets.UTF_8);
        try {
            this.compressedMetadata = CompressionUtil.compress(metadataBytes, this.internalCompression);
        }
        catch (UnsupportedCompressionException e) {
            throw new IOException("Failed to compress metadata", e);
        }
    }

    @Override
    public void complete() throws IOException {
        this.checkNotCompletedOrClosed();
        if (this.tileRegistry.isEmpty()) {
            throw new IllegalStateException("No tiles added to the PMTiles file");
        }
        try {
            this.writePMTilesFile();
            this.completed = true;
        }
        catch (UnsupportedCompressionException e) {
            throw new IOException("Failed to compress data", e);
        }
    }

    @Override
    public void setProgressListener(PMTilesWriter.ProgressListener listener) {
        this.progressListener = listener;
    }

    @Override
    public void close() throws IOException {
        this.closed = true;
    }

    private void writePMTilesFile() throws IOException, UnsupportedCompressionException {
        this.reportProgress(0.0);
        List<PMTilesEntry> entries = this.tileRegistry.getOptimizedEntries();
        this.reportProgress(0.1);
        DirectoryUtil.DirectoryResult directoryResult = DirectoryUtil.buildRootLeaves(entries, this.internalCompression, 16384);
        this.reportProgress(0.2);
        List<TileContent> tileContents = this.tileRegistry.getUniqueContents();
        this.reportProgress(0.3);
        FileLayout layout = this.calculateLayout(directoryResult, this.compressedMetadata, tileContents);
        this.reportProgress(0.4);
        PMTilesHeader header = this.createHeader(layout, directoryResult, tileContents.size(), entries.size());
        this.reportProgress(0.5);
        try (FileChannel channel = FileChannel.open(this.outputPath, StandardOpenOption.CREATE, StandardOpenOption.WRITE, StandardOpenOption.TRUNCATE_EXISTING);){
            byte[] headerBytes = header.serialize();
            ByteBuffer headerBuffer = ByteBuffer.wrap(headerBytes);
            channel.write(headerBuffer);
            this.reportProgress(0.6);
            ByteBuffer rootDirBuffer = ByteBuffer.wrap(directoryResult.rootDirectory());
            channel.write(rootDirBuffer);
            this.reportProgress(0.65);
            ByteBuffer metadataBuffer = ByteBuffer.wrap(this.compressedMetadata);
            channel.write(metadataBuffer);
            this.reportProgress(0.7);
            ByteBuffer leafDirsBuffer = ByteBuffer.wrap(directoryResult.leafDirectories());
            channel.write(leafDirsBuffer);
            this.reportProgress(0.75);
            this.writeTileData(channel, tileContents);
            this.reportProgress(1.0);
        }
    }

    private FileLayout calculateLayout(DirectoryUtil.DirectoryResult directoryResult, byte[] metadata, List<TileContent> tileContents) {
        long rootDirOffset = 127L;
        long rootDirBytes = directoryResult.rootDirectory().length;
        long metadataOffset = rootDirOffset + rootDirBytes;
        long metadataBytes = metadata.length;
        long leafDirsOffset = metadataOffset + metadataBytes;
        long leafDirsBytes = directoryResult.leafDirectories().length;
        long tileDataOffset = leafDirsOffset + leafDirsBytes;
        long tileDataBytes = this.calculateTotalTileSize(tileContents);
        return new FileLayout(rootDirOffset, rootDirBytes, metadataOffset, metadataBytes, leafDirsOffset, leafDirsBytes, tileDataOffset, tileDataBytes);
    }

    private long calculateTotalTileSize(List<TileContent> tileContents) {
        return tileContents.stream().mapToLong(content -> content.data.length).sum();
    }

    private PMTilesHeader createHeader(FileLayout layout, DirectoryUtil.DirectoryResult directoryResult, int uniqueTileCount, int entryCount) {
        return PMTilesHeader.builder().rootDirOffset(layout.rootDirOffset).rootDirBytes(layout.rootDirBytes).jsonMetadataOffset(layout.metadataOffset).jsonMetadataBytes(layout.metadataBytes).leafDirsOffset(layout.leafDirsOffset).leafDirsBytes(layout.leafDirsBytes).tileDataOffset(layout.tileDataOffset).tileDataBytes(layout.tileDataBytes).addressedTilesCount(this.tileRegistry.getTileCount()).tileEntriesCount(entryCount).tileContentsCount(uniqueTileCount).clustered(true).internalCompression(this.internalCompression).tileCompression(this.tileCompression).tileType(this.tileType).minZoom((byte)this.minZoom).maxZoom((byte)this.maxZoom).minLonE7(this.minLonE7).minLatE7(this.minLatE7).maxLonE7(this.maxLonE7).maxLatE7(this.maxLatE7).centerZoom(this.centerZoom).centerLonE7(this.centerLonE7).centerLatE7(this.centerLatE7).build();
    }

    private void writeTileData(FileChannel channel, List<TileContent> tileContents) throws IOException {
        long totalBytes = this.calculateTotalTileSize(tileContents);
        long writtenBytes = 0L;
        double baseProgress = 0.75;
        double progressWeight = 0.25;
        Collections.sort(tileContents, Comparator.comparingLong(content -> content.offset));
        for (TileContent content2 : tileContents) {
            ByteBuffer buffer = ByteBuffer.wrap(content2.data);
            channel.write(buffer);
            double progress = baseProgress + (double)(writtenBytes += (long)content2.data.length) / (double)totalBytes * progressWeight;
            this.reportProgress(progress);
            if (!this.isCancelled()) continue;
            throw new IOException("Operation cancelled by user");
        }
    }

    private void reportProgress(double progress) {
        if (this.progressListener != null) {
            this.progressListener.onProgress(progress);
        }
    }

    private boolean isCancelled() {
        return this.progressListener != null && this.progressListener.isCancelled();
    }

    private void checkNotCompletedOrClosed() {
        if (this.completed) {
            throw new IllegalStateException("PMTilesWriter has already been completed");
        }
        if (this.closed) {
            throw new IllegalStateException("PMTilesWriter has been closed");
        }
    }

    private static class TileRegistry {
        private final Map<Long, String> tileIdToHash = new TreeMap<Long, String>();
        private final Map<String, TileContent> contentMap = new HashMap<String, TileContent>();

        private TileRegistry() {
        }

        public void addTile(TileIndex tileIndex, byte[] data) {
            long tileId = HilbertCurve.tileIndexToTileId(tileIndex);
            String hash = this.computeHash(data);
            this.tileIdToHash.put(tileId, hash);
            if (!this.contentMap.containsKey(hash)) {
                this.contentMap.put(hash, new TileContent(hash, data, 0L));
            }
        }

        public long getTileCount() {
            return this.tileIdToHash.size();
        }

        public boolean isEmpty() {
            return this.tileIdToHash.isEmpty();
        }

        public List<PMTilesEntry> getOptimizedEntries() {
            ArrayList<PMTilesEntry> entries = new ArrayList<PMTilesEntry>();
            long offset = 0L;
            for (TileContent content : this.getUniqueContents()) {
                content.offset = offset;
                offset += (long)content.data.length;
            }
            if (this.tileIdToHash.isEmpty()) {
                return entries;
            }
            long runStart = -1L;
            String runHash = null;
            long runLength = 0L;
            for (Map.Entry<Long, String> entry : this.tileIdToHash.entrySet()) {
                long tileId = entry.getKey();
                String hash = entry.getValue();
                TileContent content = this.contentMap.get(hash);
                if (runStart == -1L) {
                    runStart = tileId;
                    runHash = hash;
                    runLength = 1L;
                    continue;
                }
                if (tileId == runStart + runLength && hash.equals(runHash)) {
                    ++runLength;
                    continue;
                }
                TileContent runContent = this.contentMap.get(runHash);
                entries.add(new PMTilesEntry(runStart, runContent.offset, runContent.data.length, (int)runLength));
                runStart = tileId;
                runHash = hash;
                runLength = 1L;
            }
            if (runStart != -1L) {
                TileContent runContent = this.contentMap.get(runHash);
                entries.add(new PMTilesEntry(runStart, runContent.offset, runContent.data.length, (int)runLength));
            }
            return entries;
        }

        public List<TileContent> getUniqueContents() {
            ArrayList<TileContent> result = new ArrayList<TileContent>(this.contentMap.values());
            Collections.sort(result, Comparator.comparingLong(content -> content.offset));
            return result;
        }

        private String computeHash(byte[] data) {
            try {
                MessageDigest digest = MessageDigest.getInstance("SHA-256");
                byte[] hash = digest.digest(data);
                return this.bytesToHex(hash);
            }
            catch (NoSuchAlgorithmException e) {
                return String.valueOf(data.length) + "_" + Arrays.hashCode(data);
            }
        }

        private String bytesToHex(byte[] bytes) {
            StringBuilder sb = new StringBuilder();
            for (byte b : bytes) {
                sb.append(String.format("%02x", b));
            }
            return sb.toString();
        }
    }

    private record FileLayout(long rootDirOffset, long rootDirBytes, long metadataOffset, long metadataBytes, long leafDirsOffset, long leafDirsBytes, long tileDataOffset, long tileDataBytes) {
    }

    private static class TileContent {
        private final String hash;
        private final byte[] data;
        private long offset;

        public TileContent(String hash, byte[] data, long offset) {
            this.hash = hash;
            this.data = data;
            this.offset = offset;
        }
    }

    static class BuilderImpl
    implements PMTilesWriter.Builder {
        private Path outputPath;
        private int minZoom = 0;
        private int maxZoom = 0;
        private byte tileCompression = (byte)2;
        private byte internalCompression = (byte)2;
        private byte tileType = 1;
        private int minLonE7 = -1800000000;
        private int minLatE7 = -850000000;
        private int maxLonE7 = 1800000000;
        private int maxLatE7 = 850000000;
        private byte centerZoom = 0;
        private int centerLonE7 = 0;
        private int centerLatE7 = 0;

        BuilderImpl() {
        }

        @Override
        public PMTilesWriter.Builder outputPath(Path path) {
            this.outputPath = path;
            return this;
        }

        @Override
        public PMTilesWriter.Builder minZoom(int minZoom) {
            this.minZoom = minZoom;
            return this;
        }

        @Override
        public PMTilesWriter.Builder maxZoom(int maxZoom) {
            this.maxZoom = maxZoom;
            return this;
        }

        @Override
        public PMTilesWriter.Builder tileCompression(byte compressionType) {
            this.tileCompression = compressionType;
            return this;
        }

        @Override
        public PMTilesWriter.Builder internalCompression(byte compressionType) {
            this.internalCompression = compressionType;
            return this;
        }

        @Override
        public PMTilesWriter.Builder tileType(byte tileType) {
            this.tileType = tileType;
            return this;
        }

        @Override
        public PMTilesWriter.Builder bounds(double minLon, double minLat, double maxLon, double maxLat) {
            this.minLonE7 = (int)(minLon * 1.0E7);
            this.minLatE7 = (int)(minLat * 1.0E7);
            this.maxLonE7 = (int)(maxLon * 1.0E7);
            this.maxLatE7 = (int)(maxLat * 1.0E7);
            return this;
        }

        @Override
        public PMTilesWriter.Builder center(double lon, double lat, byte zoom) {
            this.centerLonE7 = (int)(lon * 1.0E7);
            this.centerLatE7 = (int)(lat * 1.0E7);
            this.centerZoom = zoom;
            return this;
        }

        @Override
        public PMTilesWriter build() throws IOException {
            if (this.outputPath == null) {
                throw new IllegalArgumentException("Output path must be specified");
            }
            return new PMTilesWriterImpl(this.outputPath, (byte)this.minZoom, (byte)this.maxZoom, this.tileCompression, this.internalCompression, this.tileType, this.minLonE7, this.minLatE7, this.maxLonE7, this.maxLatE7, this.centerZoom, this.centerLonE7, this.centerLatE7);
        }
    }
}

