/*
 * Decompiled with CFR 0.152.
 */
package io.tileverse.pmtiles;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import io.tileverse.io.ByteBufferPool;
import io.tileverse.io.ByteRange;
import io.tileverse.jackson.databind.pmtiles.v3.PMTilesMetadata;
import io.tileverse.pmtiles.ByteBufferInputStream;
import io.tileverse.pmtiles.CompressionUtil;
import io.tileverse.pmtiles.DirectoryUtil;
import io.tileverse.pmtiles.HilbertCurve;
import io.tileverse.pmtiles.InvalidHeaderException;
import io.tileverse.pmtiles.PMTilesDirectory;
import io.tileverse.pmtiles.PMTilesEntry;
import io.tileverse.pmtiles.PMTilesHeader;
import io.tileverse.pmtiles.UnsupportedCompressionException;
import io.tileverse.tiling.pyramid.TileIndex;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.SeekableByteChannel;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.IntStream;
import java.util.stream.Stream;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PMTilesReader {
    private static final Logger log = LoggerFactory.getLogger(PMTilesReader.class);
    private static final ObjectMapper objectMapper = new ObjectMapper();
    private final Supplier<SeekableByteChannel> channelSupplier;
    private static final Duration expireAfterAccess = Duration.ofSeconds(30L);
    private final LoadingCache<ByteRange, PMTilesDirectory> directoryCache;
    private final PMTilesHeader header;
    private PMTilesMetadata parsedMetadata;

    public PMTilesReader(Path path) throws IOException, InvalidHeaderException {
        this(PMTilesReader.fileChannelSupplier(path));
    }

    private static Supplier<SeekableByteChannel> fileChannelSupplier(Path path) {
        return () -> {
            try {
                return FileChannel.open(path, StandardOpenOption.READ);
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        };
    }

    public PMTilesReader(Supplier<SeekableByteChannel> rangeReaderSupplier) throws IOException, InvalidHeaderException {
        this.channelSupplier = Objects.requireNonNull(rangeReaderSupplier, "rangeReaderSupplier cannot be null");
        try (SeekableByteChannel channel = this.channel();){
            this.header = PMTilesHeader.readHeader(channel);
        }
        this.directoryCache = Caffeine.newBuilder().softValues().expireAfterAccess(expireAfterAccess).build(this::readDirectory);
    }

    private SeekableByteChannel channel() throws IOException {
        try {
            return this.channelSupplier.get();
        }
        catch (UncheckedIOException e) {
            throw e.getCause();
        }
    }

    public PMTilesHeader getHeader() {
        return this.header;
    }

    public Optional<ByteBuffer> getTile(int z, int x, int y) throws IOException {
        return this.getTile(TileIndex.xyz((long)x, (long)y, (int)z));
    }

    public Optional<ByteBuffer> getTile(TileIndex tileIndex) throws IOException {
        return this.getTile(tileIndex, Function.identity());
    }

    public <D> Optional<D> getTile(TileIndex tileIndex, Function<ByteBuffer, D> mapper) throws IOException {
        if (tileIndex.z() < 0) {
            throw new IllegalArgumentException("z can't be < 0");
        }
        if (tileIndex.x() < 0L) {
            throw new IllegalArgumentException("x can't be < 0");
        }
        if (tileIndex.y() < 0L) {
            throw new IllegalArgumentException("y can't be < 0");
        }
        long tileId = HilbertCurve.tileIndexToTileId(tileIndex);
        Optional<ByteRange> tileLocation = this.findTileLocation(tileId);
        if (log.isDebugEnabled()) {
            log.debug("PMTilesReader.getTile({}): tileId={}, location: {}", new Object[]{tileIndex, tileId, tileLocation.orElse(null)});
        }
        return tileLocation.map(this::readTile).map(mapper);
    }

    public Stream<TileIndex> getTileIndices() {
        IntStream zooms = IntStream.rangeClosed(this.header.minZoom(), this.header.maxZoom());
        return zooms.mapToObj(Integer::valueOf).flatMap(this::getTileIndicesByZoomLevel);
    }

    public Stream<TileIndex> getTileIndicesByZoomLevel(int zoomLevel) {
        if (zoomLevel < 0 || zoomLevel > 31) {
            throw new IllegalArgumentException("Zoom level must be between 0 and 31, got: " + zoomLevel);
        }
        ArrayList<TileIndex> tileIndices = new ArrayList<TileIndex>();
        try {
            this.collectTileIndicesForZoomLevel(ByteRange.of((long)this.header.rootDirOffset(), (int)((int)this.header.rootDirBytes())), zoomLevel, tileIndices, false);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        return tileIndices.stream();
    }

    private ByteBuffer readTile(ByteRange absolutePosition) {
        return this.readData(absolutePosition, this.header.tileCompression());
    }

    public ByteBuffer getRawMetadata() throws IOException, UnsupportedCompressionException {
        long offset = this.header.jsonMetadataOffset();
        int length = (int)this.header.jsonMetadataBytes();
        return this.readData(ByteRange.of((long)offset, (int)length), this.header.internalCompression());
    }

    public String getMetadataAsString() throws IOException, UnsupportedCompressionException {
        return PMTilesReader.toString(this.getRawMetadata());
    }

    public PMTilesMetadata getMetadata() throws IOException, UnsupportedCompressionException {
        if (this.parsedMetadata == null) {
            this.parsedMetadata = PMTilesReader.parseMetadata(this.getMetadataAsString());
        }
        return this.parsedMetadata;
    }

    static PMTilesMetadata parseMetadata(String jsonMetadata) throws IOException {
        if (jsonMetadata == null || jsonMetadata.isBlank()) {
            return PMTilesMetadata.of(null);
        }
        try {
            return (PMTilesMetadata)objectMapper.readValue(jsonMetadata, PMTilesMetadata.class);
        }
        catch (Exception e) {
            throw new IOException("Failed to parse PMTiles metadata JSON: " + e.getMessage() + "\n" + jsonMetadata, e);
        }
    }

    private Optional<ByteRange> findTileLocation(long tileId) throws IOException, UnsupportedCompressionException {
        long rootDirOffset = this.header.rootDirOffset();
        int rootDirLength = (int)this.header.rootDirBytes();
        return this.searchDirectory(ByteRange.of((long)rootDirOffset, (int)rootDirLength), tileId);
    }

    private Optional<ByteRange> searchDirectory(ByteRange entryRange, long tileId) throws IOException, UnsupportedCompressionException {
        PMTilesDirectory entries = this.getDirectory(entryRange);
        Optional<PMTilesEntry> entry = this.findEntryForTileId(entries, tileId);
        if (entry.isEmpty()) {
            return Optional.empty();
        }
        PMTilesEntry found = entry.get();
        if (found.isLeaf()) {
            return this.searchDirectory(this.header.leafDirDataRange(found), tileId);
        }
        return Optional.of(found).map(this.header::tileDataRange);
    }

    private PMTilesDirectory getDirectory(ByteRange directoryRange) throws IOException, UnsupportedCompressionException {
        try {
            return (PMTilesDirectory)this.directoryCache.get((Object)directoryRange);
        }
        catch (UncheckedIOException e) {
            throw e.getCause();
        }
    }

    private PMTilesDirectory readDirectory(ByteRange directoryRange) {
        try {
            long start = System.nanoTime();
            ByteBuffer dirBytes = this.readDirectoryBytes(directoryRange);
            long read = System.nanoTime() - start;
            PMTilesDirectory directory = DirectoryUtil.deserializeDirectory(dirBytes);
            long decode = System.nanoTime() - read - start;
            if (log.isDebugEnabled()) {
                log.debug("--> PMTilesDirectory lookup: [%,d +%,d], read: %,dms, decode: %,dms: %s".formatted(directoryRange.offset(), directoryRange.length(), Duration.ofNanos(read).toMillis(), Duration.ofNanos(decode).toMillis(), directory));
            }
            return directory;
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private Optional<PMTilesEntry> findEntryForTileId(PMTilesDirectory entries, long tileId) {
        int low = 0;
        int high = entries.size() - 1;
        while (low <= high) {
            int mid = low + high >>> 1;
            PMTilesEntry entry = entries.get(mid);
            if (tileId < entry.tileId()) {
                high = mid - 1;
                continue;
            }
            if (entry.isLeaf() || entry.runLength() == 0) {
                if (tileId == entry.tileId()) {
                    return Optional.of(entry);
                }
                if (tileId > entry.tileId()) {
                    low = mid + 1;
                    continue;
                }
                high = mid - 1;
                continue;
            }
            long entryEnd = entry.tileId() + (long)entry.runLength() - 1L;
            if (tileId <= entryEnd) {
                return Optional.of(entry);
            }
            low = mid + 1;
        }
        return this.findContainingEntry(entries, high, tileId);
    }

    private Optional<PMTilesEntry> findContainingEntry(PMTilesDirectory entries, int insertionPoint, long tileId) {
        long rangeEnd;
        if (insertionPoint < 0) {
            return Optional.empty();
        }
        PMTilesEntry candidate = entries.get(insertionPoint);
        if (candidate.isLeaf()) {
            return Optional.of(candidate);
        }
        if (candidate.runLength() > 0 && tileId <= (rangeEnd = candidate.tileId() + (long)candidate.runLength() - 1L)) {
            return Optional.of(candidate);
        }
        return Optional.empty();
    }

    private ByteBuffer readDirectoryBytes(ByteRange byteRange) throws IOException, UnsupportedCompressionException {
        return this.readData(byteRange, this.header.internalCompression());
    }

    private void collectTileIndicesForZoomLevel(ByteRange entryRange, int targetZoomLevel, List<TileIndex> tileIndices, boolean isLeafDir) throws IOException, UnsupportedCompressionException {
        PMTilesDirectory entries = this.getDirectory(entryRange);
        for (PMTilesEntry entry : entries) {
            if (entry.isLeaf()) {
                this.collectTileIndicesForZoomLevel(this.header.leafDirDataRange(entry), targetZoomLevel, tileIndices, true);
                continue;
            }
            TileIndex tileCoord = HilbertCurve.tileIdToTileIndex(entry.tileId());
            if (tileCoord.z() != targetZoomLevel) continue;
            for (int i = 0; i < entry.runLength(); ++i) {
                TileIndex currentTile = HilbertCurve.tileIdToTileIndex(entry.tileId() + (long)i);
                if (currentTile.z() != targetZoomLevel) continue;
                tileIndices.add(currentTile);
            }
        }
    }

    private ByteBuffer readData(ByteRange range, byte compression) {
        ByteBuffer buffer = ByteBufferPool.getDefault().borrowHeap(range.length());
        try {
            ByteBuffer byteBuffer;
            block11: {
                SeekableByteChannel channel = this.channelSupplier.get();
                try {
                    channel.position(range.offset());
                    channel.read(buffer);
                    buffer.flip();
                    byteBuffer = CompressionUtil.decompress(buffer, compression);
                    if (channel == null) break block11;
                }
                catch (Throwable throwable) {
                    try {
                        if (channel != null) {
                            try {
                                channel.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    catch (IOException e) {
                        throw new UncheckedIOException(e);
                    }
                }
                channel.close();
            }
            return byteBuffer;
        }
        finally {
            ByteBufferPool.getDefault().returnBuffer(buffer);
        }
    }

    static String toString(ByteBuffer byteBuffer) {
        try {
            return IOUtils.toString((InputStream)new ByteBufferInputStream(byteBuffer), (Charset)StandardCharsets.UTF_8);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }
}

