package com.cusc.nirvana.user.util.crypt;

import com.cusc.nirvana.user.util.CuscStringUtils;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.crypto.*;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.*;
import java.util.Base64;

/**
 * Description: 国密SM4
 * <br />
 * BC库从1.59版本开始已经基本实现了国密算法（SM2、SM3、SM4）
 * CreateDate 2021-11-17 14:37
 *
 * @author yuyi
 **/
public class Sm4Utils {

    private static final Logger LOGGER = LoggerFactory.getLogger(Sm4Utils.class);

    //算法名称
    public static final String ALGORITHM_NAME = "SM4";
    //ECB P5填充
    public static final String ALGORITHM_NAME_ECB_PADDING = "SM4/ECB/PKCS5Padding";
    //CBC P5填充
    public static final String ALGORITHM_NAME_CBC_PADDING = "SM4/CBC/PKCS5Padding";
    //密钥长度
    public static final int DEFAULT_KEY_SIZE = 128;

    static {
        Security.addProvider(new BouncyCastleProvider());
    }


    private Sm4Utils() {
    }

    /**
     * Description: ecb加密默认 hexString true
     * <br />
     * CreateDate 2021-11-17 15:53:34
     *
     * @author yuyi
     **/
    public static String encryptEcbPaddingHexString(String key, String data) {
        return encryptEcbPaddingHex(key, data, true, null);
    }

    public static String decryptEcbPaddingHexDefault(String key, byte[] base64Bytes) {
        return decryptEcbPaddingHex(key, base64Bytes, true, null);
    }

    /**
     * Description: ecb解密，可以指定混淆
     * <br />
     * CreateDate 2021-11-17 15:53:34
     *
     * @author yuyi
     **/
    private static String decryptEcbPaddingHex(String key, byte[] base64Bytes, boolean hexString, String charset) {

        try {
            byte[] keyBytes;
            if (hexString) {
                keyBytes = HexUtil.hexStringToBytes(key);
            } else {
                keyBytes = key.getBytes();
            }
            byte[] decrypted = decryptEcbPaddingByte(keyBytes, base64Bytes);
            if (CuscStringUtils.isNotEmpty(charset)) {
                return new String(decrypted, charset);
            }
            return new String(decrypted, StandardCharsets.UTF_8);
        } catch (Exception e) {
            LOGGER.error("Sm4Util.decryptEcbPaddingHex error ! ", e);
            return null;
        }
    }

    /**
     * ECB P5填充解密
     *
     * @param key        密钥
     * @param cipherText 加密后的数据
     * @return byte
     * @throws IllegalBlockSizeException
     * @throws BadPaddingException
     * @throws InvalidKeyException
     * @throws NoSuchAlgorithmException
     * @throws NoSuchProviderException
     * @throws NoSuchPaddingException
     */
    private static byte[] decryptEcbPaddingByte(byte[] key, byte[] cipherText)
            throws IllegalBlockSizeException, BadPaddingException, InvalidKeyException,
            NoSuchAlgorithmException, NoSuchProviderException, NoSuchPaddingException {
        Cipher cipher = generateEcbCipher(ALGORITHM_NAME_ECB_PADDING, Cipher.DECRYPT_MODE, key);
        return cipher.doFinal(cipherText);
    }

    /**
     * Description: ecb加密，可以指定混淆
     * <br />
     * CreateDate 2021-11-17 15:53:34
     *
     * @author yuyi
     **/
    private static String encryptEcbPaddingHex(String key, String data, boolean hexString, String charset) {
        if (CuscStringUtils.isEmpty(data)) {
            return null;
        }
        try {
            byte[] keyBytes;
            if (hexString) {
                keyBytes = HexUtil.hexStringToBytes(key);
            } else {
                keyBytes = key.getBytes();
            }
            byte[] dataBytes;
            if (CuscStringUtils.isNotEmpty(charset)) {
                dataBytes = data.getBytes(charset);
            } else {
                dataBytes = data.getBytes(StandardCharsets.UTF_8);
            }
            byte[] encrypted = encryptEcbPaddingByte(keyBytes, dataBytes);
            return base64Encoder(encrypted);
        } catch (Exception e) {
            LOGGER.error("Sm4Util.encryptEcbPaddingHex error ! ", e);
            return null;
        }
    }

    /**
     * Description: base64编码
     * <br />
     * CreateDate 2021-11-17 15:54:52
     *
     * @author yuyi
     **/
    private static String base64Encoder(byte[] encrypted) {
        return Base64.getEncoder().encodeToString(encrypted);
    }

    /**
     * ECB P5填充加密
     *
     * @param key  密钥
     * @param data 明文数据
     * @return byte
     * @throws InvalidKeyException
     * @throws NoSuchAlgorithmException
     * @throws NoSuchProviderException
     * @throws NoSuchPaddingException
     * @throws IllegalBlockSizeException
     * @throws BadPaddingException
     */
    private static byte[] encryptEcbPaddingByte(byte[] key, byte[] data)
            throws InvalidKeyException, NoSuchAlgorithmException, NoSuchProviderException,
            NoSuchPaddingException, IllegalBlockSizeException, BadPaddingException {
        Cipher cipher = generateEcbCipher(ALGORITHM_NAME_ECB_PADDING, Cipher.ENCRYPT_MODE, key);
        return cipher.doFinal(data);
    }

    /**
     * ECB P5填充加解密Cipher初始化
     *
     * @param algorithmName 算法名称
     * @param mode          1 加密  2解密
     * @param key           密钥
     * @return Cipher
     * @throws NoSuchAlgorithmException
     * @throws NoSuchProviderException
     * @throws NoSuchPaddingException
     * @throws InvalidKeyException
     */
    private static Cipher generateEcbCipher(String algorithmName, int mode, byte[] key)
            throws NoSuchAlgorithmException, NoSuchProviderException, NoSuchPaddingException, InvalidKeyException {
        Cipher cipher = Cipher.getInstance(algorithmName, BouncyCastleProvider.PROVIDER_NAME);
        Key sm4Key = new SecretKeySpec(key, ALGORITHM_NAME);
        cipher.init(mode, sm4Key);
        return cipher;
    }
}
