package com.cusc.nirvana.user.util.crypt;

import com.cusc.nirvana.spring.boot.SpringContextHelper;
import org.apache.commons.codec.binary.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.crypto.Cipher;
import java.io.*;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.security.Key;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;

public class CryptKeyUtil {

	private static final Logger logger = LoggerFactory.getLogger(CryptKeyUtil.class);

	/*
	 * 以下为对password的加解密
	 */
	public static final String PADDING = "RSA/ECB/PKCS1Padding";

	public static Key getKey(byte[] arrBTmp) {
		// 创建一个空的8位字节数组（默认值为0）
		byte[] arrB = new byte[8];
		// 将原始字节数组转换为8位
		int length = arrBTmp.length > 8 ? 8 : arrBTmp.length;
		System.arraycopy(arrBTmp, 0, arrB, 0, length);

		// 生成密钥
		Key key = new javax.crypto.spec.SecretKeySpec(arrB, "DES");
		return key;
	}

	/**
	 * 使用公钥加密
	 *
	 * @param text      需要加密的明文
	 * @param publicKey 公钥
	 * @return 加密后的密文
	 */
	public static String encrypt(String text, PublicKey publicKey) {
		String cipherText = null;
		try {
			final Cipher cipher = Cipher.getInstance(PADDING);

			// 使用公钥加密
			cipher.init(Cipher.ENCRYPT_MODE, publicKey);
			byte[] encryptedText = cipher.doFinal(text.getBytes(StandardCharsets.UTF_8));
			cipherText = byte2hex(encryptedText);
		} catch (Exception e) {
			logger.error("encrypt error", e);
		}
		return cipherText;
	}

	/**
	 * 使用私钥解密
	 *
	 * @param text 需要解密的密文
	 * @return 解密后的明文
	 */
	public static String decrypt(String text, PrivateKey privateKey) {
		String decryptString = null;
		try {
			final Cipher cipher = Cipher.getInstance(PADDING);
			// 使用私钥解密
			cipher.init(Cipher.DECRYPT_MODE, privateKey);
			byte[] dectyptedText = cipher.doFinal(hex2byte(text));
			decryptString = new String(dectyptedText, StandardCharsets.UTF_8);
		} catch (Exception e) {
			logger.error("decrypt error", e);
		}
		return decryptString;
	}

	public static String byte2hex(byte[] bytes) {
		StringBuilder retString = new StringBuilder();

		for (int i = 0; i < bytes.length; ++i) {
			retString.append(Integer.toHexString(0x0100 + (bytes[i] & 0x00FF)).substring(1).toUpperCase());
		}

		return retString.toString();
	}

	/**
	 * 转换的过程 16进转byte
	 * <p>
	 * Description:
	 * <p>
	 *
	 * @param hex
	 * @return
	 */
	public static byte[] hex2byte(String hex) {
		byte[] bts = new byte[hex.length() / 2];

		for (int i = 0; i < bts.length; i++) {
			bts[i] = (byte) Integer.parseInt(hex.substring(2 * i, (2 * i) + 2), 16);
		}

		return bts;
	}

	/**
	 * 获取私钥
	 *
	 * @return
	 */
	public static PrivateKey getPrivateKey(String name) {
		PrivateKey privateKey = null;
		try {
			byte[] decodeKey = Base64.decodeBase64(name);
			// 创建x509证书封装类
			PKCS8EncodedKeySpec x509 = new PKCS8EncodedKeySpec(decodeKey);
			KeyFactory keyFactory = KeyFactory.getInstance("RSA");
			privateKey = keyFactory.generatePrivate(x509);
		} catch (Exception ex) {
			logger.error("CryptUtil getPrivateKey error:{}", ex.getMessage());
		}
		return privateKey;
	}

	/**
	 * 获取公钥
	 *
	 * @param base64
	 * @return
	 */
	public static PublicKey getPublicKey(String base64) {
		PublicKey publicKey = null;
		try {
			byte[] decodeKey = Base64.decodeBase64(base64);
			X509EncodedKeySpec x509 = new X509EncodedKeySpec(decodeKey);
			KeyFactory keyFactory = KeyFactory.getInstance("RSA");
			publicKey = keyFactory.generatePublic(x509);
		} catch (Exception ex) {
			logger.error("CrypoUtil getPublicKey error:{}", ex.getMessage());
		}
		return publicKey;
	}

	/**
	 * 生成公钥私钥
	 *
	 * @param keySize 加密串大小
	 * @return
	 */
	public static KeyPair generateKey(int keySize) {
		try {
			final KeyPairGenerator keyGen = KeyPairGenerator.getInstance("RSA");
			keyGen.initialize(keySize == 0 ? 1024 : keySize);
			final KeyPair key = keyGen.generateKeyPair();
			return key;
		} catch (Exception e) {
			logger.error("generateKey error:{}", e);
		}
		return null;
	}

	public static File getPublicKeyFile(PublicKey publicKey, String path) {
		File publicKeyFile = null;
		ObjectOutputStream publicKeyOS = null;
		FileOutputStream fos = null;
		try {
			publicKeyFile = new File(path);
			if (publicKeyFile.getParentFile() != null) {
				publicKeyFile.getParentFile().mkdirs();
			}
			if (publicKeyFile.exists()) {
				Files.delete(Paths.get(path));
			}

			// Saving the Public key in a file
			if (publicKeyFile.createNewFile()) {
				fos = new FileOutputStream(publicKeyFile);
				publicKeyOS = new ObjectOutputStream(fos);
				publicKeyOS.writeObject(publicKey);
			}
		} catch (Exception e) {
			logger.error("getPublicKeyFile error:{}", e);
		} finally {
			if (publicKeyOS != null) {
				try {
					publicKeyOS.close();
				} catch (IOException e) {
					logger.error("close stream error", e);
				}
			}
			if (fos != null) {
				try {
					fos.close();
				} catch (IOException e) {
					logger.error("close stream error", e);
				}
			}
		}
		return publicKeyFile;
	}

	public static String encryptToBase64(String content) {
		return SpringContextHelper.getBean(DataCryptService.class).encryptData(content);
	}

	public static String decryptByBase64(String content) {
		return SpringContextHelper.getBean(DataCryptService.class).decryptData(content);
	}
}
