package com.cusc.nirvana.user.util;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.math.BigDecimal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Calendar;
import java.util.Date;

public class DateUtils {

    private static final Logger LOGGER = LoggerFactory.getLogger(DateUtils.class);

    private static final String FORMAT_YYYYMMDDHHMMSS = "yyyy-MM-dd HH:mm:ss";
    private static final String FORMAT_YYYYMMDD = "yyyy-MM-dd";

    public static String currentTimeStr(String pattern) {
        return DateTimeFormatter.ofPattern(pattern).format(LocalDateTime.now());
    }

    /**
     * 获取当前日期 yyyy-MM-dd HH:mm:ss
     *
     * @return
     */
    public static String currentTimeStr() {
        return DateTimeFormatter.ofPattern(FORMAT_YYYYMMDDHHMMSS).format(LocalDateTime.now());
    }

    public static long stringHourToMills(String hour) {
        if (StringUtils.isBlank(hour)) {
            return 0;
        }
        double h = Double.parseDouble(hour);
        return BigDecimal.valueOf(h * 3600 * 1000).longValue();
    }

    public static Date getTime(int hour, int intervalDays) {
        LocalDateTime endTime = LocalDateTime.now().withHour(hour).withMinute(0).withSecond(0).withNano(0)
                .minusDays(intervalDays);

        ZoneId zone = ZoneId.systemDefault();
        Instant instant = endTime.atZone(zone).toInstant();
        return Date.from(instant);
    }

    public static LocalDateTime getDayEnd(LocalDateTime time) {
        return time.withHour(23).withMinute(59).withSecond(59).withNano(999999999);
    }

    public static LocalDateTime getDayStart(LocalDateTime time) {
        return time.withHour(00).withMinute(00).withSecond(00).withNano(000);
    }

    public static LocalDateTime convertDateToLDT(Date date) {
        return LocalDateTime.ofInstant(date.toInstant(), ZoneId.systemDefault());
    }

    public static Date convertLDTToDate(LocalDateTime time) {
        return Date.from(time.atZone(ZoneId.systemDefault()).toInstant());
    }

    /**
     * @Title formatOtherZoneTimestampToLocalDateTime @Description
     * 其他时区时间戳转成当前时区时间 @param timestamp 其他时区时间戳 @param zoneId 其他时区ID @return
     * LocalDateTime 返回类型 @throws
     */
    public static String formatOtherZoneTimestampToLocalDateTime(long timestamp, String pattern, String zoneId) {
        return DateTimeFormatter.ofPattern(pattern).format(ZonedDateTime
                .of(LocalDateTime.ofInstant(Instant.ofEpochMilli(timestamp), ZoneId.systemDefault()), ZoneId.of(zoneId))
                .withZoneSameInstant(ZoneId.systemDefault()).toLocalDateTime());
    }

    /**
     * @Title formatOtherZoneTimeStrToLocalDateTime @Description
     * 其他时区时间字符串转成当前时区时间 @param timeStr 其他时区时间字符串 @param pattern
     * 时间字符串格式 @param zoneId 其他时区ID @return 参数 @return LocalDateTime
     * 返回类型 @throws
     */
    public static String formatOtherZoneTimeStrToLocalDateTime(String timeStr, String pattern, String zoneId) {
        return DateTimeFormatter.ofPattern(pattern)
                .format(ZonedDateTime
                        .of(LocalDateTime.parse(timeStr, DateTimeFormatter.ofPattern(pattern)), ZoneId.of(zoneId))
                        .withZoneSameInstant(ZoneId.systemDefault()).toLocalDateTime());
    }

    public static String formatAutomaticZeros(String timeStr, String patternBefore, String patternAfter) {
        DateTimeFormatter inputFormat = DateTimeFormatter.ofPattern(patternBefore);
        DateTimeFormatter outputFormat = DateTimeFormatter.ofPattern(patternAfter);
        return LocalDateTime.parse(timeStr, inputFormat).format(outputFormat);
    }

    /**
     * 格式化日期 yyyy-MM-dd HH:mm:ss
     *
     * @param date
     * @return
     */
    public static String formatDatetime(Date date) {
        return DateTimeFormatter.ofPattern(FORMAT_YYYYMMDDHHMMSS)
                .format(LocalDateTime.ofInstant(date.toInstant(), ZoneId.systemDefault()));
    }

    /**
     * This method subtracts the specified amount from the days field decrementing the
     * month and year fields as necessary to ensure the result remains valid.
     * The result is only invalid if the maximum/minimum year is exceeded.
     *
     * @param pattern
     * @return
     */
    public static String getTheDayBefore(String pattern, long minusDays) {
        return DateTimeFormatter.ofPattern(pattern).format(LocalDateTime.now().minusDays(minusDays));
    }


    /**
     * 获取当前日期 yyyy-MM-dd HH:mm:ss
     *
     * @return
     */
    public static String currentDateStr() {
        return DateTimeFormatter.ofPattern(FORMAT_YYYYMMDD).format(LocalDateTime.now());
    }

    /**
     * 字符串 转换成localdateTime
     *
     * @param strTime
     * @return
     */
    public static LocalDateTime str2LocalDateTime(String strTime) {
        DateTimeFormatter df = DateTimeFormatter.ofPattern(FORMAT_YYYYMMDDHHMMSS);
        LocalDateTime dateTime = LocalDateTime.parse(strTime, df);
        return dateTime;
    }

    /**
     * 字符串 转换成localdateTime
     *
     * @param strTime
     * @return
     */
    public static Date str2Date(String strTime) {
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat(FORMAT_YYYYMMDDHHMMSS);
        try {
            return simpleDateFormat.parse(strTime);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return null;
    }

    public static String formatDatetimeForSrlNum() {
        return DateTimeFormatter.ofPattern("yyyyMMddHHmmssSSS").format(LocalDateTime.now());
    }

    /**
     * Description:
     * <br />
     * CreateDate 2021-11-03 15:51:54
     *
     * @author yuyi
     **/
    public static int getYear(Date date) {
        Calendar calendar = Calendar.getInstance();
        if (date != null) {
            calendar.setTime(new Date());
        }
        return calendar.get(Calendar.YEAR);
    }

    /**
     * 比较两个日期的大小
     *
     * @param date1 起始日期
     * @param date2 结束日期
     * @return date1小于date2返回-1，date1大于date2返回1，相等返回0
     */
    public static int compareDate(Date date1, Date date2) {
        return date1.compareTo(date2);
    }

    /**
     * 计算两个日期之间相差的天数
     *
     * @param beginDate 起始日期
     * @param endDate   结束日期
     * @return 相差天数
     */
    public static int daysBetween(Date beginDate, Date endDate) {
        Instant instantBegin = beginDate.toInstant();
        Instant instantEnd = endDate.toInstant();
        ZoneId zoneId = ZoneId.systemDefault();
        LocalDate beginLocalDate = instantBegin.atZone(zoneId).toLocalDate();
        LocalDate endLocalDate = instantEnd.atZone(zoneId).toLocalDate();
        Long ret = endLocalDate.toEpochDay() - beginLocalDate.toEpochDay();
        return ret.intValue();
    }

    /**
     * 解析字符串类型的日期
     *
     * @param dateStr 日期
     * @param format  格式化
     * @return 日期类型
     */
    public static Date parseDate(String dateStr, String format) {
        if (CuscStringUtils.isEmpty(format)) {
            format = FORMAT_YYYYMMDDHHMMSS;
        }
        SimpleDateFormat formatter = new SimpleDateFormat(format);
        try {
            return formatter.parse(dateStr);
        } catch (ParseException e) {
            LOGGER.error("DateUtil.parseDate error ! ", e);
            return null;
        }
    }

    /**
     * 计算两个时间之间相差的毫秒
     *
     * @param beginDate 起始时间
     * @param endDate   结束时间
     * @return 相差的毫秒数
     */
    public static long millisBetween(LocalDateTime beginDate, LocalDateTime endDate) {
        return Duration.between(beginDate, endDate).toMillis();
    }

    /**
     * 计算两个时间之间相差的秒
     *
     * @param beginDate 起始时间
     * @param endDate   结束时间
     * @return 相差的秒数
     */
    public static long secondBetween(LocalDateTime beginDate, LocalDateTime endDate) {
        return millisBetween(beginDate, endDate) / 1000L;
    }

    /**
     * 计算两个时间之间相差的分钟
     *
     * @param beginDate 起始时间
     * @param endDate   结束时间
     * @return 相差的分钟数
     */
    public static long minutesBetween(LocalDateTime beginDate, LocalDateTime endDate) {
        return Duration.between(beginDate, endDate).toMinutes();
    }

    /**
     * 计算两个时间之间相差的天
     *
     * @param beginDate 起始时间
     * @param endDate   结束时间
     * @return 相差的天数
     */
    public static long daysBetween(LocalDateTime beginDate, LocalDateTime endDate) {
        return Duration.between(beginDate, endDate).toDays();
    }

    /**
     * 计算两个时间之间相差的小时
     *
     * @param beginDate 起始时间
     * @param endDate   结束时间
     * @return 相差的小时数
     */
    public static long hoursBetween(LocalDateTime beginDate, LocalDateTime endDate) {
        return Duration.between(beginDate, endDate).toHours();
    }

    /**
     * 解析字符串类型的日期
     *
     * @param dateStr 日期
     * @param format  格式化
     * @return 日期类型
     */
    public static LocalDateTime parseLocalDateTime(String dateStr, String format) {
        if (CuscStringUtils.isEmpty(format)) {
            format = FORMAT_YYYYMMDDHHMMSS;
        }
        return LocalDateTime.parse(dateStr, DateTimeFormatter.ofPattern(format));
    }

    /**
     * 格式化日期为字符串
     *
     * @param dateTime 日期
     * @param format  格式化
     * @return 字符串
     */
    public static String formatDate(LocalDateTime dateTime, String format) {
        if (CuscStringUtils.isEmpty(format)) {
            format = FORMAT_YYYYMMDDHHMMSS;
        }
        return dateTime.format(DateTimeFormatter.ofPattern(format));
    }

    /**
     * 解析字符串类型的日期
     *
     * @param dateStr 日期
     * @param format  格式化
     * @return 日期类型
     */
    public static LocalDate parseLocalDate(String dateStr, String format) {
        if (CuscStringUtils.isEmpty(format)) {
            format = FORMAT_YYYYMMDDHHMMSS;
        }
        return LocalDate.parse(dateStr, DateTimeFormatter.ofPattern(format));
    }

    /**
     * 格式化日期为字符串
     *
     * @param dateTime 日期
     * @param format  格式化
     * @return 字符串
     */
    public static String formatDate(LocalDate dateTime, String format) {
        if (CuscStringUtils.isEmpty(format)) {
            format = FORMAT_YYYYMMDDHHMMSS;
        }
        return dateTime.format(DateTimeFormatter.ofPattern(format));
    }

    public static void main(String[] args) {
        LocalDateTime begin = LocalDateTime.now();
        System.out.println(DateUtils.secondBetween(begin, DateUtils.getDayEnd(begin)));
    }
}
