package com.cusc.nirvana.user.eiam.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.cache.CacheFactory;
import com.cache.exception.CacheException;
import com.cusc.nirvana.common.result.PageResult;
import com.cusc.nirvana.common.result.Response;
import com.cusc.nirvana.rds.mybatis.PageHelper;
import com.cusc.nirvana.user.eiam.constants.CommonDeleteEnum;
import com.cusc.nirvana.user.eiam.constants.CommonStatusEnum;
import com.cusc.nirvana.user.eiam.constants.CommonYesOrNoEnum;
import com.cusc.nirvana.user.eiam.constants.RedisConstant;
import com.cusc.nirvana.user.eiam.constants.ResponseCode;
import com.cusc.nirvana.user.eiam.converter.UserConverter;
import com.cusc.nirvana.user.eiam.dao.UserDao;
import com.cusc.nirvana.user.eiam.dao.entity.UserPO;
import com.cusc.nirvana.user.eiam.dto.ApplicationDTO;
import com.cusc.nirvana.user.eiam.dto.SmsSendConfig;
import com.cusc.nirvana.user.eiam.dto.UserDTO;
import com.cusc.nirvana.user.eiam.dto.UserPasswordDTO;
import com.cusc.nirvana.user.eiam.dto.UserRoleDTO;
import com.cusc.nirvana.user.eiam.dto.UserSimpleDTO;
import com.cusc.nirvana.user.eiam.dto.UserTokenListDTO;
import com.cusc.nirvana.user.eiam.service.*;
import com.cusc.nirvana.user.eiam.util.CuscSqlUtils;
import com.cusc.nirvana.user.eiam.util.PasswordChecker;
import com.cusc.nirvana.user.exception.CuscUserException;
import com.cusc.nirvana.user.util.CuscStringUtils;
import com.cusc.nirvana.user.util.crypt.Sm4Util;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Random;

/**
 * <p>
 * 针对内部员工、生态合作伙伴、分级线下店铺等企业内部用户 服务实现类
 * </p>
 *
 * @author yuy336
 * @since 2021-10-20
 */
@Service
@Slf4j
public class UserServiceImpl extends ServiceImpl<UserDao, UserPO> implements IUserService {

    @Autowired
    private IUserRoleService userRoleService;

    @Autowired
    private IEiamSmsService smsService;

    @Autowired
    @Lazy
    private IUrlService urlService;

    @Autowired
    private IApplicationService applicationService;

    @Autowired
    private EiamTokenService tokenService;

    @Autowired
    private CacheFactory cacheFactory;

    @Override
    @Transactional
    public UserDTO addUser(UserDTO entity) {
        UserPO entity0 = new UserPO();
        BeanUtils.copyProperties(entity, entity0);
        entity0.setUuid(CuscStringUtils.generateUuid());
        entity0.setStatus(CommonStatusEnum.ENABLE.getCode());

        if (CuscStringUtils.isNotEmpty(entity.getPassword())) {
            //密码复杂度检查
            PasswordChecker.checkPasswordRule(entity.getPassword(), entity.getUserName(), entity.getPhone(),
                    entity.getEmail());
            entity0.setPassword(Sm4Util.encryptEcbPadding(entity0.getUuid().substring(16), entity.getPassword()));
        }

        UserDTO checkUserRepeat;
        List<UserPO> checkUserList;

        //检查用户名是否重复
        if (CuscStringUtils.isNotEmpty(entity.getUserName())) {
            checkUserRepeat = new UserDTO();
            checkUserRepeat.setUserName(entity.getUserName());
            checkUserRepeat.setTenantNo(entity.getTenantNo());
            checkUserList = queryUserByParams(checkUserRepeat);
            if (!CollectionUtils.isEmpty(checkUserList)) {
                throw new CuscUserException(ResponseCode.USER_NAME_REPEAT.getCode(),
                        ResponseCode.USER_NAME_REPEAT.getMsg());
            }
        }

        //检查手机号是否重复
        if (CuscStringUtils.isNotEmpty(entity.getPhone())) {
            checkUserRepeat = new UserDTO();
            checkUserRepeat.setPhone(entity.getPhone());
            checkUserRepeat.setTenantNo(entity.getTenantNo());
            checkUserList = queryUserByParams(checkUserRepeat);
            if (!CollectionUtils.isEmpty(checkUserList)) {
                throw new CuscUserException(ResponseCode.USER_PHONE_REPEAT.getCode(),
                        ResponseCode.USER_PHONE_REPEAT.getMsg());
            }
        }

        //检查邮箱是否重复
        if (CuscStringUtils.isNotEmpty(entity.getEmail())) {
            checkUserRepeat = new UserDTO();
            checkUserRepeat.setEmail(entity.getEmail());
            checkUserRepeat.setTenantNo(entity.getTenantNo());
            checkUserList = queryUserByParams(checkUserRepeat);
            if (!CollectionUtils.isEmpty(checkUserList)) {
                throw new CuscUserException(ResponseCode.USER_EMAIL_REPEAT.getCode(),
                        ResponseCode.USER_EMAIL_REPEAT.getMsg());
            }
        }
        this.save(entity0);
        entity.setUuid(entity0.getUuid());
        return entity;
    }

    @Override
    @Transactional
    public UserDTO updateUser(UserDTO entity) {
        UserPO entity0 = this.getPoByUuid(entity.getUuid(), entity.getTenantNo());
        if (entity0 == null || CommonDeleteEnum.DELETED.getCode() == entity0.getIsDelete()) {
            throw new CuscUserException(ResponseCode.USER_INVALID.getCode(),
                    ResponseCode.USER_INVALID.getMsg());
        }
        entity.setId(entity0.getId());
        BeanUtils.copyProperties(entity, entity0);

        UserDTO checkUserRepeat;
        List<UserPO> checkUserList;
        //检查用户名是否重复
        if (CuscStringUtils.isNotEmpty(entity.getUserName())) {
            checkUserRepeat = new UserDTO();
            checkUserRepeat.setUserName(entity.getUserName());
            checkUserRepeat.setId(entity.getId());
            checkUserRepeat.setTenantNo(entity.getTenantNo());
            checkUserList = queryUserByParams(checkUserRepeat);
            if (!CollectionUtils.isEmpty(checkUserList)) {
                throw new CuscUserException(ResponseCode.USER_NAME_REPEAT.getCode(),
                        ResponseCode.USER_NAME_REPEAT.getMsg());
            }
        }

        //检查手机号是否重复
        if (CuscStringUtils.isNotEmpty(entity.getPhone())) {
            checkUserRepeat = new UserDTO();
            checkUserRepeat.setPhone(entity.getPhone());
            checkUserRepeat.setId(entity.getId());
            checkUserRepeat.setTenantNo(entity.getTenantNo());
            checkUserList = queryUserByParams(checkUserRepeat);
            if (!CollectionUtils.isEmpty(checkUserList)) {
                throw new CuscUserException(ResponseCode.USER_PHONE_REPEAT.getCode(),
                        ResponseCode.USER_PHONE_REPEAT.getMsg());
            }
        }

        //检查邮箱是否重复
        if (CuscStringUtils.isNotEmpty(entity.getEmail())) {
            checkUserRepeat = new UserDTO();
            checkUserRepeat.setEmail(entity.getEmail());
            checkUserRepeat.setId(entity.getId());
            checkUserRepeat.setTenantNo(entity.getTenantNo());
            checkUserList = queryUserByParams(checkUserRepeat);
            if (!CollectionUtils.isEmpty(checkUserList)) {
                throw new CuscUserException(ResponseCode.USER_EMAIL_REPEAT.getCode(),
                        ResponseCode.USER_EMAIL_REPEAT.getMsg());
            }
        }

        //登录名不允许修改
        entity0.setUuid(null);
        entity0.setPassword(null);
        entity0.setTenantNo(null);
        this.updateById(entity0);
        return entity;
    }

    @Override
    @Transactional
    public boolean deleteUser(UserDTO entity) {
        UserPO entity0 = this.getPoByUuid(entity.getUuid(), entity.getTenantNo());
        if (entity0 == null) {
            throw new CuscUserException(ResponseCode.USER_INVALID.getCode(),
                    ResponseCode.USER_INVALID.getMsg());
        }
        entity.setId(entity0.getId());
        String userId = entity0.getUuid();
        //检查是否有租户权限
        if (CommonDeleteEnum.DELETED.getCode() == entity0.getIsDelete()) {
            return true;
        }

        //删除用户与角色的关系
        UserRoleDTO userRole = new UserRoleDTO();
        userRole.setUserId(entity0.getUuid());
        userRoleService.deleteByUser(userRole);

        entity0 = new UserPO();
        entity0.setId(entity.getId());
        entity0.setIsDelete(CommonDeleteEnum.DELETED.getCode());
        //删除的同时冻结用户
        entity0.setStatus(CommonStatusEnum.DISABLE.getCode());

        boolean ret = this.updateById(entity0);
        //删除redis中用户对应的url
        urlService.delUserRelUrlRedis(userId, entity0.getTenantNo(), null);

        //执行踢出
        tokenService.kickOutByUserId(userId, entity.getTenantNo());
        return ret;
    }

    @Override
    public UserDTO getUser(UserDTO entity) {
        QueryWrapper queryWrapper = new QueryWrapper();
        queryWrapper.eq(entity.getId() != null, "id", entity.getId());
        queryWrapper.eq("is_delete", CommonDeleteEnum.NORMAL.getCode());
        queryWrapper.eq("tenant_no", entity.getTenantNo());

        queryWrapper.in(!CollectionUtils.isEmpty(entity.getTenantNoList()), "tenant_no", entity.getTenantNoList());
        queryWrapper.eq(CuscStringUtils.isNotEmpty(entity.getPhone()), "phone", entity.getPhone());
        queryWrapper.eq(CuscStringUtils.isNotEmpty(entity.getEmail()), "email", entity.getEmail());
        queryWrapper.eq(CuscStringUtils.isNotEmpty(entity.getUserName()), "user_name", entity.getUserName());
        queryWrapper.eq(entity.getIsTenantAdmin() != null, "is_tenant_admin", entity.getIsTenantAdmin());
        queryWrapper.eq(entity.getStatus() != null, "status", entity.getStatus());
        queryWrapper.eq(CuscStringUtils.isNotEmpty(entity.getUuid()), "uuid", entity.getUuid());
        UserPO record = this.getOne(queryWrapper);
        return UserConverter.INSTANCE.poToDto(record);
    }

    @Override
    public PageResult<UserDTO> pageListUser(UserDTO entity) {
        QueryWrapper queryWrapper = new QueryWrapper();
        queryWrapper.orderByDesc("create_time");
        queryWrapper.eq("is_delete", 0);
        queryWrapper.eq("tenant_no", entity.getTenantNo());
        queryWrapper.like(CuscStringUtils.isNotEmpty(entity.getUserName()), "user_name", entity.getUserName());
        queryWrapper.like(CuscStringUtils.isNotEmpty(entity.getPhone()), "phone", entity.getPhone());
        queryWrapper.like(CuscStringUtils.isNotEmpty(entity.getFullName()), "full_name", entity.getFullName());
        queryWrapper.eq(entity.getStatus() != null, "status", entity.getStatus());
        queryWrapper.eq(CuscStringUtils.isNotEmpty(entity.getUuid()), "uuid", entity.getUuid());
        queryWrapper.eq(entity.getId() != null, "id", entity.getId());

        //关联组织查询
        if (CuscStringUtils.isNotEmpty(entity.getQueryOrganId())) {
            String userOrganStr =
                    "select user_id from eiam_user_organ where eiam_user.uuid = eiam_user_organ.user_id and "
                            + "eiam_user_organ.organ_id = '"
                            + CuscSqlUtils.transactSQLInjection(entity.getQueryOrganId()) + "'";
            queryWrapper.exists(userOrganStr);
        }

        Page<UserDTO> page =
                this.page(new Page<>(entity.getCurrPage(), entity.getPageSize()), queryWrapper);
        return PageHelper.convert(page, UserDTO.class);
    }

    @Override
    public List<UserSimpleDTO> queryListUser(UserDTO entity) {
        QueryWrapper queryWrapper = new QueryWrapper();
        queryWrapper.eq("is_delete", 0);
//        queryWrapper.orderByDesc("create_time");
        queryWrapper.eq(CuscStringUtils.isNotEmpty(entity.getTenantNo()), "tenant_no", entity.getTenantNo());

        queryWrapper.eq(CuscStringUtils.isNotEmpty(entity.getUserName()), "user_name", entity.getUserName());
        queryWrapper.eq(CuscStringUtils.isNotEmpty(entity.getPhone()), "phone",
                entity.getPhone());
        queryWrapper.like(CuscStringUtils.isNotEmpty(entity.getFullName()), "full_name", entity.getFullName());
        queryWrapper.eq(CuscStringUtils.isNotEmpty(entity.getUuid()), "uuid", entity.getUuid());
        queryWrapper.in(!CollectionUtils.isEmpty(entity.getUuidList()), "uuid", entity.getUuidList());
        queryWrapper.eq(entity.getIsTenantAdmin() != null, "is_tenant_admin", entity.getIsTenantAdmin());
        queryWrapper.orderByAsc("update_time");

        List<UserPO> recordList = this.list(queryWrapper);
        return getUserSimple(recordList);
    }

    @Override
    @Transactional
    public boolean frozen(UserDTO entity) {
        UserPO entity0 = this.getPoByUuid(entity.getUuid(), entity.getTenantNo());
        if (entity0 == null || CommonDeleteEnum.DELETED.getCode() == entity0.getIsDelete()) {
            throw new CuscUserException(ResponseCode.USER_INVALID.getCode(),
                    ResponseCode.USER_INVALID.getMsg());
        }
        String userId = entity0.getUuid();

        UserPO bean = new UserPO();
        bean.setId(entity0.getId());
        bean.setStatus(CommonStatusEnum.DISABLE.getCode());
        boolean ret = this.updateById(bean);

        //删除redis中用户对应的url
        urlService.delUserRelUrlRedis(userId, entity0.getTenantNo(), null);

        //执行踢出
        tokenService.kickOutByUserId(userId, entity.getTenantNo());
        return ret;
    }

    @Override
    @Transactional
    public boolean unfreeze(UserDTO entity) {
        UserPO entity0 = this.getPoByUuid(entity.getUuid(), entity.getTenantNo());
        if (entity0 == null || CommonDeleteEnum.DELETED.getCode() == entity0.getIsDelete()) {
            throw new CuscUserException(ResponseCode.USER_INVALID.getCode(),
                    ResponseCode.USER_INVALID.getMsg());
        }
        String userId = entity0.getUuid();
        //检查是否有租户权限
        UserPO bean = new UserPO();
        bean.setId(entity0.getId());
        bean.setStatus(CommonStatusEnum.ENABLE.getCode());

        boolean ret = this.updateById(bean);
        //将用户授权的应用的url放到redis中
        urlService.userRelAppUrlToRedis(userId, entity0.getTenantNo());
        return ret;
    }

    @Override
    public UserDTO getByUuid(UserDTO entity) {
        UserPO record = this.getPoByUuid(entity.getUuid(), entity.getTenantNo());
        return UserConverter.INSTANCE.poToDto(record);
    }

    @Override
    public List<UserSimpleDTO> getByUuids(UserDTO entity) {
        QueryWrapper queryWrapper = new QueryWrapper();
        queryWrapper.in("uuid", entity.getUuidList());
        queryWrapper.eq("is_delete", CommonDeleteEnum.NORMAL.getCode());
        queryWrapper.eq("tenant_no", entity.getTenantNo());

        List<UserPO> recordList = this.list(queryWrapper);
        return getUserSimple(recordList);
    }

    /**
     * Description: 将user对象转为user简单对象
     * <br />
     * CreateDate 2021-10-29 15:57:53
     *
     * @author yuyi
     **/
    private List<UserSimpleDTO> getUserSimple(List<UserPO> recordList) {
        List<UserSimpleDTO> retList = new ArrayList<>();
        UserSimpleDTO usDto;

        for (UserPO user : recordList) {
            usDto = new UserSimpleDTO();
            BeanUtils.copyProperties(user, usDto);
            retList.add(usDto);
        }
        return retList;
    }

    /**
     * Description: 通过角色id查询用户信息
     * <br />
     * CreateDate 2021-10-29 22:25:38
     *
     * @author yuyi
     **/
    @Override
    public List<UserDTO> queryUserByRoleId(UserRoleDTO entity) {
        return baseMapper.queryUserByRoleId(entity);
    }

    /**
     * Description: 更换租户管理员账号
     * <br />
     * CreateDate 2022-01-20 11:48:36
     *
     * @author yuyi
     **/
    @Override
    @Transactional
    public boolean changeTenantAdmin(UserDTO bean) {
        //检查参数是否正确
        if (bean == null || CuscStringUtils.isEmpty(bean.getTenantNo()) || CuscStringUtils.isEmpty(bean.getUuid())
                || CuscStringUtils.isEmpty(bean.getNewUserId())) {
            throw new CuscUserException(ResponseCode.PARAMETER_NULL.getCode(),
                    ResponseCode.PARAMETER_NULL.getMsg());
        }

        //老用户
        UserDTO oldUser = getByUuid(bean);
        if (oldUser == null || CommonYesOrNoEnum.YES.getCode() != oldUser.getIsTenantAdmin().intValue()) {
            throw new CuscUserException(ResponseCode.USER_INVALID.getCode(),
                    ResponseCode.USER_INVALID.getMsg());
        }

        //将老用户的管理员标记去掉
        UserDTO oldUserUpdate = new UserDTO();
        oldUserUpdate.setId(oldUser.getId());
        oldUserUpdate.setIsTenantAdmin(CommonYesOrNoEnum.NO.getCode());
        updateUser(oldUserUpdate);

        //新用户
        UserDTO newUser = new UserDTO();
        newUser.setUuid(bean.getNewUserId());
        newUser.setTenantNo(bean.getTenantNo());
        newUser = getByUuid(newUser);
        if (newUser == null) {
            throw new CuscUserException(ResponseCode.USER_INVALID.getCode(),
                    ResponseCode.USER_INVALID.getMsg());
        }
        //将新用户的标记为租户管理员
        UserDTO newUserUpdate = new UserDTO();
        newUserUpdate.setId(oldUser.getId());
        newUserUpdate.setIsTenantAdmin(CommonYesOrNoEnum.YES.getCode());
        updateUser(newUserUpdate);
        return true;
    }

    @Override
    public void allUserRelUrlToRedis(UserDTO bean) {
        if (bean == null || CuscStringUtils.isEmpty(bean.getTenantNo()) || CuscStringUtils.isEmpty(
                bean.getApplicationId())) {
            throw new CuscUserException(ResponseCode.PARAMETER_NULL.getCode(),
                    ResponseCode.PARAMETER_NULL.getMsg());
        }

        List<UserPO> userList = queryUserByParams(bean);
        if (CollectionUtils.isEmpty(userList)) {
            log.info("allUserRelUrlToRedis user list is empty!");
            return;
        }
        List<String> userIdList = new ArrayList<>();
        for (UserPO user : userList) {
            userIdList.add(user.getUuid());
        }
        urlService.userListRelUrlToRedis(userIdList, bean.getTenantNo(), bean.getApplicationId());
    }

    /**
     * Description: 修改密码
     * <br />
     * CreateDate 2022-01-21 15:36:34
     *
     * @author yuyi
     **/
    @Override
    @Transactional
    public boolean changePassword(UserDTO bean) {
        UserPO entity0 = this.getPoByUuid(bean.getUuid(), bean.getTenantNo());
        if (entity0 == null || CommonDeleteEnum.DELETED.getCode() == entity0.getIsDelete()) {
            throw new CuscUserException(ResponseCode.USER_INVALID.getCode(),
                    ResponseCode.USER_INVALID.getMsg());
        }

        //验证原密码是否正确
        if (!Sm4Util.encryptEcbPadding(entity0.getUuid().substring(16), bean.getPassword())
                .equals(entity0.getPassword())) {
            throw new CuscUserException(ResponseCode.USER_OLD_PASSWORD_ERROR.getCode(),
                    ResponseCode.USER_OLD_PASSWORD_ERROR.getMsg());
        }

        //密码复杂度检查
        PasswordChecker.checkPasswordRule(bean.getNewPassword(), entity0.getUserName(), entity0.getPhone(),
                entity0.getEmail());

        //修改密码
        UserPO userPO = new UserPO();
        userPO.setId(entity0.getId());
        userPO.setPassword(Sm4Util.encryptEcbPadding(entity0.getUuid().substring(16), bean.getNewPassword()));
        return this.updateById(userPO);
    }

    /**
     * Description: 重置密码
     * <br />
     * CreateDate 2022-01-21 15:36:34
     *
     * @author yuyi
     **/
    @Override
    @Transactional
    public boolean resetPassword(UserPasswordDTO bean) {
        UserPO entity0 = this.getPoByUuid(bean.getUuid(), bean.getTenantNo());
        if (entity0 == null || CommonDeleteEnum.DELETED.getCode() == entity0.getIsDelete()) {
            throw new CuscUserException(ResponseCode.USER_INVALID.getCode(),
                    ResponseCode.USER_INVALID.getMsg());
        }

        if (CuscStringUtils.isEmpty(entity0.getPhone())) {
            throw new CuscUserException(ResponseCode.RESET_PASSWORDPHONE_NOT_NULL.getCode(),
                    ResponseCode.RESET_PASSWORDPHONE_NOT_NULL.getMsg());
        }

        //是否验证短信验证
        if (CuscStringUtils.isNotEmpty(bean.getSmsCode())) {
            Response<Boolean> smsResp = smsService.checkSmsCaptcha(entity0.getPhone(), bean.getTenantNo(),
                    bean.getApplicationId(),
                    bean.getSmsCode());
            if(!smsResp.isSuccess()){
                throw new CuscUserException(smsResp.getCode(), smsResp.getMsg());
            }
        }

        //生成新密码
        String newPassword;
        boolean ret;
        //密码为空时，需要生成新密码
        if(CuscStringUtils.isEmpty(bean.getPassword())){
            SmsSendConfig smsConfig = bean.getSmsSendConfig();

            //查询应用配置
            ApplicationDTO appConfig = applicationService.getCacheByCode(bean.getApplicationId());
            //短信配置检查和读取应用的配置
            smsService.convertToSmsConfig(appConfig, smsConfig);

            //生成新密码
            newPassword = createNewPassword();

            //短信发送限制检查
            smsConfig.setIntervalLimitKey(RedisConstant.SMS_SEND_RESET_PASSWORD_INTERVAL_LIMIT);
            smsConfig.setTotalLimitKey(RedisConstant.SMS_SEND_RESET_PASSWORD_DAY_LIMIT);
            smsConfig.setTenantNo(bean.getTenantNo());
            smsConfig.setAppId(bean.getApplicationId());
            smsService.checkSmsSendLimit(entity0.getPhone(), smsConfig);

            //修改密码
            UserPO userPO = new UserPO();
            userPO.setId(entity0.getId());
            userPO.setPassword(Sm4Util.encryptEcbPadding(entity0.getUuid().substring(16), newPassword));
            ret = this.updateById(userPO);

            //将密码以短信的方式发送给用户
            smsService.sendSms(entity0.getPhone(), newPassword, smsConfig);
        }else{
            newPassword = bean.getPassword();
            //密码复杂度检查
            PasswordChecker.checkPasswordRule(newPassword, entity0.getUserName(), entity0.getPhone(),
                    entity0.getEmail());

            //修改密码
            UserPO userPO = new UserPO();
            userPO.setId(entity0.getId());
            userPO.setPassword(Sm4Util.encryptEcbPadding(entity0.getUuid().substring(16), newPassword));
            ret = this.updateById(userPO);
        }
        return ret;
    }

    /**
     * Description: 查询用户是否在线
     * <br />
     * CreateDate 2022-01-21 15:36:34
     *
     * @author yuyi
     **/
    @Override
    public boolean getUserOnline(UserDTO bean) {
        //通过用户id和应用id找到对应的token信息
        try {
            String userTokenListKey = RedisConstant.TOKEN_USER_TOKEN_INFO + bean.getTenantNo() + ":" + bean.getUuid();
            List<UserTokenListDTO> userList =
                    cacheFactory.getExpireListService().getList(userTokenListKey, UserTokenListDTO.class);
            if (CollectionUtils.isEmpty(userList)) {
                return false;
            }
            for (UserTokenListDTO userToken : userList) {
                if (userToken.getAppId().equals(bean.getApplicationId())) {
                    return true;
                }
            }
        } catch (CacheException e) {
            log.error("getUserOnline 访问reids失败 :{}", e);
            return false;
        }
        return false;
    }

    @Override
    public List<UserDTO> queryAdminUserListByUserIdList(List<String> userIdList, boolean tenantAdmin) {
        if (CollectionUtils.isEmpty(userIdList)) {
            return Collections.emptyList();
        }
        //租户管理员
        List<UserPO> list;
        if (tenantAdmin) {
            list = this.lambdaQuery().in(UserPO::getUuid, userIdList).eq(UserPO::getIsTenantAdmin, 1).list();
        } else {
            list = this.lambdaQuery().in(UserPO::getUuid, userIdList).ne(UserPO::getOrdinaryAdmin, 0).list();
        }


        return UserConverter.INSTANCE.poListToDtoList(list);
    }

    @Override
    public List<UserDTO> queryAdminUserListByUserIdListNew(List<String> userIdList, boolean tenantAdmin) {
        if (CollectionUtils.isEmpty(userIdList)) {
            return Collections.emptyList();
        }
        //租户管理员
        List<UserPO> list = this.lambdaQuery().in(UserPO::getUuid, userIdList).ne(UserPO::getOrdinaryAdmin, 0).list();

        return UserConverter.INSTANCE.poListToDtoList(list);
    }
    //-----------------私有方法区-----------------------

    /**
     * Description:查询用户信息
     * <br />
     * CreateDate 2021-10-26 17:42:09
     *
     * @author yuyi
     **/
    private List<UserPO> queryUserByParams(UserDTO entity) {
        QueryWrapper queryWrapper = new QueryWrapper();
        queryWrapper.eq("is_delete", 0);
        queryWrapper.eq("tenant_no", entity.getTenantNo());

        queryWrapper.in(!CollectionUtils.isEmpty(entity.getTenantNoList()), "tenant_no", entity.getTenantNoList());
        queryWrapper.eq(CuscStringUtils.isNotEmpty(entity.getUserName()), "user_name", entity.getUserName());
        queryWrapper.eq(CuscStringUtils.isNotEmpty(entity.getPhone()), "phone",
                entity.getPhone());
        queryWrapper.eq(CuscStringUtils.isNotEmpty(entity.getEmail()), "email",
                entity.getEmail());
        queryWrapper.ne(entity.getId() != null, "id", entity.getId());
        return this.list(queryWrapper);
    }

    /**
     * 通过UUID查询单条数据
     *
     * @param uuid
     * @return 实例对象
     */
    private UserPO getPoByUuid(String uuid, String tenantNo) {
        QueryWrapper queryWrapper = new QueryWrapper();
        queryWrapper.eq("uuid", uuid);
        queryWrapper.eq("is_delete", CommonDeleteEnum.NORMAL.getCode());
        queryWrapper.eq(CuscStringUtils.isNotEmpty(tenantNo), "tenant_no", tenantNo);
        return this.getOne(queryWrapper);
    }

    /**
     * 创建新密码
     */
    private String createNewPassword() {
        char[] chars = new char[9];
        Random rnd = new Random();
        int specialPosition = rnd.nextInt(7) + 1;
        for (int i = 0; i < 9; i++) {
            if (specialPosition == i) {
                chars[i] = '@';
            } else {
                chars[i] = nextChar(rnd);
            }
        }
        return new String(chars);
    }

    /**
     * 创建随机字符，包含数字、大小写字母
     */
    private static char nextChar(Random rnd) {
        switch (rnd.nextInt(3)) {
            case 0:
                return (char) ('a' + rnd.nextInt(26));
            case 1:
                return (char) ('A' + rnd.nextInt(26));
            default:
                return (char) ('0' + rnd.nextInt(10));
        }
    }
}
