package com.cusc.nirvana.user.auth.identification.service.impl;

import com.cache.CacheFactory;
import com.cache.constants.CacheConstants;
import com.cache.exception.CacheException;
import com.cusc.nirvana.common.result.Response;
import com.cusc.nirvana.user.auth.common.constants.AppConfigConstant;
import com.cusc.nirvana.user.auth.common.constants.RedisConstant;
import com.cusc.nirvana.user.auth.common.constants.ResponseCode;
import com.cusc.nirvana.user.auth.common.dto.AccessTokenHashDTO;
import com.cusc.nirvana.user.auth.common.dto.LogoutDTO;
import com.cusc.nirvana.user.auth.common.dto.RefreshTokenHashDTO;
import com.cusc.nirvana.user.auth.common.dto.UserTokenListDTO;
import com.cusc.nirvana.user.auth.common.service.AppConfigService;
import com.cusc.nirvana.user.auth.identification.dto.MobileLoginReq;
import com.cusc.nirvana.user.auth.identification.dto.Oauth2Token;
import com.cusc.nirvana.user.auth.identification.dto.UserLoginResp;
import com.cusc.nirvana.user.auth.identification.dto.UserNameLoginReq;
import com.cusc.nirvana.user.auth.identification.service.ITokenService;
import com.cusc.nirvana.user.eiam.dto.ApplicationDTO;
import com.cusc.nirvana.user.eiam.dto.UserDTO;
import com.cusc.nirvana.user.eiam.dto.UserOrganDTO;
import com.cusc.nirvana.user.eiam.service.IUserOrganService;
import com.cusc.nirvana.user.util.CuscStringUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

/**
 * Description: 令牌service实现类
 * <br />
 * CreateDate 2021-11-02 20:25:49
 *
 * @author yuyi
 **/
@Service
@Slf4j
public class TokenServiceImpl implements ITokenService {

    @Autowired
    private CacheFactory cacheFactory;

    @Autowired
    private AppConfigService appConfigService;

    @Resource
	private IUserOrganService userOrganService;


    /**
     * Description: 创建Oauth2Token
     * <br />
     * CreateDate 2021-11-04 19:53:41
     *
     * @author yuyi
     **/
    @Override
    public Response<Oauth2Token> createOauth2TokenByMobile(MobileLoginReq bean, UserDTO user) {

        //创建用户信息
        UserLoginResp userLogin = new UserLoginResp();
        userLogin.setUserId(user.getUuid());
        userLogin.setLoginName(bean.getPhone().substring(0, 3) + "****"
                + bean.getPhone().substring(bean.getPhone().length() - 4));

        return createOauth2Token(userLogin, bean.getApplicationId(), user.getTenantNo());
    }


    /**
     * Description: 创建Oauth2Token小鹏
     * <br />
     * CreateDate 2021-11-04 19:53:41
     *
     * @author yuyi
     **/
    @Override
    public Response<Oauth2Token> createOauth2TokenByMobileXP(MobileLoginReq bean, UserDTO user) {

        //创建用户信息
        UserLoginResp userLogin = new UserLoginResp();
        userLogin.setUserId(UUID.randomUUID().toString().replace("-", ""));
        userLogin.setLoginName(bean.getPhone().substring(0, 3) + "****"
                + bean.getPhone().substring(bean.getPhone().length() - 4));

        return createOauth2Token(userLogin, bean.getApplicationId(), user.getTenantNo());
    }
    @Override
    public Response<Oauth2Token> createOauth2TokenByUserName(UserNameLoginReq bean, UserDTO user) {
        //创建用户信息
        UserLoginResp userLogin = new UserLoginResp();
        userLogin.setUserId(user.getUuid());
        userLogin.setLoginName(bean.getUserName());
        return createOauth2Token(userLogin, bean.getApplicationId(), user.getTenantNo());
    }

    /**
     * Description: 退出
     * <br />
     * CreateDate 2021-11-09 16:09:36
     *
     * @author yuyi
     **/
    @Override
    public Response logout(LogoutDTO logoutDTO) {
        String accessToken = logoutDTO.getAccessToken();
        AccessTokenHashDTO accessTokenHashDTO;
        try {
            accessTokenHashDTO = cacheFactory.getExpireHashService()
                    .getHash(RedisConstant.TOKEN_ACCESS_TOKEN_INFO + accessToken,
                            AccessTokenHashDTO.class);
            if (accessTokenHashDTO != null) {
                //删除刷新token
                cacheFactory.getExpireHashService()
                        .delete(RedisConstant.TOKEN_REFRESH_TOKEN_INFO + accessTokenHashDTO.getRefresh());

                String userTokenListKey = RedisConstant.TOKEN_USER_TOKEN_INFO + accessTokenHashDTO.getTenantNo()
                        + ":" + accessTokenHashDTO.getUserId();
                //删除用户对应的token
                List<UserTokenListDTO> userList =
                        cacheFactory.getExpireListService().getList(userTokenListKey, UserTokenListDTO.class);
                userList = deleteListContent(userList, accessToken);

                //删除用户token集合
                cacheFactory.getListService().delete(userTokenListKey);
                if (!CollectionUtils.isEmpty(userList)) {
                    int userTokenInfoTtl =
                            Math.toIntExact(cacheFactory.getListService().getKeyExpireTime(userTokenListKey));
                    cacheFactory.getExpireListService().setExpireList(userTokenListKey, userList, userTokenInfoTtl);
                }
                //删除访问token
                cacheFactory.getExpireHashService()
                        .delete(RedisConstant.TOKEN_ACCESS_TOKEN_INFO + accessToken);
            }
        } catch (CacheException e) {
            log.error("logout 访问reids失败 :{}", e);
            return Response.createError(ResponseCode.LOGOUT_FAIL.getMsg(), ResponseCode.LOGOUT_FAIL.getCode() + "");
        }
        return Response.createSuccess();
    }

    @Override
    public Response kickOutByUserId(String userId, String tenantNo, String appId) {
        //通过用户id和应用id找到对应的token信息
        try {
            String userTokenListKey = RedisConstant.TOKEN_USER_TOKEN_INFO + tenantNo + ":" + userId;
            List<UserTokenListDTO> userList =
                    cacheFactory.getExpireListService().getList(userTokenListKey, UserTokenListDTO.class);
            if (CollectionUtils.isEmpty(userList)) {
                return Response.createSuccess();
            }
            List<UserTokenListDTO> userListNew = new ArrayList<>();
            for (UserTokenListDTO userToken : userList) {
                if (!userToken.getAppId().equals(appId)) {
                    userListNew.add(userToken);
                    continue;
                }
                //删除刷新token
                cacheFactory.getExpireHashService()
                        .delete(RedisConstant.TOKEN_REFRESH_TOKEN_INFO + userToken.getRefresh());

                //删除访问token
                cacheFactory.getExpireHashService()
                        .delete(RedisConstant.TOKEN_ACCESS_TOKEN_INFO + userToken.getAccess());
            }

            if (!CollectionUtils.isEmpty(userListNew)) {
                int userTokenInfoTtl =
                        Math.toIntExact(cacheFactory.getListService().getKeyExpireTime(userTokenListKey));
                cacheFactory.getExpireListService().delete(userTokenListKey);
                cacheFactory.getExpireListService().setExpireList(userTokenListKey, userListNew, userTokenInfoTtl);
            }
        } catch (CacheException e) {
            log.error("kickOutByUserId 访问reids失败 :{}", e);
            return Response.createError(ResponseCode.KICK_OUT_FAIL.getMsg(), ResponseCode.KICK_OUT_FAIL.getCode() + "");
        }
        return Response.createSuccess();
    }

    /**
     * Description: access token续期
     * <br />
     * CreateDate 2022-01-10 15:24:51
     *
     * @author yuyi
     **/
    @Override
    public Response tokenRenewal(String accessToken, String appId) {
        ApplicationDTO appBean = appConfigService.getAppConfigByCode(appId);
        try {
            cacheFactory.getExpireHashService()
                    .expireKey(RedisConstant.TOKEN_REFRESH_TOKEN_INFO + accessToken, appBean.getRenewalTokenTime(),
                            CacheConstants.TimeType.EX);

            //同时续期用户id对应的token信息
            AccessTokenHashDTO accessTokenHashDTO = cacheFactory.getExpireHashService()
                    .getHash(RedisConstant.TOKEN_ACCESS_TOKEN_INFO + accessToken,
                            AccessTokenHashDTO.class);
            if (accessTokenHashDTO != null) {
                cacheFactory.getExpireListService()
                        .updateExpire(RedisConstant.TOKEN_USER_TOKEN_INFO + accessTokenHashDTO.getTenantNo() + ":"
                                        + accessTokenHashDTO.getUserId(),
                                appBean.getRenewalTokenTime());
            }
        } catch (CacheException e) {
            log.error("tokenRenewal 访问reids失败 :{}", e);
            return Response.createError();
        }
        return Response.createSuccess();
    }


    //--------------------------私有方法区--------------------------------

    /**
     * Description: 检查已过期的key
     * <br />
     * CreateDate 2021-11-05 12:50:00
     *
     * @author yuyi
     **/
    private List<UserTokenListDTO> checkExpireContent(List<UserTokenListDTO> userList) {
        if (!CollectionUtils.isEmpty(userList)) {
            List<UserTokenListDTO> ret = new ArrayList<>();
            for (UserTokenListDTO userListDTO : userList) {
                //只需要检查refresh token是否已过期，过期则删除
                try {
                    Long expireIn = cacheFactory.getExpireHashService()
                            .getKeyExpireTime(RedisConstant.TOKEN_REFRESH_TOKEN_INFO + userListDTO.getRefresh());
                    if (expireIn > 0) {
                        ret.add(userListDTO);
                    }
                } catch (CacheException e) {
                    log.error("checkExpireContent 获取reids失败 :{}", e);
                    ret.add(userListDTO);
                }
            }
            return ret;
        }
        return userList;
    }

    /**
     * Description: 删除list中的元素
     * <br />
     * CreateDate 2021-11-05 12:50:00
     *
     * @author yuyi
     **/
    private List<UserTokenListDTO> deleteListContent(List<UserTokenListDTO> userList, String accessToken) {
        if (!CollectionUtils.isEmpty(userList)) {
            List<UserTokenListDTO> ret = new ArrayList<>();
            for (UserTokenListDTO userListDTO : userList) {
                if (!accessToken.equals(userListDTO.getAccess())) {
                    ret.add(userListDTO);
                }
            }
            return ret;
        }
        return userList;
    }




    /**
     * Description: 创建Oauth2 Token信息
     * <br />
     * CreateDate 2022-01-10 15:29:37
     *
     * @author yuyi
     **/
    private Response<Oauth2Token> createOauth2Token(UserLoginResp userLogin, String appId, String tenantNo) {
        String userId = userLogin.getUserId();
        if(StringUtils.isEmpty(userId)){
            return Response.createError(ResponseCode.TOEKN_CREATE_FAIL.getMsg()+",userId为空",
                    ResponseCode.TOEKN_CREATE_FAIL.getCode() + "");
        }
        Oauth2Token oauth2Token = new Oauth2Token();
        //创建token
        oauth2Token.setAccess_token(CuscStringUtils.generateUuid());
        oauth2Token.setRefresh_token(CuscStringUtils.generateUuid());
        oauth2Token.setToken_type("bearer");
        oauth2Token.setScope("ALL");

        //读取应用配置信息
        ApplicationDTO appBean = appConfigService.getAppConfigByCode(appId);

        //设置token失效时间
        oauth2Token.setExpires_in(appBean.getAccessTokenTerm());


        if (appBean.getIsDeviceLogin() != null && AppConfigConstant.IS_DEVICE_LOGIN_1 == appBean.getIsDeviceLogin()) {
            //单设备登录
            kickOutByUserId(userId, tenantNo, appId);
        }

        //创建用户信息
        oauth2Token.setInfo(userLogin);

        //存放redis
        try {
			String organId = "";

			//用户对应的组织id
			UserOrganDTO userOrgDto = new UserOrganDTO();
			userOrgDto.setUserId(userId);
			userOrgDto.setTenantNo(tenantNo);
			List<UserOrganDTO> userOrganList = userOrganService.queryByList(userOrgDto);
			if (!CollectionUtils.isEmpty(userOrganList)) {
				organId = userOrganList.stream().filter(userOrganDTO -> userId.equals(userOrganDTO.getUserId()))
						.findFirst().map(UserOrganDTO::getOrganId).orElse("");
			}


			//access_token对应的用户id、refresh_token、scope  map格式
            AccessTokenHashDTO accessTokenHashDTO = new AccessTokenHashDTO();
            accessTokenHashDTO.setUserId(userId);
            accessTokenHashDTO.setRefresh(oauth2Token.getRefresh_token());
            accessTokenHashDTO.setScope(oauth2Token.getScope());
            accessTokenHashDTO.setAppId(appId);
            accessTokenHashDTO.setTenantNo(tenantNo);
			accessTokenHashDTO.setOrganId(organId);
            cacheFactory.getExpireHashService()
                    .setExpireHash(RedisConstant.TOKEN_ACCESS_TOKEN_INFO + oauth2Token.getAccess_token(),
                            accessTokenHashDTO,
                            appBean.getAccessTokenTerm());

            //refresh_token对应的token和用户id
            RefreshTokenHashDTO refreshTokenHashDTO = new RefreshTokenHashDTO();
            refreshTokenHashDTO.setAccess(oauth2Token.getAccess_token());
            refreshTokenHashDTO.setUserId(userId);
            refreshTokenHashDTO.setScope(oauth2Token.getScope());
            refreshTokenHashDTO.setAppId(appId);
            refreshTokenHashDTO.setTenantNo(tenantNo);
            cacheFactory.getExpireHashService()
                    .setExpireHash(RedisConstant.TOKEN_REFRESH_TOKEN_INFO + oauth2Token.getRefresh_token(),
                            refreshTokenHashDTO,
                            appBean.getRefreshTokenTerm());

            //用户id对应的access_token、refresh_token

            String userTokenListKey = RedisConstant.TOKEN_USER_TOKEN_INFO + tenantNo + ":" + userId;
            List<UserTokenListDTO> userList =
                    cacheFactory.getExpireListService()
                            .getList(userTokenListKey,
                                    UserTokenListDTO.class);
            if (CollectionUtils.isEmpty(userList)) {
                userList = new ArrayList<>();
            }
            //检查现有list中是否有已过期的key,过期则删除
            userList = checkExpireContent(userList);

            UserTokenListDTO userListDTO = new UserTokenListDTO();
            userListDTO.setAccess(oauth2Token.getAccess_token());
            userListDTO.setRefresh(oauth2Token.getRefresh_token());
            userListDTO.setAppId(appId);
            userList.add(userListDTO);
            //先删除后新增
            cacheFactory.getExpireListService().delete(userTokenListKey);
            cacheFactory.getExpireListService().setExpireList(userTokenListKey, userList, appBean.getAccessTokenTerm());
        } catch (CacheException e) {
            log.error("createOauth2Token 存放reids失败 :{}", e);
            return Response.createError(ResponseCode.TOEKN_CREATE_FAIL.getMsg(),
                    ResponseCode.TOEKN_CREATE_FAIL.getCode() + "");
        }
        return Response.createSuccess(oauth2Token);
    }



    /**
     * Description: 创建Oauth2 Token信息小鹏
     * <br />
     * CreateDate 2022-01-10 15:29:37
     *
     * @author yuyi
     **/
    private Response<Oauth2Token> createOauth2TokenXP(UserLoginResp userLogin, String appId, String tenantNo) {
        String userId = userLogin.getUserId();
        Oauth2Token oauth2Token = new Oauth2Token();
        //创建token
        oauth2Token.setAccess_token(CuscStringUtils.generateUuid());
        oauth2Token.setRefresh_token(CuscStringUtils.generateUuid());
        oauth2Token.setToken_type("bearer");
        oauth2Token.setScope("ALL");

        //读取应用配置信息
        ApplicationDTO appBean = appConfigService.getAppConfigByCode(appId);

        //设置token失效时间
        oauth2Token.setExpires_in(appBean.getAccessTokenTerm());


        if (appBean.getIsDeviceLogin() != null && AppConfigConstant.IS_DEVICE_LOGIN_1 == appBean.getIsDeviceLogin()) {
            //单设备登录
            kickOutByUserId(userId, tenantNo, appId);
        }

        //创建用户信息
        oauth2Token.setInfo(userLogin);

        //存放redis
        try {
            //access_token对应的用户id、refresh_token、scope  map格式
            AccessTokenHashDTO accessTokenHashDTO = new AccessTokenHashDTO();
            accessTokenHashDTO.setUserId(userId);
            accessTokenHashDTO.setRefresh(oauth2Token.getRefresh_token());
            accessTokenHashDTO.setScope(oauth2Token.getScope());
            accessTokenHashDTO.setAppId(appId);
            accessTokenHashDTO.setTenantNo(tenantNo);
            cacheFactory.getExpireHashService()
                    .setExpireHash(RedisConstant.TOKEN_ACCESS_TOKEN_INFO + oauth2Token.getAccess_token(),
                            accessTokenHashDTO,
                            appBean.getAccessTokenTerm());

            //refresh_token对应的token和用户id
            RefreshTokenHashDTO refreshTokenHashDTO = new RefreshTokenHashDTO();
            refreshTokenHashDTO.setAccess(oauth2Token.getAccess_token());
            refreshTokenHashDTO.setUserId(userId);
            refreshTokenHashDTO.setScope(oauth2Token.getScope());
            refreshTokenHashDTO.setAppId(appId);
            refreshTokenHashDTO.setTenantNo(tenantNo);
            cacheFactory.getExpireHashService()
                    .setExpireHash(RedisConstant.TOKEN_REFRESH_TOKEN_INFO + oauth2Token.getRefresh_token(),
                            refreshTokenHashDTO,
                            appBean.getRefreshTokenTerm());

            //用户id对应的access_token、refresh_token
            String userTokenListKey = RedisConstant.TOKEN_USER_TOKEN_INFO + tenantNo + ":" + userId;
            List<UserTokenListDTO> userList =
                    cacheFactory.getExpireListService()
                            .getList(userTokenListKey,
                                    UserTokenListDTO.class);
            if (CollectionUtils.isEmpty(userList)) {
                userList = new ArrayList<>();
            }
            //检查现有list中是否有已过期的key,过期则删除
            userList = checkExpireContent(userList);

            UserTokenListDTO userListDTO = new UserTokenListDTO();
            userListDTO.setAccess(oauth2Token.getAccess_token());
            userListDTO.setRefresh(oauth2Token.getRefresh_token());
            userListDTO.setAppId(appId);
            userList.add(userListDTO);
            //先删除后新增
            cacheFactory.getExpireListService().delete(userTokenListKey);
            cacheFactory.getExpireListService().setExpireList(userTokenListKey, userList, appBean.getAccessTokenTerm());
        } catch (CacheException e) {
            log.error("createOauth2Token 存放reids失败 :{}", e);
            return Response.createError(ResponseCode.TOEKN_CREATE_FAIL.getMsg(),
                    ResponseCode.TOEKN_CREATE_FAIL.getCode() + "");
        }
        return Response.createSuccess(oauth2Token);
    }
}
