package com.cusc.nirvana.user.auth.identification.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.TypeReference;
import com.cache.CacheFactory;
import com.cache.exception.CacheException;
import com.cusc.nirvana.common.encrypt.sign.HMAC;
import com.cusc.nirvana.common.result.Response;
import com.cusc.nirvana.user.auth.common.constants.ResponseCode;
import com.cusc.nirvana.user.auth.common.dto.SmsResponseDTO;
import com.cusc.nirvana.user.auth.common.dto.SmsSendDTO;
import com.cusc.nirvana.user.auth.identification.dto.SmsSendConfig;
import com.cusc.nirvana.user.auth.identification.service.ISmsService;
import com.cusc.nirvana.user.config.SignConstants;
import com.cusc.nirvana.user.config.SmsPropertyConfig;
import com.cusc.nirvana.user.eiam.dto.ApplicationDTO;
import com.cusc.nirvana.user.exception.CuscUserException;
import com.cusc.nirvana.user.util.CuscStringUtils;
import com.cusc.nirvana.user.util.DateUtils;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;

/**
 * Description: 短信service
 * <br />
 * CreateDate 2021-11-02 20:25:49
 *
 * @author yuyi
 **/
@Service
@Slf4j
public class SmsServiceImpl implements ISmsService {

    private static final Logger LOGGER = LoggerFactory.getLogger(SmsServiceImpl.class);

    @Autowired
    private RestTemplate restTemplate;

    @Autowired
    private SmsPropertyConfig smsPropertyConfig;

    @Autowired
    private CacheFactory cacheFactory;

    @Value("${sms.cusc.strategyCode:}")
    private String strategyCode;

    @Value("${sms.cusc.templateCode:}")
    private String smsTemplate;

    @Override
    public SmsResponseDTO sendSms(String phone, List<String> paramterList, SmsSendConfig config) {
        SmsSendDTO send = new SmsSendDTO();
        if (CuscStringUtils.isEmpty(smsPropertyConfig.getAccessKey())) {
            send.setAccesskey(config.getSmsPlatformKey());
        } else {
            send.setAccesskey(smsPropertyConfig.getAccessKey());
        }

        List<String> phoneList = new ArrayList<>();
        phoneList.add(phone);
        send.setPhoneNumbers(phoneList);
        send.setTemplateParams(paramterList);
        if (CuscStringUtils.isEmpty(smsPropertyConfig.getSignatureCode())) {
            send.setSignatureCode(config.getSmsSignatureCode());
        } else {
            send.setSignatureCode(smsPropertyConfig.getSignatureCode());
        }
        send.setStrategyCode(strategyCode);
        send.setTemplateCode(smsTemplate);
        Response<SmsResponseDTO> retResp;
        try {
            //判断当前验证码是否达到发送间隔
            String intervalKey =
                    config.getIntervalLimitKey() + phone + "_" + config.getTenantNo() + "_" + config.getAppId();
            int expireLock = config.getSmsIntervalLimit() * 1000;
            log.info("sendSms intervalKey : {} , expire : {}", intervalKey, expireLock);
            if (!cacheFactory.getLockService().lock(intervalKey, expireLock)) {
                throw new CuscUserException(ResponseCode.SMS_CAPTCHA_INTERVAL_FAIL.getCode(),
                        ResponseCode.SMS_CAPTCHA_INTERVAL_FAIL.getMsg());
            }

            HttpEntity httpEntity = new HttpEntity(JSON.toJSONString(send), headers());

            String url = smsPropertyConfig.getSmsUrl() + smsPropertyConfig.getSendUrl();

            log.info("SmsServiceImpl sendSms 短信平台  url : {}, request : {}",
                    url, JSON.toJSONString(httpEntity));
            ResponseEntity<String> entity =
                    restTemplate.exchange(url, HttpMethod.POST, httpEntity, String.class);
            retResp = JSON.parseObject(entity.getBody(),
                    new TypeReference<Response<SmsResponseDTO>>(SmsResponseDTO.class) {
                    }.getType());
            log.info("SmsServiceImpl sendSms 短信平台  url : {}, response : {}", url, JSON.toJSONString(retResp));
        } catch (Exception e) {
            LOGGER.error("短信发送失败： ", e);
            throw new CuscUserException(ResponseCode.SMS_CAPTCHA_SEND_FAIL.getCode() + "",
                    ResponseCode.SMS_CAPTCHA_SEND_FAIL.getMsg());
        }

        //记录短信发送次数和间隔
        saveSmsSendLimitToRedis(phone, config);

        if (retResp != null) {
            return retResp.getData();
        }
        return null;
    }

    @Override
    public SmsResponseDTO sendSms(String phone, String parameter, SmsSendConfig config) {
        List<String> list = new ArrayList<>();
        list.add(parameter);
        return sendSms(phone, list, config);
    }

    @Override
    public boolean checkSmsConfigNotNull(SmsSendConfig bean) {
        //return bean != null && CuscStringUtils.isNotEmpty(bean.getSmsTemplateCode());
        return bean != null;
    }

    @Override
    public void convertToSmsConfig(ApplicationDTO fromBean, SmsSendConfig toBean) {
        //短信配置为空，从应用配置中取
        if (!checkSmsConfigNotNull(toBean)) {
            throw new CuscUserException(ResponseCode.SMS_SEND_CONFIG_NOT_NULL.getCode() + "",
                    ResponseCode.SMS_SEND_CONFIG_NOT_NULL.getMsg());
        }

        if (toBean.getSmsTotalLimit() == null) {
            if (fromBean.getSmsTotalLimit() == null) {
                log.warn("sms config smsTotalLimit is null");
                throw new CuscUserException(ResponseCode.SMS_SEND_CONFIG_NOT_NULL.getCode() + "",
                        ResponseCode.SMS_SEND_CONFIG_NOT_NULL.getMsg());
            }
            toBean.setSmsTotalLimit(fromBean.getSmsTotalLimit());
        }

        if (toBean.getSmsIntervalLimit() == null) {
            if (fromBean.getSmsIntervalLimit() == null) {
                log.warn("sms config smsIntervalLimit is null");
                throw new CuscUserException(ResponseCode.SMS_SEND_CONFIG_NOT_NULL.getCode() + "",
                        ResponseCode.SMS_SEND_CONFIG_NOT_NULL.getMsg());
            }
            toBean.setSmsIntervalLimit(fromBean.getSmsIntervalLimit());
        }

        if (toBean.getSmsPlatformKey() == null) {
            if (fromBean.getSmsPlatformKey() == null) {
                log.warn("sms config smsPlatformKey is null");
                throw new CuscUserException(ResponseCode.SMS_SEND_CONFIG_NOT_NULL.getCode() + "",
                        ResponseCode.SMS_SEND_CONFIG_NOT_NULL.getMsg());
            }
            toBean.setSmsPlatformKey(fromBean.getSmsPlatformKey());
        }

        if (toBean.getSmsSignatureCode() == null) {
            if (fromBean.getSmsSignatureCode() == null) {
                log.warn("sms config smsSignatureCode is null");
                throw new CuscUserException(ResponseCode.SMS_SEND_CONFIG_NOT_NULL.getCode() + "",
                        ResponseCode.SMS_SEND_CONFIG_NOT_NULL.getMsg());
            }
            toBean.setSmsSignatureCode(fromBean.getSmsSignatureCode());
        }
    }


    /**
     * Description: 短信发送限制检查
     * <br />
     * CreateDate 2022-01-27 14:43:41
     *
     * @author yuyi
     **/
    @Override
    public void checkSmsSendLimit(String phone, SmsSendConfig bean) {
        try {
            if (bean.getSmsTotalLimit() != null && bean.getSmsTotalLimit() > 0 && CuscStringUtils.isNotEmpty(
                    bean.getTotalLimitKey())) {
                //记录发送总次数限制
                Integer smsTotal = cacheFactory.getExpireStringService()
                        .getValue(bean.getTotalLimitKey() + phone + "_" + bean.getTenantNo() + "_" + bean.getAppId(),
                                Integer.class);
                if (smsTotal != null && smsTotal >= bean.getSmsTotalLimit()) {
                    throw new CuscUserException(ResponseCode.SMS_TOTAL_LIMIT_OVERRUN.getCode(),
                            ResponseCode.SMS_TOTAL_LIMIT_OVERRUN.getMsg());
                }
            }

            if (bean.getSmsIntervalLimit() != null && bean.getSmsIntervalLimit() > 0 && CuscStringUtils.isNotEmpty(
                    bean.getIntervalLimitKey())) {
                //记录发送间隔限制
                boolean isExists =
                        cacheFactory.getExpireStringService()
                                .containsKey(bean.getIntervalLimitKey() + phone + "_" + bean.getTenantNo() + "_"
                                        + bean.getAppId());
                if (isExists) {
                    throw new CuscUserException(ResponseCode.SMS_INTERVAL_LIMIT_OVERRUN.getCode(),
                            ResponseCode.SMS_INTERVAL_LIMIT_OVERRUN.getMsg());
                }
            }
        } catch (Exception e) {
            //只记录，不抛出异常，屏蔽对业务的影响
            log.error("检查短信发送限制信息时访问redis 异常：", e);
        }
    }

    //----------------私有方法区域--------------------------

    /**
     * Description: 保存短信发送限制信息到redis
     * <br />
     * CreateDate 2022-02-16 09:50:25
     *
     * @author yuyi
     **/
    private void saveSmsSendLimitToRedis(String phone, SmsSendConfig bean) {
        try {
            if (bean.getSmsTotalLimit() != null && bean.getSmsTotalLimit() > 0 && CuscStringUtils.isNotEmpty(
                    bean.getTotalLimitKey())) {
                //记录发送总次数限制
                Integer smsTotal =
                        cacheFactory.getExpireStringService().getValue(
                                bean.getTotalLimitKey() + phone + "_" + bean.getTenantNo() + "_" + bean.getAppId(),
                                Integer.class);
                Long expireTime;
                if (smsTotal == null) {
                    smsTotal = 1;
                    LocalDateTime begin = LocalDateTime.now();
                    expireTime = DateUtils.secondBetween(begin, DateUtils.getDayEnd(begin));
                } else {
                    smsTotal++;
                    expireTime =
                            cacheFactory.getExpireStringService().getKeyExpireTime(
                                    bean.getTotalLimitKey() + phone + "_" + bean.getTenantNo() + "_" + bean.getAppId());
                }
                cacheFactory.getExpireStringService().setExpireValue(
                        bean.getTotalLimitKey() + phone + "_" + bean.getTenantNo() + "_" + bean.getAppId(), smsTotal,
                        expireTime.intValue());
            }
        } catch (CacheException e) {
            //只记录，不抛出异常，屏蔽对业务的影响
            log.error("保存短信发送限制信息到redis 异常：", e);
        }
    }

    /**
     * 生成请求头
     *
     * @return
     */
    public HttpHeaders headers() {
        HttpHeaders httpHeaders = new HttpHeaders();
        httpHeaders.add(SignConstants.APP_ID, smsPropertyConfig.getAPPID());
        httpHeaders.add(SignConstants.NONCE_STR, CuscStringUtils.generateUuid());
        httpHeaders.add(SignConstants.TIMESTAMP, String.valueOf(System.currentTimeMillis()));
        httpHeaders.add(SignConstants.VERSION, smsPropertyConfig.getVERSION());
        httpHeaders.setContentType(MediaType.parseMediaType("application/json; charset=UTF-8"));
        StringBuilder sb = new StringBuilder();
        sb.append(SignConstants.APP_ID + smsPropertyConfig.getAPPID());
        sb.append(SignConstants.NONCE_STR + httpHeaders.get(SignConstants.NONCE_STR).get(0));
        sb.append(SignConstants.TIMESTAMP + httpHeaders.get(SignConstants.TIMESTAMP).get(0));
        sb.append(SignConstants.VERSION + httpHeaders.get(SignConstants.VERSION).get(0));
        String scret = HMAC.sign(sb.toString(), smsPropertyConfig.getAPPSCRET(), HMAC.Type.HmacSHA256);
        httpHeaders.add(SignConstants.SIGN, scret);
        return httpHeaders;
    }
}
