package com.cusc.nirvana.user.auth.identification.service.impl;

import com.cache.CacheFactory;
import com.cache.exception.CacheException;
import com.cusc.nirvana.user.auth.common.constants.RedisConstant;
import com.cusc.nirvana.user.auth.common.constants.ResponseCode;
import com.cusc.nirvana.user.auth.identification.dto.CaptchaCreateReq;
import com.cusc.nirvana.user.auth.identification.dto.CaptchaCreateResp;
import com.cusc.nirvana.user.auth.identification.dto.CaptchaVerificationReq;
import com.cusc.nirvana.user.auth.identification.service.ICaptchaService;
import com.cusc.nirvana.user.config.SmsPropertyConfig;
import com.cusc.nirvana.user.exception.CuscUserException;
import com.wf.captcha.SpecCaptcha;
import com.wf.captcha.base.Captcha;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.awt.FontFormatException;
import java.io.*;
import java.util.UUID;

/**
 * Description: 图形验证码业务层
 * <br />
 * CreateDate 2022-01-24 10:32:09
 *
 * @author yuyi
 **/
@Service
@Slf4j
public class CaptchaServiceImpl implements ICaptchaService {

    //验证码的失效时间：秒
    private static final int CAPTCHA_EXPIRE_TIME = 10800;

    @Autowired
    private CacheFactory cacheFactory;

    @Autowired
    private SmsPropertyConfig smsPropertyConfig;


    /**
     * Description: 生成图形验证码
     * <br />
     * CreateDate 2022-01-24 10:32:21
     *
     * @param bean 图形验证码请求对象
     * @author yuyi
     **/
    @Override
    public CaptchaCreateResp generateCaptcha(CaptchaCreateReq bean) {
        CaptchaCreateResp ret = new CaptchaCreateResp();
        ret.setRequestId(UUID.randomUUID().toString());
        //生成随机4位
        SpecCaptcha captcha = new SpecCaptcha(bean.getCaptchaWidth(), bean.getCaptchaHeight(), bean.getCaptchaLength());
        captcha.setCharType(bean.getCaptchaType().getCode());
        //不区分大小写
        String code = captcha.text().toLowerCase();
        //放到redis
        try {
            cacheFactory.getExpireStringService()
                    .setExpireValue(
                            RedisConstant.IMAGE_CAPTCHA_KEY + ret.getRequestId() + "_" + bean.getApplicationId(),
                            code, CAPTCHA_EXPIRE_TIME);
        } catch (CacheException e) {
            log.error("generateCaptcha 请求reids失败 ：", e);
            throw new CuscUserException(ResponseCode.REDIS_OPT_FAIL.getCode() + "",
                    ResponseCode.REDIS_OPT_FAIL.getMsg());
        }
        try {
            captcha.setFont(Captcha.FONT_2);
        } catch (IOException e) {
            log.error("generateCaptcha IOException 设置字体失败 ：", e);
        } catch (FontFormatException e) {
            log.error("generateCaptcha FontFormatException 设置字体失败 ：", e);
        }
        ret.setCaptchaImg(captcha.toBase64());
        return ret;
    }

    /**
     * Description: 验证图形验证码
     * <br />
     * CreateDate 2022-01-24 10:32:21
     *
     * @param bean 图形验证码请求对象
     * @author yuyi
     **/
    @Override
    public boolean verificationCaptcha(CaptchaVerificationReq bean) {
        // 获取redis中的验证码
        String redisCode;
        try {

            String imageCaptchaKey =
                    RedisConstant.IMAGE_CAPTCHA_KEY + bean.getRequestId() + "_" + bean.getApplicationId();

            redisCode =
                    cacheFactory.getExpireStringService()
                            .getValue(imageCaptchaKey, String.class);
            //清理图形验证码
            cacheFactory.getExpireStringService().delete(imageCaptchaKey);
        } catch (CacheException e) {
            log.error("verificationCaptcha 请求reids失败 ：", e);
            throw new CuscUserException(ResponseCode.REDIS_OPT_FAIL.getCode() + "",
                    ResponseCode.REDIS_OPT_FAIL.getMsg());
        }
        // 判断验证码
        if (bean.getCaptchaValue().toLowerCase().equals(redisCode)) {
            return true;
        }
        log.info("verificationCaptcha false . param requestId:{} , input code:{} , redis code:{} ",
                bean.getRequestId(), bean.getCaptchaValue(), redisCode);
        return false;
    }

    /**
     * Description: 检查短信验证码验证错误次数
     * <br />
     * CreateDate 2022-07-08 19:41:51
     *
     * @author yuyi
     **/
    public void checkSmsCaptchaErrorCount(String phone, String tenantNo, String appId) {
        Integer errorCount;
        try {
            String redisKey =
                    RedisConstant.SMS_CAPTCHA_ERROR_COUNT_KEY + phone + "_" + tenantNo + "_"
                            + appId;
            errorCount = cacheFactory.getExpireStringService().getValue(redisKey, Integer.class);
            if (errorCount == null) {
                errorCount = 0;
            }
            errorCount++;
            if (errorCount.intValue() > smsPropertyConfig.errorCount.intValue()) {
                //超过错误次数之后清理验证码
                cacheFactory.getExpireStringService()
                        .delete(RedisConstant.SMS_CAPTCHA_KEY + phone + "_" + tenantNo + "_"
                                + appId);
                cacheFactory.getExpireStringService().delete(redisKey);
                return;
            }
            cacheFactory.getExpireStringService()
                    .setExpireValue(redisKey, errorCount, RedisConstant.SMS_CAPTCHA_ERROR_COUNT_EXPIRE);
        } catch (CacheException e) {
            log.error("checkSmsCaptchaErrorCount 检查短信验证码验证错误次数 访问redis异常：", e);
        }
    }

    @Override
    public void delSmsCaptchaErrorCount(String phone, String tenantNo, String appId) {
        try {
            cacheFactory.getExpireStringService()
                    .delete(RedisConstant.SMS_CAPTCHA_ERROR_COUNT_KEY + phone + "_" + tenantNo + "_"
                            + appId);
        } catch (CacheException e) {
            log.error("delSmsCaptchaErrorCount 检查短信验证码验证错误次数 访问redis异常：", e);
        }
    }
}
