package com.cusc.nirvana.user.auth.identification.login;

import com.cusc.nirvana.common.result.Response;
import com.cusc.nirvana.user.auth.common.constants.ResponseCode;
import com.cusc.nirvana.user.auth.identification.dto.RandomIdReq;
import com.cusc.nirvana.user.auth.identification.dto.RandomIdResp;
import com.cusc.nirvana.user.auth.identification.service.IRandomIdService;
import com.cusc.nirvana.user.util.CuscRequestUtil;
import com.cusc.nirvana.user.util.CuscStringUtils;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletRequest;


@Slf4j
@RestController
@RequestMapping("/random")
public class RandomIdController {

    private static final Logger LOGGER = LoggerFactory.getLogger(RandomIdController.class);

    //随机请求id的失效时间：秒   3小时
    private static final int REQUEST_ID_EXPIRE_TIME = 10800;

    @Autowired
    private IRandomIdService randomIdService;

    @PostMapping("/getRequestIdByCaptcha")
    @ApiOperation(value = "获取随机请求id-图形验证码", notes = "获取随机请求id-图形验证码")
    public Response<String> getRequestIdByCaptcha(@RequestBody RandomIdReq bean, HttpServletRequest request) {
        if (bean == null || CuscStringUtils.isEmpty(bean.getApplicationId())) {
            return Response.createError(ResponseCode.PARAMETER_NULL.getMsg(),
                    ResponseCode.PARAMETER_NULL.getCode());
        }
        Response ret = Response.createSuccess();
        ret.setData(randomIdService.getRequestIdToRedis(bean.getApplicationId(), REQUEST_ID_EXPIRE_TIME,
                CuscRequestUtil.getIpAddr(request)));
        return ret;
    }

    @PostMapping("/getRequestIdByLogin")
    @ApiOperation(value = "获取随机请求id-登录密码加密", notes = "获取随机请求id-登录密码加密")
    public Response<RandomIdResp> getRequestIdByLogin(@RequestBody RandomIdReq bean) {
        if (bean == null || CuscStringUtils.isEmpty(bean.getApplicationId())) {
            return Response.createError(ResponseCode.PARAMETER_NULL.getMsg(),
                    ResponseCode.PARAMETER_NULL.getCode());
        }
        int expireTime = REQUEST_ID_EXPIRE_TIME;
        if (bean.getExpireTime() != null && bean.getExpireTime() > 0) {
            expireTime = bean.getExpireTime();
        }
        String secretKey = CuscStringUtils.generateUuid().substring(16);
        String requestId = randomIdService.getRequestIdToRedis(bean.getApplicationId(), expireTime, secretKey);
        RandomIdResp ret = new RandomIdResp();
        ret.setSecretKey(secretKey);
        ret.setRequestId(requestId);
        return Response.createSuccess(ret);
    }

    @PostMapping("/getContentByRequestId")
    @ApiOperation(value = "获取随机请求id对应的内容", notes = "获取随机请求id对应的内容")
    public Response<String> getContentByRequestId(@RequestBody RandomIdReq bean) {
        if (CuscStringUtils.isEmpty(bean.getRequestId()) || CuscStringUtils.isEmpty(bean.getApplicationId())) {
            return Response.createError(ResponseCode.PARAMETER_NULL.getMsg(),
                    ResponseCode.PARAMETER_NULL.getCode());
        }
        Response ret = Response.createSuccess();
        ret.setData(randomIdService.getByRequestId(bean.getRequestId(), bean.getApplicationId()));
        return ret;
    }

    @PostMapping("/delByRequestId")
    @ApiOperation(value = "删除随机请求id对应的内容", notes = "删除随机请求id对应的内容")
    public Response<Boolean> delByRequestId(@RequestBody RandomIdReq bean) {
        if (CuscStringUtils.isEmpty(bean.getRequestId()) || CuscStringUtils.isEmpty(bean.getApplicationId())) {
            return Response.createError(ResponseCode.PARAMETER_NULL.getMsg(),
                    ResponseCode.PARAMETER_NULL.getCode());
        }
        Response ret = Response.createSuccess();
        ret.setData(randomIdService.delRequestIdRedis(bean.getRequestId(), bean.getApplicationId()));
        return ret;
    }

}
