package com.cusc.nirvana.user.eiam.controller;

import com.cusc.nirvana.common.result.Response;
import com.cusc.nirvana.user.eiam.constants.ResponseCode;
import com.cusc.nirvana.user.eiam.dto.UserDTO;
import com.cusc.nirvana.user.eiam.dto.UserResourceDTO;
import com.cusc.nirvana.user.eiam.dto.UserSimpleDTO;
import com.cusc.nirvana.user.eiam.service.IUrlService;
import com.cusc.nirvana.user.eiam.service.IUserResourceService;
import com.cusc.nirvana.user.eiam.service.IUserService;
import com.cusc.nirvana.user.eiam.util.CommonParamterCheck;
import com.cusc.nirvana.user.util.CuscStringUtils;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;

/**
 * 用户资源关系(UserResource)表控制层
 *
 * @author yuy336
 * @since 2022-01-19 13:41:03
 */
@RestController
@RequestMapping("/userResource")
public class UserResourceController {
    /**
     * 服务对象
     */
    @Autowired
    private IUserResourceService userResourceService;

    @Autowired
    private IUserService userService;

    @Autowired
    private IUrlService urlService;

    /**
     * 查询数据
     *
     * @param bean 实体
     * @return 单条数据
     */
    @PostMapping("/queryByList")
    public Response<List<UserResourceDTO>> queryByList(@RequestBody UserResourceDTO bean) {
        return Response.createSuccess(userResourceService.queryByList(bean));
    }

    /**
     * 新增数据
     *
     * @param bean 实体
     * @return 新增结果
     */
    @PostMapping("/add")
    public Response add(@RequestBody UserResourceDTO bean) {
        return Response.createSuccess(userResourceService.insert(bean));
    }

    @PostMapping("/allUserRelUrlToRedis")
    @ApiOperation(value = "刷新所有用户的URL至缓存", notes = "刷新所有用户的URL至缓存")
    public Response allUserRelUrlToRedis(@RequestBody UserResourceDTO userResDTO) {
        CommonParamterCheck.appIdAndTenantNoRequired(userResDTO.getApplicationId(), userResDTO.getTenantNo());
        //查询所有用户信息
        UserDTO user = new UserDTO();
        user.setTenantNo(userResDTO.getTenantNo());
        List<UserSimpleDTO> userList = userService.queryListUser(user);
        if (!CollectionUtils.isEmpty(userList)) {
            for (UserSimpleDTO userSimpleDTO : userList) {
                urlService.userRelRolUrlToRedis(userSimpleDTO.getUuid(), userResDTO.getTenantNo(),
                        userResDTO.getApplicationId());
            }
        }
        return Response.createSuccess();
    }

    /**
     * 通过用户id批量新增用户资源数据
     *
     * @param bean 实体
     * @return 新增结果
     */
    @PostMapping("/insertBatchResource")
    public Response insertBatchResource(@RequestBody UserResourceDTO bean) {
        if (CuscStringUtils.isEmpty(bean.getUserId())) {
            return Response.createError(ResponseCode.USER_UUID_NOT_NULL.getMsg(),
                    ResponseCode.USER_UUID_NOT_NULL.getCode());
        }
        CommonParamterCheck.appIdAndTenantNoRequired(bean.getApplicationId(), bean.getTenantNo());
        int ret = userResourceService.insertBatchResource(bean);
        //刷新用户对应的url
        urlService.userRelUrlToRedis(bean.getUserId(), bean.getTenantNo(), bean.getApplicationId());
        return Response.createSuccess(ret);
    }

    /**
     * 根据用户id删除用户资源信息
     *
     * @param bean 实体
     * @return 新增结果
     */
    @PostMapping("/deleteByUserId")
    public Response deleteByUserId(@RequestBody UserResourceDTO bean) {
        if (CuscStringUtils.isEmpty(bean.getUserId())) {
            return Response.createError(ResponseCode.USER_UUID_NOT_NULL.getMsg(),
                    ResponseCode.USER_UUID_NOT_NULL.getCode());
        }
        CommonParamterCheck.appIdAndTenantNoRequired(bean.getApplicationId(), bean.getTenantNo());
        return Response.createSuccess(userResourceService.deleteByUserId(bean));
    }
}

