package com.cusc.nirvana.user.auth.authentication.service;

import com.cache.CacheFactory;
import com.cache.exception.CacheException;
import com.cusc.nirvana.common.result.Response;
import com.cusc.nirvana.user.auth.authentication.dto.AccessVerifyReq;
import com.cusc.nirvana.user.auth.common.constants.RedisConstant;
import com.cusc.nirvana.user.auth.common.constants.ResponseCode;
import com.cusc.nirvana.user.auth.common.dto.AccessTokenHashDTO;
import com.cusc.nirvana.user.util.CuscStringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * Description: url白名单service
 * <br />
 * CreateDate 2021-12-01 19:13:36
 *
 * @author yuyi
 **/
@Component
public class AccessAuthService {

    private static final Logger LOGGER = LoggerFactory.getLogger(AccessAuthService.class);

    @Autowired
    private CacheFactory cacheFactory;

    /**
     * Description: 判断url是否是白名单
     * <br />
     * CreateDate 2021-12-01 19:14:45
     *
     * @author yuyi
     **/
    public boolean isWhiteList(AccessVerifyReq bean) {
        try {
            return cacheFactory.getSetService().containsValue(RedisConstant.URL_WHITE_LIST,
                    bean.getServerName() + bean.getUrl());
        } catch (CacheException e) {
            LOGGER.error("isWhiteList fail ", e);
            return false;
        }
    }

    /**
     * Description: 检查token
     * <br />
     * CreateDate 2021-11-05 15:34:02
     *
     * @author yuyi
     **/
    public Response<AccessTokenHashDTO> checkToken(AccessVerifyReq bean) {
        AccessTokenHashDTO accessTokenHashDTO = getTokenInfo(bean);
        if (accessTokenHashDTO == null || CuscStringUtils.isEmpty(accessTokenHashDTO.getUserId())) {
            LOGGER.info("checkToken token:{} 通过token获取信息失败", bean.getToken());
            return Response.createError(ResponseCode.TOKEN_INVALID.getMsg(),
                    ResponseCode.TOKEN_INVALID.getCode());
        }
        return Response.createSuccess(accessTokenHashDTO);
    }

    /**
     * Description: 获取token对应的信息
     * <br />
     * CreateDate 2021-11-05 15:34:02
     *
     * @author yuyi
     **/
    public AccessTokenHashDTO getTokenInfo(AccessVerifyReq bean) {
        AccessTokenHashDTO accessTokenHashDTO = null;
        try {
            accessTokenHashDTO =
                    cacheFactory.getExpireHashService().getHash(RedisConstant.TOKEN_ACCESS_TOKEN_INFO + bean.getToken(),
                            AccessTokenHashDTO.class);
        } catch (CacheException e) {
            LOGGER.error("checkToken 检查token失败 :{}", e);
        }
        return accessTokenHashDTO;
    }

    /**
     * Description: url鉴权
     * <br />
     * CreateDate 2021-11-05 15:34:02
     *
     * @return true 鉴权通过，false 鉴权失败
     * @author yuyi
     **/
    public boolean checkUrlAccess(AccessVerifyReq bean, AccessTokenHashDTO token) {
        String url = bean.getUrl();
        String appId = bean.getAppId();
        String userId = token.getUserId();
        String tenantNo = token.getTenantNo();

        try {
            //匹配url是不是在用户对应的url中
            boolean isPass = cacheFactory.getSetService()
                    .containsValue(RedisConstant.USER_URL_LIST + tenantNo + ":" + userId + "_" + appId, url);
            if (isPass) {
                //匹配上则返回通过，否则返回失败
                return true;
            }
        } catch (CacheException e) {
            LOGGER.error(
                    "checkUrlAccess fail url:" + url + " , userId:" + userId + " , appId:" + appId + " , tenantNo" + ":"
                            + tenantNo + " ", e);
            return false;
        }
        LOGGER.info("checkUrlAccess url:{} , userId:{} , appId:{} , tenantNo:{} , 未授权 ", url, userId, appId, tenantNo);
        return false;
    }
}
