package com.cusc.nirvana.user.eiam.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.cache.CacheFactory;
import com.cache.exception.CacheException;
import com.cusc.nirvana.common.result.Response;
import com.cusc.nirvana.user.auth.common.constants.AppConfigConstant;
import com.cusc.nirvana.user.eiam.constants.*;
import com.cusc.nirvana.user.eiam.dao.UrlDao;
import com.cusc.nirvana.user.eiam.dao.entity.UrlPO;
import com.cusc.nirvana.user.eiam.dto.EiamUrlDTO;
import com.cusc.nirvana.user.eiam.dto.ResourceUrlSimpleDTO;
import com.cusc.nirvana.user.eiam.dto.UserDTO;
import com.cusc.nirvana.user.eiam.dto.UserRoleDTO;
import com.cusc.nirvana.user.eiam.service.IUrlService;
import com.cusc.nirvana.user.eiam.service.IUserService;
import com.cusc.nirvana.user.exception.CuscUserException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * <p>
 * 服务实现类
 * </p>
 *
 * @author auto-generator
 * @since 2021-10-20
 */
@Service
@Slf4j
public class UrlServiceImpl extends ServiceImpl<UrlDao, UrlPO> implements IUrlService {

    @Autowired
    private CacheFactory cacheFactory;

    @Autowired
    @Lazy
    private IUserService userService;

    @Override
    public Response<List<ResourceUrlSimpleDTO>> queryAll(Integer isAuth) {
        QueryWrapper queryWrapper = new QueryWrapper();
        queryWrapper.eq("is_delete", 0);
        queryWrapper.eq("is_auth", isAuth);

        List<UrlPO> recordList = this.list(queryWrapper);
        return Response.createSuccess(getResourceUrlSimple(recordList));
    }

    @Override
    @Transactional
    public Response addList(List<EiamUrlDTO> resourceUrlList) {
        return Response.createSuccess(baseMapper.addResourceUrlBatch(resourceUrlList));
    }

    @Override
    public Long queryByUrl(EiamUrlDTO bean) {
        QueryWrapper queryWrapper = new QueryWrapper();
        queryWrapper.eq("is_delete", 0);
        queryWrapper.eq("url", bean.getUrl());
        queryWrapper.eq("application_id", bean.getApplicationId());

        UrlPO resourceUrl = this.getOne(queryWrapper);
        if (resourceUrl != null) {
            return resourceUrl.getId();
        }
        return null;
    }

    @Override
    public Integer whiteListToRedis() {
        QueryWrapper queryWrapper = new QueryWrapper();
        queryWrapper.eq("is_delete", CommonDeleteEnum.NORMAL);
        queryWrapper.eq("is_auth", EiamConstant.URL_IS_AUTH_NO);

        List<UrlPO> urlList = this.list(queryWrapper);
        Set<String> whiteList = new HashSet<>();
        if (CollectionUtils.isEmpty(urlList)) {
            urlList = new ArrayList<>();
        }
        for (UrlPO url : urlList) {
            whiteList.add(url.getServerName() + url.getUrl());
        }
        try {
            cacheFactory.getSetService().addSet(RedisConstant.URL_WHITE_LIST, whiteList, true);
        } catch (CacheException e) {
            throw new CuscUserException(ResponseCode.SYS_BUSY.getCode() + "", ResponseCode.SYS_BUSY.getMsg());
        }
        return urlList.size();
    }

    @Override
    @Transactional
    public Long add(EiamUrlDTO bean) {
        //检查url是否重复
        Long urlId = queryByUrl(bean);
        if (urlId != null && urlId > 0) {
            return urlId;
        }
        UrlPO urlDO = new UrlPO();
        BeanUtils.copyProperties(bean, urlDO);
        this.save(urlDO);
        return urlDO.getId();
    }

    /**
     * Description: 将ResourceUrl对象转为ResourceUrl简单对象
     * <br />
     * CreateDate 2021-10-29 15:57:53
     *
     * @author yuyi
     **/
    private List<ResourceUrlSimpleDTO> getResourceUrlSimple(List<UrlPO> recordList) {
        List<ResourceUrlSimpleDTO> retList = new ArrayList<>();
        ResourceUrlSimpleDTO rusDto;

        for (UrlPO resource : recordList) {
            rusDto = new ResourceUrlSimpleDTO();
            BeanUtils.copyProperties(resource, rusDto);
            retList.add(rusDto);
        }
        return retList;
    }

    /**
     * Description: 将用户授权的url写入redis
     * <br />
     * CreateDate 2022-01-20 17:13:48
     *
     * @author yuyi
     **/
    @Override
    @Async("dataToRedisExecutor")
    public void userRelUrlToRedis(String userId, String tenantNo, String appId) {
        //查询用户，判断是不是租户管理员
        UserDTO user = new UserDTO();
        user.setUuid(userId);
        user.setTenantNo(tenantNo);
        user = userService.getByUuid(user);
        if (user == null) {
            log.warn("userRelUrlToRedis：用户信息无效。参数：userId:{}, tenantNo:{}, appId:{}", userId, tenantNo, appId);
            return;
        }
        Set<String> urlSet = null;
        if (user.getIsTenantAdmin() == CommonYesOrNoEnum.YES.getCode()) {
            //是租户管理员：查询租户的角色url
            urlSet = baseMapper.queryUrlByTenantNo(tenantNo, appId);
        } else {
            //不是租户管理员：查询用户的角色url
            urlSet = baseMapper.queryUrlByUserId(userId, tenantNo, appId);
        }
        putUrlRedis(userId, tenantNo, appId, urlSet);
    }

    @Override
    @Async("dataToRedisExecutor")
    public void userRelRolUrlToRedis(String userId, String tenantNo, String appId) {
        //查询用户，判断是不是租户管理员
        UserDTO user = new UserDTO();
        user.setUuid(userId);
        user.setTenantNo(tenantNo);
        user = userService.getByUuid(user);
        if (user == null) {
            log.warn("userRelUrlToRedis：用户信息无效。参数：userId:{}, tenantNo:{}, appId:{}", userId, tenantNo, appId);
            return;
        }
        Set<String> urlSet = null;
        if (user.getIsTenantAdmin() == CommonYesOrNoEnum.YES.getCode()) {
            //是租户管理员：查询租户的角色url
            urlSet = baseMapper.queryUrlByTenantNo(tenantNo, appId);
        } else {
            //不是租户管理员：查询用户的角色url
            urlSet = baseMapper.queryRoleUrlByUserId(userId, tenantNo, appId);
        }
        putUrlRedis(userId, tenantNo, appId, urlSet);
    }

    /**
     * Description: 将用户授权应用的所有url写入redis
     * <br />
     * CreateDate 2022-01-20 17:13:48
     *
     * @author yuyi
     **/
    @Override
    @Async("dataToRedisExecutor")
    public void userRelAppUrlToRedis(String userId, String tenantNo) {
        for (String appId : AppConfigConstant.APP_LIST) {
            userRelUrlToRedis(userId, tenantNo, appId);
        }
    }

    /**
     * Description: 删除用户对应的
     * <br />
     * CreateDate 2022-01-20 17:13:48
     *
     * @author yuyi
     **/
    @Override
    public void delUserRelUrlRedis(String userId, String tenantNo, String appId) {
        Set<String> appIdSet = new HashSet<>();
        appIdSet.addAll(AppConfigConstant.APP_LIST);
        for (String appIdStr : appIdSet) {
            delUrlRedis(userId, tenantNo, appIdStr);
        }
    }

    /**
     * Description: 将角色对应的所有用户url写入redis
     * <br />
     * CreateDate 2022-01-20 17:13:48
     *
     * @author yuyi
     **/
    @Override
    @Async("dataToRedisExecutor")
    public void roleRelUrlToRedis(String roleId, Integer roleScene, String tenantNo, String appId) {
        //判断角色类型
        if (RoleSceneEnum.USER.getCode() == roleScene) {
            //用户角色：查询角色对应的所有用户
            UserRoleDTO userRole = new UserRoleDTO();
            userRole.setRoleId(roleId);
            userRole.setTenantNo(tenantNo);
            userRole.setIsDelete(CommonDeleteEnum.NORMAL.getCode());
            List<UserDTO> userList = userService.queryUserByRoleId(userRole);
            if (!CollectionUtils.isEmpty(userList)) {
                for (UserDTO user : userList) {
                    userRelUrlToRedis(user.getUuid(), tenantNo, appId);
                }
            }
        }
    }

    /**
     * Description: 将用户集合授权的url写入redis
     * <br />
     * CreateDate 2022-01-20 17:13:48
     *
     * @author yuyi
     **/
    @Override
    @Async("dataToRedisExecutor")
    public void userListRelUrlToRedis(List<String> userIdList, String tenantNo, String appId) {
        if (CollectionUtils.isEmpty(userIdList)) {
            log.warn("userListRelUrlToRedis userIdList is empty. ");
        }
        for (String userId : userIdList) {
            userRelUrlToRedis(userId, tenantNo, appId);
        }
    }

    @Override
    @Async("dataToRedisExecutor")
    public void tenantRelUrlToRedis(String tenantNo, String appId) {
        //通过租户查询出租户管理员
        UserDTO user = new UserDTO();
        user.setTenantNo(tenantNo);
        user.setIsTenantAdmin(CommonYesOrNoEnum.YES.getCode());
        user = userService.getUser(user);
        if (user == null) {
            log.warn("tenantRelUrlToRedis：未找到有效的租户管理员账号。tenantNo:{}, appId:{}", tenantNo, appId);
            return;
        }
        Set<String> urlSet = baseMapper.queryUrlByTenantNo(tenantNo, appId);
        putUrlRedis(user.getUuid(), tenantNo, appId, urlSet);
    }

    //------------私有方法区----------------------
    private void putUrlRedis(String userId, String tenantNo, String appId, Set<String> urlSet) {
        String redisKey = RedisConstant.USER_URL_LIST + tenantNo + ":" + userId + "_" + appId;
        if (CollectionUtils.isEmpty(urlSet)) {
            log.warn("putUrlRedis urlSet is empty. key:" + redisKey);
            return;
        }
        try {
            //将角色对应的url集合放到redis
            cacheFactory.getSetService()
                    .addSet(redisKey, urlSet,
                            true);
        } catch (CacheException e) {
            log.warn("putUrlRedis fail key: " + redisKey + " ", e);
        }
    }

    private void delUrlRedis(String userId, String tenantNo, String appId) {
        String redisKey = RedisConstant.USER_URL_LIST + tenantNo + ":" + userId + "_" + appId;
        try {
            //删除redis中角色对应的url集合
            cacheFactory.getSetService().delete(redisKey);
        } catch (CacheException e) {
            log.warn("delUrlRedis fail key: " + redisKey + " ", e);
        }
    }
}
