package com.cusc.nirvana.user.eiam.service.impl;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.cusc.nirvana.common.result.PageResult;
import com.cusc.nirvana.common.result.Response;
import com.cusc.nirvana.rds.mybatis.PageHelper;
import com.cusc.nirvana.user.eiam.constants.CommonDeleteEnum;
import com.cusc.nirvana.user.eiam.constants.CommonStatusEnum;
import com.cusc.nirvana.user.eiam.constants.ResourceAuthTypeEnum;
import com.cusc.nirvana.user.eiam.constants.ResponseCode;
import com.cusc.nirvana.user.eiam.converter.ResourceConverter;
import com.cusc.nirvana.user.eiam.dao.ResourceDao;
import com.cusc.nirvana.user.eiam.dao.entity.ResourcePO;
import com.cusc.nirvana.user.eiam.dto.*;
import com.cusc.nirvana.user.eiam.service.IResourceService;
import com.cusc.nirvana.user.exception.CuscUserException;
import com.cusc.nirvana.user.util.CuscStringUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * <p>
 * 资源service
 * </p>
 *
 * @author yuy336
 * @since 2021-10-20
 */
@Service
@Slf4j
public class ResourceServiceImpl extends ServiceImpl<ResourceDao, ResourcePO> implements IResourceService {

    private final static String RESOURCE_SEPARATOR = "-";

    @Override
    public List<ResourceDTO> queryResourceByRoleId(RoleResourceDTO entity) {
        List<ResourcePO> resourceList = baseMapper.queryResourceByRoleId(entity);
        List<ResourceDTO> retList = new ArrayList<>();
        if (CollectionUtils.isEmpty(resourceList)) {
            return retList;
        }
        ResourceDTO resDTO;
        for (ResourcePO resource : resourceList) {
            resDTO = new ResourceDTO();
            BeanUtils.copyProperties(resource, resDTO);
            retList.add(resDTO);
        }
        return retList;
    }

    @Override
    public Response<List<ResourceSimpleDTO>> query(ResourceDTO entity) {
        List<ResourcePO> recordList = queryResourceList(entity);
        return Response.createSuccess(getResourceSimple(recordList));
    }

    private List<ResourcePO> queryResourceList(ResourceDTO entity) {
        QueryWrapper queryWrapper = new QueryWrapper();
        queryWrapper.eq("is_delete", CommonDeleteEnum.NORMAL);
        queryWrapper.orderByDesc("create_time");
        queryWrapper.eq(CuscStringUtils.isNotEmpty(entity.getUuid()), "uuid", entity.getUuid());
        queryWrapper.eq(CuscStringUtils.isNotEmpty(entity.getApplicationId()), "application_id", entity.getApplicationId());
        queryWrapper.like(CuscStringUtils.isNotEmpty(entity.getResourceName()), "resource_name",
                entity.getResourceName());
        queryWrapper.eq(CuscStringUtils.isNotEmpty(entity.getParentId()), "parent_id", entity.getParentId());
        queryWrapper.eq(entity.getResourceType() != null, "resource_type", entity.getResourceType());

        return this.list(queryWrapper);
    }

    @Override
    @Transactional
    public Response<Integer> addList(ResourceBatchDTO resourceBatch) {

        List<ResourceTreeDTO> resouceList = resourceBatch.getResourceList();
        if (CollectionUtils.isEmpty(resouceList)) {
            return Response.createSuccess(0);
        }
        //查询应用下的所有资源code
        Set<String> resCodeSetDb = baseMapper.queryResourceCodeSet(resourceBatch.getApplicationId());
        Set<String> resCodeSetInput = new HashSet<>();

        //树形解析
        List<ResourcePO> addList = new ArrayList<>();
        getAllResourceNode(resouceList, addList, "0", resourceBatch.getApplicationId(), resourceBatch.getCreator(),
                resCodeSetInput, resCodeSetDb);
        log.info("ResourceServiceImpl.addList addResourceBatch 的数据：{}", JSON.toJSONString(addList));
        baseMapper.addResourceBatch(addList);

        //比对出需要删除的资源
        List<String> delList = getDelResource(resCodeSetDb, resCodeSetInput);

        //删除无效的资源
        log.info("ResourceServiceImpl.addList deleteByCode 的数据：{}", JSON.toJSONString(delList));
        deleteByCode(delList, resourceBatch.getApplicationId());

        return Response.createSuccess(addList.size());
    }

    @Override
    public List<ResourceSimpleDTO> queryResourceByUserId(UserRoleDTO entity) {
        List<ResourcePO> resourceList;
        if (entity.getAuthType() != null && ResourceAuthTypeEnum.USER.getCode() == entity.getAuthType()) {
            resourceList =  baseMapper.queryResourceUserByUserId(entity);
        } else {
            resourceList = baseMapper.queryResourceRoleByUserId(entity);
        }
        return getResourceSimple(resourceList);
    }

    @Override
    public ResourcePO getResourceDOByCode(ResourceDTO entity) {
        QueryWrapper queryWrapper = new QueryWrapper();
        queryWrapper.eq("resource_code", entity.getResourceCode());
        queryWrapper.eq("application_id", entity.getApplicationId());
        queryWrapper.eq("is_delete", CommonDeleteEnum.NORMAL);
        return this.getOne(queryWrapper);
    }

    /**
     * Description:根据资源编码和应用id删除资源信息
     * <br />
     * CreateDate 2021-10-26 17:40:39
     *
     * @author yuyi
     **/
    @Override
    @Transactional
    public Integer deleteByCode(List<String> codeList, String appId) {
        if (CollectionUtils.isEmpty(codeList)) {
            return 0;
        }
        UpdateWrapper updateWrapper = new UpdateWrapper();
        updateWrapper.eq("application_id", appId);
        updateWrapper.eq("is_delete", CommonDeleteEnum.NORMAL.getCode());
        updateWrapper.in("resource_code", codeList);
        ResourcePO rrDO = new ResourcePO();
        rrDO.setIsDelete(CommonDeleteEnum.DELETED.getCode());
        this.update(rrDO, updateWrapper);
        return codeList.size();
    }

    @Override
    public void compareResource(List<ResourceTreeDTO> resourceList) {

        List<ResourcePO> dbList = queryResourceList(new ResourceDTO());
        Set<String> resCodeSetDb = baseMapper.queryResourceCodeSet("1");
        Set<String> resCodeSetInput = new HashSet<>();
        //树形解析
        List<ResourcePO> inputList = new ArrayList<>();
        getAllResourceNode(resourceList, inputList, "0", "1", "creater",
                resCodeSetInput, resCodeSetDb);

        log.info("inputList :" + JSON.toJSONString(inputList));
        for (ResourcePO res : inputList) {
            boolean isNotExists = true;
            for (ResourcePO resTmp : dbList) {
                if (res.getResourceCode().equals(resTmp.getResourceCode())) {
                    isNotExists = false;
                }
            }
            if (isNotExists) {
                log.info("compareResource:" + JSON.toJSONString(res));
            }
        }
    }

    @Override
    public PageResult<ResourceDTO> queryByPage(ResourceDTO bean) {
        QueryWrapper queryWrapper = new QueryWrapper();
        queryWrapper.orderByDesc("create_time");
        queryWrapper.eq("is_delete", CommonDeleteEnum.NORMAL.getCode());
        queryWrapper.like(CuscStringUtils.isNotEmpty(bean.getResourceCode()), "resource_code", bean.getResourceCode());
        queryWrapper.like(CuscStringUtils.isNotEmpty(bean.getResourceName()), "resource_name", bean.getResourceName());
        queryWrapper.eq(CuscStringUtils.isNotEmpty(bean.getApplicationId()), "application_id", bean.getApplicationId());
        queryWrapper.eq(bean.getParentId() != null, "parent_id", bean.getParentId());
        queryWrapper.eq(bean.getStatus() != null, "status", bean.getStatus());

        Page<ResourcePO> page =
                this.page(new Page<>(bean.getCurrPage(), bean.getPageSize()), queryWrapper);
        return PageHelper.convert(page, ResourceDTO.class);
    }

    @Override
    public ResourceDTO getByUuid(ResourceDTO bean) {
        ResourcePO resourcePO = this.getPoByUuid(bean.getUuid());
        return ResourceConverter.INSTANCE.poToDto(resourcePO);
    }

    @Override
    @Transactional
    public ResourceDTO add(ResourceDTO resource) {
        ResourcePO resourcePO = ResourceConverter.INSTANCE.dtoToPo(resource);
        resourcePO.setUuid(CuscStringUtils.generateUuid());
        this.save(resourcePO);
        resource.setUuid(resourcePO.getUuid());
        return resource;
    }

    @Override
    @Transactional
    public ResourceDTO update(ResourceDTO bean) {
        ResourcePO resourcePO = this.getPoByUuid(bean.getUuid());
        if(resourcePO == null){
            return null;
        }
        ResourcePO tmpBean = ResourceConverter.INSTANCE.dtoToPo(bean);
        tmpBean.setId(resourcePO.getId());
        this.updateById(tmpBean);
        return bean;
    }

    @Override
    @Transactional
    public boolean deleteById(ResourceDTO resource) {
        ResourcePO resourcePO = this.getPoByUuid(resource.getUuid());
        if (resourcePO == null) {
            throw new CuscUserException(ResponseCode.RESOURCE_INVALID.getCode(),
                    ResponseCode.RESOURCE_INVALID.getMsg());
        }
        ResourcePO tmpApp = new ResourcePO();
        tmpApp.setId(resourcePO.getId());
        tmpApp.setIsDelete(CommonDeleteEnum.DELETED.getCode());
        return this.updateById(tmpApp);
    }

    @Override
    @Transactional
    public boolean frozen(ResourceDTO resource) {
        ResourcePO resourcePO = this.getPoByUuid(resource.getUuid());
        if (resourcePO == null) {
            throw new CuscUserException(ResponseCode.RESOURCE_INVALID.getCode(),
                    ResponseCode.RESOURCE_INVALID.getMsg());
        }
        ResourcePO tmpApp = new ResourcePO();
        tmpApp.setId(resourcePO.getId());
        tmpApp.setStatus(CommonStatusEnum.DISABLE.getCode());
        return this.updateById(tmpApp);
    }

    @Override
    @Transactional
    public boolean unfreeze(ResourceDTO resource) {
        ResourcePO resourcePO = this.getPoByUuid(resource.getUuid());
        if (resourcePO == null) {
            throw new CuscUserException(ResponseCode.RESOURCE_INVALID.getCode(),
                    ResponseCode.RESOURCE_INVALID.getMsg());
        }
        ResourcePO tmpApp = new ResourcePO();
        tmpApp.setId(resourcePO.getId());
        tmpApp.setStatus(CommonStatusEnum.ENABLE.getCode());
        return this.updateById(tmpApp);
    }

    /**
     * 生成资源sql
     *
     * @param bean 实体
     * @return sql集合
     */
    @Override
    public Response<List<String>> generateResSql(ResourceDTO bean) {
        ResourceDTO paramRes = new ResourceDTO();
        List<String> ret = new ArrayList<>();
        paramRes.setApplicationId(bean.getApplicationId());
        if (bean.isAllChild()) {
            ResourcePO tmpRes = this.getPoByUuid(bean.getUuid());
            if (tmpRes == null) {
                return Response.createSuccess(ret);
            }
            paramRes.setTreeCode(tmpRes.getTreeCode() + RESOURCE_SEPARATOR);
        } else {
            paramRes.setUuid(bean.getUuid());
        }
        ret = baseMapper.generateResSql(paramRes);
        return Response.createSuccess(ret);
    }

    //----------------私有方法区--------------------------------

    /**
     * Description: 将Resource对象转为Resource简单对象
     * <br />
     * CreateDate 2021-10-29 15:57:53
     *
     * @author yuyi
     **/
    private List<ResourceSimpleDTO> getResourceSimple(List<ResourcePO> recordList) {
        List<ResourceSimpleDTO> retList = new ArrayList<>();
        ResourceSimpleDTO rsDto;

        for (ResourcePO resource : recordList) {
            rsDto = new ResourceSimpleDTO();
            BeanUtils.copyProperties(resource, rsDto);
            retList.add(rsDto);
        }
        return retList;
    }

    /**
     * 通过UUID查询单条数据
     *
     * @param uuid
     * @return 实例对象
     */
    private ResourcePO getPoByUuid(String uuid) {
        QueryWrapper queryWrapper = new QueryWrapper();
        queryWrapper.eq("uuid", uuid);
        queryWrapper.eq("is_delete", 0);
        return this.getOne(queryWrapper);
    }


    /**
     * Description: 获取需要删除的资源
     * <br />
     * CreateDate 2021-12-02 11:02:13
     *
     * @author yuyi
     **/
    private List<String> getDelResource(Set<String> resCodeSetDb, Set<String> resCodeSetInput) {
        List<String> ret = new ArrayList<>();
        for (String resCode : resCodeSetDb) {
            if (!resCodeSetInput.contains(resCode)) {
                ret.add(resCode);
            }
        }
        return ret;
    }

    /**
     * Description: 将输入的属性转为List数据
     * <br />
     * CreateDate 2021-12-02 11:01:18
     *
     * @author yuyi
     **/
    private void getAllResourceNode(List<ResourceTreeDTO> children, List<ResourcePO> recordList, String parentUuid,
                                    String appId, String creator, Set<String> resCodeSetInput,
                                    Set<String> resCodeSetDb) {
        ResourcePO resource;
        ResourcePO temp;
        ResourceDTO resDTO;
        for (ResourceTreeDTO treeNode : children) {
            temp = null;
            if (CuscStringUtils.isEmpty(treeNode.getResourceCode())) {
                continue;
            }
            resCodeSetInput.add(treeNode.getResourceCode());
            //构建数据库资源
            resource = new ResourcePO();
            BeanUtils.copyProperties(treeNode, resource);
            resource.setParentId(parentUuid);
            resource.setApplicationId(appId);

            //判断数据库是否存在,存在则查询库中的数据
            if (resCodeSetDb.contains(treeNode.getResourceCode())) {
                resDTO = new ResourceDTO();
                resDTO.setApplicationId(appId);
                resDTO.setResourceCode(treeNode.getResourceCode());
                //通过code查询资源信息
                temp = getResourceDOByCode(resDTO);
            }

            if (temp != null && CuscStringUtils.isNotEmpty(temp.getUuid())) {
                resource.setUuid(temp.getUuid());
                resource.setCreator(temp.getCreator());
            } else {
                resource.setUuid(CuscStringUtils.generateUuid());
                resource.setCreator(creator);
            }

            recordList.add(resource);
            if (!CollectionUtils.isEmpty(treeNode.getChildren())) {
                getAllResourceNode(treeNode.getChildren(), recordList, resource.getUuid(), appId, creator,
                        resCodeSetInput,
                        resCodeSetDb);
            }
        }
    }
}
