package com.cusc.nirvana.user.eiam.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.cusc.nirvana.common.result.PageResult;
import com.cusc.nirvana.rds.mybatis.PageHelper;
import com.cusc.nirvana.user.eiam.constants.CommonDeleteEnum;
import com.cusc.nirvana.user.eiam.constants.CommonStatusEnum;
import com.cusc.nirvana.user.eiam.constants.EiamConstant;
import com.cusc.nirvana.user.eiam.constants.ResponseCode;
import com.cusc.nirvana.user.eiam.converter.OrganizationConverter;
import com.cusc.nirvana.user.eiam.dao.OrganizationDao;
import com.cusc.nirvana.user.eiam.dao.entity.OrganizationPO;
import com.cusc.nirvana.user.eiam.dto.OrganizationDTO;
import com.cusc.nirvana.user.eiam.dto.UserDTO;
import com.cusc.nirvana.user.eiam.dto.UserOrganDTO;
import com.cusc.nirvana.user.eiam.service.IOrganizationService;
import com.cusc.nirvana.user.eiam.service.IUserOrganService;
import com.cusc.nirvana.user.eiam.service.IUserService;
import com.cusc.nirvana.user.exception.CuscUserException;
import com.cusc.nirvana.user.util.CuscStringUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 组织机构(Organization)表服务实现类
 *
 * @author yuy336
 * @since 2022-01-12 14:59:30
 */
@Service
public class OrganizationServiceImpl extends ServiceImpl<OrganizationDao, OrganizationPO>
        implements IOrganizationService {


	@Resource
	private IUserOrganService userOrganService;

	@Resource
	private IUserService userService;

    /**
     * 通过UUID查询单条数据
     *
     * @param organization
     * @return 实例对象
     */
    @Override
    public OrganizationDTO getByUuid(OrganizationDTO organization) {
        OrganizationPO organizationPO = this.getPoByUuid(organization.getUuid(), organization.getTenantNo());
        return OrganizationConverter.INSTANCE.poToDto(organizationPO);
    }

    /**
     * 通过查询条件查询集合数据
     *
     * @param organization
     * @return 集合对象
     */
    @Override
    public List<OrganizationDTO> queryByList(OrganizationDTO organization) {
        QueryWrapper queryWrapper = new QueryWrapper();
        queryWrapper.eq("is_delete", CommonDeleteEnum.NORMAL.getCode());
        queryWrapper.orderByDesc("create_time");
        queryWrapper.eq(CuscStringUtils.isNotEmpty(organization.getTenantNo()),"tenant_no", organization.getTenantNo());

        queryWrapper.like(CuscStringUtils.isNotEmpty(organization.getOrganCode()), "organ_code",
                organization.getOrganCode());
        queryWrapper.like(CuscStringUtils.isNotEmpty(organization.getOrganName()), "organ_name",
                organization.getOrganName());
        queryWrapper.eq(CuscStringUtils.isNotEmpty(organization.getParentId()), "parent_id",
                organization.getParentId());
        queryWrapper.eq(organization.getOrganType() != null, "organ_type", organization.getOrganType());
        queryWrapper.eq(organization.getStatus() != null, "status", organization.getStatus());
        queryWrapper.eq(organization.getBizType() != null, "biz_type", organization.getBizType());
        queryWrapper.likeRight(CuscStringUtils.isNotEmpty(organization.getQueryCode()), "query_code",
                organization.getQueryCode());

        if(!CollectionUtils.isEmpty(organization.getUuidList())){
            queryWrapper.in("uuid", organization.getUuidList());
        }

        List<OrganizationPO> record = this.list(queryWrapper);
        return OrganizationConverter.INSTANCE.poListToDtoList(record);
    }

    /**
     * 分页查询
     *
     * @param organization 筛选条件
     * @return 查询结果
     */
    @Override
    public PageResult<OrganizationDTO> queryByPage(OrganizationDTO organization) {
        QueryWrapper queryWrapper = new QueryWrapper();
        queryWrapper.orderByDesc("create_time");
        queryWrapper.eq("is_delete", CommonDeleteEnum.NORMAL.getCode());
        if (StringUtils.isNotBlank(organization.getTenantNo())) {
			queryWrapper.eq("tenant_no", organization.getTenantNo());
		}

        queryWrapper.like(CuscStringUtils.isNotEmpty(organization.getOrganCode()), "organ_code",
                organization.getOrganCode());
        queryWrapper.like(CuscStringUtils.isNotEmpty(organization.getOrganName()), "organ_name",
                organization.getOrganName());
        queryWrapper.like(CuscStringUtils.isNotEmpty(organization.getOrganSimpleName()),"organ_simple_name",organization.getOrganSimpleName());
        queryWrapper.eq(CuscStringUtils.isNotEmpty(organization.getParentId()), "parent_id",
                organization.getParentId());
        queryWrapper.eq(organization.getOrganType() != null, "organ_type", organization.getOrganType());
        queryWrapper.eq(organization.getStatus() != null, "status", organization.getStatus());
        queryWrapper.eq(organization.getBizType() != null, "biz_type", organization.getBizType());
        queryWrapper.eq(CuscStringUtils.isNotEmpty(organization.getUniqueCode()),"unique_code",organization.getUniqueCode());
        queryWrapper.likeRight(CuscStringUtils.isNotEmpty(organization.getQueryCode()), "query_code",
                organization.getQueryCode());

        Page<OrganizationPO> page =
                this.page(new Page<>(organization.getCurrPage(), organization.getPageSize()), queryWrapper);
		PageResult<OrganizationDTO> result = PageHelper.convert(page, OrganizationDTO.class);

		List<OrganizationPO> poList = page.getRecords();
		//查询到组织对应的管理员信息
		Map<String, UserDTO> adminMap = queryOrganAdmin(poList,null == organization.getParentId() || EiamConstant.DEFAULT_ORGAN_POSTION.equals(organization.getParentId()), organization.getType());

		List<OrganizationDTO> dtoList = result.getList();

		dtoList.forEach(dto->{
			UserDTO userDTO = adminMap.get(dto.getUuid());
			if (null != userDTO) {
				dto.setAdminUserId(userDTO.getUuid());
				dto.setAdminAccount(userDTO.getUserName());
				dto.setAdminName(userDTO.getFullName());
				dto.setAdminPhone(userDTO.getPhone());
                dto.setCreateTime(userDTO.getCreateTime());
			}
		});

		return result;
	}

	/**
	 *
	 * @param poList
	 * @param tenantAdmin 是否租户管理员
	 * @return map key--->orgId value-->userInfo
	 */
	private Map<String, UserDTO> queryOrganAdmin(List<OrganizationPO> poList, boolean tenantAdmin, String type) {
		if (CollectionUtils.isEmpty(poList)) {
			return Collections.emptyMap();
		}

		Map<String, OrganizationPO> organMap = poList.stream().collect(Collectors.toMap(OrganizationPO::getUuid, Function.identity(), (k, v) -> v));

		List<String> orgUuidList = poList.stream().map(OrganizationPO::getUuid).distinct().collect(Collectors.toList());
		UserOrganDTO userOrgDto = new UserOrganDTO();
		userOrgDto.setOrganIdList(orgUuidList);
		List<UserOrganDTO> userOrganDTOS = userOrganService.queryByList(userOrgDto);
		Map<String, String> userOrganMap = userOrganDTOS.stream().collect(Collectors.toMap(UserOrganDTO::getUserId, UserOrganDTO::getOrganId, (k, v) -> v));

		//用户id
		List<String> userIdList = userOrganDTOS.stream().map(UserOrganDTO::getUserId)
				.distinct().collect(Collectors.toList());

		//查询用户信息
		List<UserDTO> userDTOS;
        if(StringUtils.isNotEmpty(type)&&"1".equals(type)){
            userDTOS = userService.queryAdminUserListByUserIdListNew(userIdList,tenantAdmin);
        }else {
            userDTOS = userService.queryAdminUserListByUserIdList(userIdList,tenantAdmin);
        }

		Map<String, UserDTO> userMap = userDTOS.stream().collect(Collectors.toMap(UserDTO::getUuid, Function.identity(), (k, v) -> v));


		Map<String, UserDTO> resultMap = new HashMap<>(poList.size());
		userMap.forEach((userId,userDto)->{
			String organId = userOrganMap.get(userId);
			if (StringUtils.isNotBlank(organId)) {
				resultMap.put(organId,userDto);
			}
		});
		
		return resultMap;
	}

	/**
     * 新增数据
     *
     * @param organization 实例对象
     * @return 实例对象
     */
    @Override
    @Transactional
    public OrganizationDTO insert(OrganizationDTO organization) {
        OrganizationPO organizationPO = OrganizationConverter.INSTANCE.dtoToPo(organization);
        String orgId = CuscStringUtils.generateUuid();
        organizationPO.setUuid(orgId);
        //设置查询编码
        if (EiamConstant.DEFAULT_ORGAN_POSTION.equals(organization.getParentId())) {
            organizationPO.setQueryCode(EiamConstant.DEFAULT_ORGAN_POSTION + EiamConstant.QUERY_CODE_SEPARATOR + orgId);
        } else {
            //通过父节点查询querycode
            OrganizationPO parentPO = getPoByUuid(organization.getParentId(), organization.getTenantNo());
            if (parentPO == null) {
                throw new CuscUserException(ResponseCode.ORGAN_PARENT_INVALID.getCode(),
                        ResponseCode.ORGAN_PARENT_INVALID.getMsg());
            }
            organizationPO.setQueryCode(parentPO.getQueryCode() + EiamConstant.QUERY_CODE_SEPARATOR + orgId);
        }
        this.save(organizationPO);
        organization.setUuid(organizationPO.getUuid());
        return organization;
    }

    /**
     * 修改数据
     *
     * @param organization 实例对象
     * @return 实例对象
     */
    @Override
    @Transactional
    public OrganizationDTO update(OrganizationDTO organization) {
        OrganizationPO organizationPO = this.getPoByUuid(organization.getUuid(), organization.getTenantNo());
        if (null == organizationPO) {
        	return organization;
		}
		OrganizationPO updatePo = OrganizationConverter.INSTANCE.dtoToPo(organization);
        updatePo.setId(organizationPO.getId());
        updatePo.setUuid(null);
        updatePo.setOrganCode(null);
        updatePo.setTenantNo(null);
		//判断父节点是否已修改
        if (CuscStringUtils.isNotEmpty(organization.getParentId()) && !organization.getParentId()
                .equals(organizationPO.getParentId())) {
            //修改当前组织的query code 以及所有下级的query code
            //通过父节点查询querycode
            OrganizationPO parentPO = getPoByUuid(organization.getParentId(), organization.getTenantNo());
            if (parentPO == null) {
                throw new CuscUserException(ResponseCode.ORGAN_PARENT_INVALID.getCode(),
                        ResponseCode.ORGAN_PARENT_INVALID.getMsg());
            }
            String newQueryCode =
                    parentPO.getQueryCode() + EiamConstant.QUERY_CODE_SEPARATOR + organizationPO.getUuid();
            updateChildQueryCode(organizationPO.getQueryCode(), newQueryCode, organizationPO.getTenantNo());
            organizationPO.setQueryCode(newQueryCode);
        }

		this.updateById(updatePo);
        return organization;
    }

    /**
     * 通过主键删除数据
     *
     * @param organization 实例对象
     * @return 是否成功
     */
    @Override
    @Transactional
    public boolean deleteById(OrganizationDTO organization) {
        OrganizationPO organizationPO = this.getPoByUuid(organization.getUuid(), organization.getTenantNo());
        return this.updateById(organizationPO);
    }

    @Override
    @Transactional
    public boolean frozen(OrganizationDTO bean) {
        OrganizationPO organPO = this.getPoByUuid(bean.getUuid(), bean.getTenantNo());
        if (organPO == null) {
            return false;
        }
        OrganizationPO tmpBean = new OrganizationPO();
        tmpBean.setId(organPO.getId());
        tmpBean.setStatus(CommonStatusEnum.DISABLE.getCode());
        return this.updateById(tmpBean);
    }

    @Override
    @Transactional
    public boolean unfreeze(OrganizationDTO bean) {
        OrganizationPO organPO = this.getPoByUuid(bean.getUuid(), bean.getTenantNo());
        if (organPO == null) {
            return false;
        }
        OrganizationPO tmpBean = new OrganizationPO();
        tmpBean.setId(organPO.getId());
        tmpBean.setStatus(CommonStatusEnum.ENABLE.getCode());
        return this.updateById(tmpBean);
    }

    @Override
    public List<OrganizationDTO> getOrganListByUserId(UserOrganDTO bean) {
        return OrganizationConverter.INSTANCE.poListToDtoList(baseMapper.queryOrganListByUserId(bean));
    }

    //----------私有方法区------------------------

    /**
     * 通过UUID查询单条数据
     *
     * @param uuid
     * @return 实例对象
     */
    private OrganizationPO getPoByUuid(String uuid, String tenantNo) {
        QueryWrapper queryWrapper = new QueryWrapper();
        queryWrapper.eq("uuid", uuid);
        queryWrapper.eq("is_delete", CommonDeleteEnum.NORMAL.getCode());
        queryWrapper.eq(StringUtils.isNotBlank(tenantNo),"tenant_no", tenantNo);
        return this.getOne(queryWrapper);
    }

    /**
     * Description: 通过oldQuerycode修改所有下级的queryCode
     * <br />
     * CreateDate 2022-04-11 15:20:53
     *
     * @author yuyi
     **/
    private void updateChildQueryCode(String oldQueryCode, String newQueryCode, String tenantNo) {
        QueryWrapper queryWrapper = new QueryWrapper();
        queryWrapper.likeRight("query_code", oldQueryCode);
        queryWrapper.eq("is_delete", CommonDeleteEnum.NORMAL.getCode());
        queryWrapper.eq("tenant_no", tenantNo);
        List<OrganizationPO> orgList = this.list(queryWrapper);
        if (!CollectionUtils.isEmpty(orgList)) {
            OrganizationPO updatePO;
            for (OrganizationPO tempOrg : orgList) {
                updatePO = new OrganizationPO();
                updatePO.setId(tempOrg.getId());
                updatePO.setQueryCode(tempOrg.getQueryCode().replaceAll(oldQueryCode, newQueryCode));
                this.updateById(updatePO);
            }
        }
    }
}
