package com.cusc.nirvana.user.eiam.service.impl;

import com.cache.CacheFactory;
import com.cusc.nirvana.common.result.Response;
import com.cusc.nirvana.user.config.SmsPropertyConfig;
import com.cusc.nirvana.user.eiam.constants.ResponseCode;
import com.cusc.nirvana.user.eiam.dto.ApplicationDTO;
import com.cusc.nirvana.user.eiam.dto.SmsResponseDTO;
import com.cusc.nirvana.user.eiam.dto.SmsSendConfig;
import com.cusc.nirvana.user.eiam.dto.SmsSendDTO;
import com.cusc.nirvana.user.eiam.service.IEiamSmsService;
import com.cusc.nirvana.user.exception.CuscUserException;
import com.cusc.nirvana.user.util.CuscStringUtils;
import com.cusc.nirvana.user.util.DateUtils;
import com.cusc.nirvana.user.util.RestTemplateUtils;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;

/**
 * Description: 短信service
 * <br />
 * CreateDate 2021-11-02 20:25:49
 *
 * @author yuyi
 **/
@Service
@Slf4j
public class EiamSmsServiceImpl implements IEiamSmsService {

    private static final Logger LOGGER = LoggerFactory.getLogger(EiamSmsServiceImpl.class);

    @Autowired
    private RestTemplate restTemplate;

    @Autowired
    private SmsPropertyConfig smsPropertyConfig;

    @Autowired
    private CacheFactory cacheFactory;

    @Override
    public SmsResponseDTO sendSms(String phone, List<String> paramterList, SmsSendConfig config) {
        SmsSendDTO send = new SmsSendDTO();
        send.setAccesskey(config.getSmsPlatformKey());

        List<String> phoneList = new ArrayList<>();
        phoneList.add(phone);
        send.setPhoneNumbers(phoneList);
        send.setTemplateParams(paramterList);
        send.setSignatureCode(config.getSmsSignatureCode());
        send.setStrategyCode(config.getStrategyCode());
        send.setTemplateCode(config.getSmsTemplateCode());
        Response<SmsResponseDTO> retResp;
        try {
            retResp = RestTemplateUtils.postForResponse(restTemplate,
                    smsPropertyConfig.getSmsUrl() + smsPropertyConfig.getSendUrl(),
                    send,
                    SmsResponseDTO.class);
        } catch (Exception e) {
            LOGGER.error("短信发送失败： ", e);
            throw new CuscUserException(ResponseCode.SMS_SEND_ERROR.getCode(),
                    ResponseCode.SMS_SEND_ERROR.getMsg());
        }
        //记录短信发送次数和间隔
        saveSmsSendLimitToRedis(phone, config);
        if (retResp != null) {
            return retResp.getData();
        }
        return null;
    }

    @Override
    public SmsResponseDTO sendSms(String phone, String parameter, SmsSendConfig config) {
        List<String> list = new ArrayList<>();
        list.add(parameter);
        return sendSms(phone, list, config);
    }

    @Override
    public boolean checkSmsConfigNotNull(SmsSendConfig bean) {
        return bean != null && CuscStringUtils.isNotEmpty(bean.getSmsTemplateCode());
    }

    /**
     * Description: 短信发送限制检查
     * <br />
     * CreateDate 2022-01-27 14:43:41
     *
     * @author yuyi
     **/
    @Override
    public void checkSmsSendLimit(String phone, SmsSendConfig bean) {
        try {
            if (bean.getSmsTotalLimit() != null && bean.getSmsTotalLimit() > 0 && CuscStringUtils.isNotEmpty(
                    bean.getTotalLimitKey())) {
                //记录发送总次数限制
                Integer smsTotal =
                        cacheFactory.getExpireStringService().getValue(bean.getTotalLimitKey() + phone + "_" + bean.getTenantNo()+ "_" + bean.getAppId(),
                                Integer.class);
                if (smsTotal != null && smsTotal >= bean.getSmsTotalLimit()) {
                    throw new CuscUserException(ResponseCode.SMS_TOTAL_LIMIT_OVERRUN.getCode(),
                            ResponseCode.SMS_TOTAL_LIMIT_OVERRUN.getMsg());
                }
            }

            if (bean.getSmsIntervalLimit() != null && bean.getSmsIntervalLimit() > 0 && CuscStringUtils.isNotEmpty(
                    bean.getIntervalLimitKey())) {
                //记录发送间隔限制
                boolean isExists =
                        cacheFactory.getExpireStringService()
                                .containsKey(bean.getIntervalLimitKey() + phone + "_" + bean.getTenantNo()+ "_" + bean.getAppId());
                if (isExists) {
                    throw new CuscUserException(ResponseCode.SMS_INTERVAL_LIMIT_OVERRUN.getCode(),
                            ResponseCode.SMS_INTERVAL_LIMIT_OVERRUN.getMsg());
                }
            }
        } catch (Exception e) {
            //只记录，不抛出异常，屏蔽对业务的影响
            log.error("检查短信发送限制信息时访问redis 异常：", e);
        }
    }

    @Override
    public void convertToSmsConfig(ApplicationDTO fromBean, SmsSendConfig toBean) {
        //短信配置为空，从应用配置中取
        if (!checkSmsConfigNotNull(toBean)) {
            throw new CuscUserException(ResponseCode.SMS_CONFIG_NOT_NULL.getCode(),
                    ResponseCode.SMS_CONFIG_NOT_NULL.getMsg());
        }

        if (toBean.getSmsPlatformKey() == null) {
            if (fromBean.getSmsPlatformKey() == null) {
                log.warn("sms config smsPlatformKey is null");
                throw new CuscUserException(ResponseCode.SMS_CONFIG_NOT_NULL.getCode() + "",
                        ResponseCode.SMS_CONFIG_NOT_NULL.getMsg());
            }
            toBean.setSmsPlatformKey(fromBean.getSmsPlatformKey());
        }

        if (toBean.getSmsSignatureCode() == null) {
            if (fromBean.getSmsSignatureCode() == null) {
                log.warn("sms config smsSignatureCode is null");
                throw new CuscUserException(ResponseCode.SMS_CONFIG_NOT_NULL.getCode() + "",
                        ResponseCode.SMS_CONFIG_NOT_NULL.getMsg());
            }
            toBean.setSmsSignatureCode(fromBean.getSmsSignatureCode());
        }
    }

    //----------------私有方法区域--------------------------

    /**
     * Description: 保存短信发送限制信息到redis
     * <br />
     * CreateDate 2022-02-16 09:50:25
     *
     * @author yuyi
     **/
    private void saveSmsSendLimitToRedis(String phone, SmsSendConfig bean) {
        try {
            if (bean.getSmsTotalLimit() != null && bean.getSmsTotalLimit() > 0 && CuscStringUtils.isNotEmpty(
                    bean.getTotalLimitKey())) {
                //记录发送总次数限制
                Integer smsTotal =
                        cacheFactory.getExpireStringService().getValue(bean.getTotalLimitKey() + phone + "_" + bean.getTenantNo()+ "_" + bean.getAppId(),
                                Integer.class);
                Long expireTime;
                if (smsTotal == null) {
                    smsTotal = 1;
                    LocalDateTime begin = LocalDateTime.now();
                    expireTime = DateUtils.secondBetween(begin, DateUtils.getDayEnd(begin));
                } else {
                    smsTotal++;
                    expireTime =
                            cacheFactory.getExpireStringService()
                                    .getKeyExpireTime(bean.getTotalLimitKey() + phone + "_" + bean.getTenantNo()+ "_" + bean.getAppId());
                }
                cacheFactory.getExpireStringService()
                        .setExpireValue(bean.getTotalLimitKey() + phone + "_" + bean.getTenantNo()+ "_" + bean.getAppId(), smsTotal,
                                expireTime.intValue());
            }

            if (bean.getSmsIntervalLimit() != null && bean.getSmsIntervalLimit() > 0 && CuscStringUtils.isNotEmpty(
                    bean.getIntervalLimitKey())) {
                //记录发送间隔限制
                cacheFactory.getExpireStringService()
                        .setExpireValue(bean.getIntervalLimitKey() + phone + "_" + bean.getTenantNo()+ "_" + bean.getAppId(), 1,
                                bean.getSmsIntervalLimit());
            }
        } catch (Exception e) {
            //只记录，不抛出异常，屏蔽对业务的影响
            log.error("保存短信发送限制信息到redis 异常：", e);
        }
    }
}
