package com.cusc.nirvana.user.eiam.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.cache.CacheFactory;
import com.cache.exception.CacheException;
import com.cusc.nirvana.common.result.PageResult;
import com.cusc.nirvana.rds.mybatis.PageHelper;
import com.cusc.nirvana.user.eiam.constants.CommonStatusEnum;
import com.cusc.nirvana.user.eiam.constants.RedisConstant;
import com.cusc.nirvana.user.eiam.constants.ResponseCode;
import com.cusc.nirvana.user.eiam.converter.ApplicationConverter;
import com.cusc.nirvana.user.eiam.dao.ApplicationDao;
import com.cusc.nirvana.user.eiam.dao.entity.ApplicationPO;
import com.cusc.nirvana.user.eiam.dto.ApplicationDTO;
import com.cusc.nirvana.user.eiam.service.IApplicationService;
import com.cusc.nirvana.user.exception.CuscUserException;
import com.cusc.nirvana.user.util.CuscStringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;

/**
 * (Application)表服务实现类
 *
 * @author yuy336
 * @since 2022-01-11 17:09:07
 */
@Service
public class ApplicationServiceImpl extends ServiceImpl<ApplicationDao, ApplicationPO> implements IApplicationService {

    @Autowired
    private CacheFactory cacheFactory;

    /**
     * 通过UUID查询单条数据
     *
     * @param application
     * @return 实例对象
     */
    @Override
    public ApplicationDTO getById(ApplicationDTO application) {
        ApplicationPO record = getPoById(application.getId());
        return ApplicationConverter.INSTANCE.poToDto(record);
    }

    /**
     * 通过查询条件查询集合数据
     *
     * @param application
     * @return 集合对象
     */
    @Override
    public List<ApplicationDTO> queryByList(ApplicationDTO application) {
        QueryWrapper queryWrapper = new QueryWrapper();
        queryWrapper.eq("is_delete", 0);
        queryWrapper.orderByDesc("create_time");
        List<ApplicationPO> record = this.list(queryWrapper);
        return ApplicationConverter.INSTANCE.poListToDtoList(record);
    }

    /**
     * 分页查询
     *
     * @param application 筛选条件
     * @return 查询结果
     */
    @Override
    public PageResult<ApplicationDTO> queryByPage(ApplicationDTO application) {
        QueryWrapper queryWrapper = new QueryWrapper();
        queryWrapper.orderByDesc("create_time");
        queryWrapper.eq("is_delete", 0);

        queryWrapper.like(CuscStringUtils.isNotEmpty(application.getApplicationCode()), "application_code",
                application.getApplicationCode());

        queryWrapper.like(CuscStringUtils.isNotEmpty(application.getApplicationName()), "application_name",
                application.getApplicationName());

        Page<ApplicationPO> page =
                this.page(new Page<>(application.getCurrPage(), application.getPageSize()), queryWrapper);
        return PageHelper.convert(page, ApplicationDTO.class);
    }

    /**
     * 新增数据
     *
     * @param application 实例对象
     * @return 实例对象
     */
    @Override
    @Transactional
    public ApplicationDTO insert(ApplicationDTO application) {
        ApplicationPO applicationPO = ApplicationConverter.INSTANCE.dtoToPo(application);
        this.save(applicationPO);
        return application;
    }

    /**
     * 修改数据
     *
     * @param application 实例对象
     * @return 实例对象
     */
    @Override
    @Transactional
    public ApplicationDTO update(ApplicationDTO application) {
        ApplicationPO applicationPO = this.getPoById(application.getId());
        if (applicationPO == null) {
            return null;
        }
        ApplicationPO tmpBean = ApplicationConverter.INSTANCE.dtoToPo(application);
        this.updateById(tmpBean);
        return application;
    }

    /**
     * 通过主键删除数据
     *
     * @param application 实例对象
     * @return 是否成功
     */
    @Override
    @Transactional
    public boolean deleteById(ApplicationDTO application) {
        ApplicationPO applicationPO = this.getPoById(application.getId());
        if (applicationPO == null) {
            return false;
        }
        return this.updateById(applicationPO);
    }

    /**
     * 冻结
     *
     * @param application 实例对象
     * @return 是否成功
     */
    @Override
    @Transactional
    public boolean frozen(ApplicationDTO application) {
        ApplicationPO applicationPO = this.getPoById(application.getId());
        if (applicationPO == null) {
            return false;
        }
        ApplicationPO tmpApp = new ApplicationPO();
        tmpApp.setId(applicationPO.getId());
        tmpApp.setStatus(CommonStatusEnum.DISABLE.getCode());
        return this.updateById(tmpApp);
    }

    /**
     * 解冻
     *
     * @param application 实例对象
     * @return 是否成功
     */
    @Override
    @Transactional
    public boolean unfreeze(ApplicationDTO application) {
        ApplicationPO applicationPO = this.getPoById(application.getId());
        if (applicationPO == null) {
            return false;
        }
        ApplicationPO tmpApp = new ApplicationPO();
        tmpApp.setId(applicationPO.getId());
        tmpApp.setStatus(CommonStatusEnum.ENABLE.getCode());
        return this.updateById(tmpApp);
    }

    /**
     * 通过应用code查询缓存
     *
     * @param appCode
     * @return 实例对象
     */
    @Override
    public ApplicationDTO getCacheByCode(String appCode) {
        //去缓存获取
        try {
            ApplicationDTO appBean = cacheFactory.getExpireStringService()
                    .getValue(RedisConstant.APP_CODE_CONFIG + appCode, ApplicationDTO.class);
            if (appBean == null || CuscStringUtils.isEmpty(appBean.getApplicationCode())) {
                //查询数据库
                appBean = ApplicationConverter.INSTANCE.poToDto(getPoByCode(appCode));
                if (appBean == null) {
                    throw new CuscUserException(ResponseCode.APP_CONFIG_INVALID.getCode() + "",
                            ResponseCode.APP_CONFIG_INVALID.getMsg());
                }
                cacheFactory.getExpireStringService()
                        .setExpireValue(RedisConstant.APP_CODE_CONFIG + appCode, appBean,
                                RedisConstant.APP_CODE_CONFIG_TIME);
            }
            return appBean;
        } catch (CacheException e) {
            throw new CuscUserException(ResponseCode.REDIS_OPT_FAIL.getCode() + "",
                    ResponseCode.REDIS_OPT_FAIL.getMsg());
        }
    }

    @Override
    public ApplicationDTO refreshCacheByCode(String appCode) {
        ApplicationDTO appBean = ApplicationConverter.INSTANCE.poToDto(getPoByCode(appCode));
        if (appBean == null) {
            throw new CuscUserException(ResponseCode.APP_CONFIG_INVALID.getCode() + "",
                    ResponseCode.APP_CONFIG_INVALID.getMsg());
        }
        try {
            cacheFactory.getExpireStringService()
                    .setExpireValue(RedisConstant.APP_CODE_CONFIG + appCode, appBean,
                            RedisConstant.APP_CODE_CONFIG_TIME);
        } catch (CacheException e) {
            throw new CuscUserException(ResponseCode.REDIS_OPT_FAIL.getCode() + "",
                    ResponseCode.REDIS_OPT_FAIL.getMsg());
        }
        return appBean;
    }

    //-----------私有方法区------------------------

    /**
     * 通过UUID查询单条数据
     *
     * @param id
     * @return 实例对象
     */
    private ApplicationPO getPoById(Long id) {
        return this.getById(id);
    }

    /**
     * 通过UUID查询单条数据
     *
     * @param code
     * @return 实例对象
     */
    private ApplicationPO getPoByCode(String code) {
        QueryWrapper queryWrapper = new QueryWrapper();
        queryWrapper.eq("application_code", code);
        queryWrapper.eq("is_delete", 0);
        return this.getOne(queryWrapper);
    }
}
