package com.cusc.nirvana.user.eiam.controller;

import com.cusc.nirvana.common.result.PageResult;
import com.cusc.nirvana.common.result.Response;
import com.cusc.nirvana.user.eiam.constants.CommonDeleteEnum;
import com.cusc.nirvana.user.eiam.constants.ResponseCode;
import com.cusc.nirvana.user.eiam.dto.UserDTO;
import com.cusc.nirvana.user.eiam.dto.UserRoleDTO;
import com.cusc.nirvana.user.eiam.dto.UserSimpleDTO;
import com.cusc.nirvana.user.eiam.service.IUserService;
import com.cusc.nirvana.user.eiam.util.CommonParamterCheck;
import com.cusc.nirvana.user.util.CuscStringUtils;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;

/**
 * <p>
 * 针对内部员工、生态合作伙伴、分级线下店铺等企业内部用户 前端控制器
 * </p>
 *
 * @author yuy336
 * @since 2021-10-20
 */
@Slf4j
@RestController
@RequestMapping("/user")
public class UserController {

    @Autowired
    IUserService userService;

    @PostMapping("/add")
    @ApiOperation(value = "新增用户", notes = "新增用户")
    public Response add(@RequestBody UserDTO entity) {
        if(entity == null || CuscStringUtils.isEmpty(entity.getPhone())){
            return Response.createError(ResponseCode.USER_PHONE_NOT_NULL.getMsg(),
                    ResponseCode.USER_PHONE_NOT_NULL.getCode());
        }
        CommonParamterCheck.appIdAndTenantNoRequired(entity.getApplicationId(), entity.getTenantNo());
        return Response.createSuccess(userService.addUser(entity));
    }

    @PostMapping("/delete")
    @ApiOperation(value = "删除用户", notes = "删除用户")
    public Response delete(@RequestBody UserDTO entity) {
        if (entity == null || CuscStringUtils.isEmpty(entity.getUuid())) {
            return Response.createError(ResponseCode.USER_UUID_NOT_NULL.getMsg(),
                    ResponseCode.USER_UUID_NOT_NULL.getCode());
        }
        CommonParamterCheck.appIdAndTenantNoRequired(entity.getApplicationId(), entity.getTenantNo());
        return Response.createSuccess(userService.deleteUser(entity));
    }

    @PostMapping("/update")
    @ApiOperation(value = "更新用户信息", notes = "更新用户信息")
    public Response update(@RequestBody UserDTO entity) {
        if (entity == null || CuscStringUtils.isEmpty(entity.getUuid())) {
            return Response.createError(ResponseCode.USER_UUID_NOT_NULL.getMsg(),
                    ResponseCode.USER_UUID_NOT_NULL.getCode());
        }
        CommonParamterCheck.appIdAndTenantNoRequired(entity.getApplicationId(), entity.getTenantNo());
        return Response.createSuccess(userService.updateUser(entity));
    }

    @PostMapping("/query")
    @ApiOperation(value = "查询用户集合信息", notes = "查询用户集合信息")
    public Response<List<UserSimpleDTO>> query(@RequestBody UserDTO entity) {
        CommonParamterCheck.appIdAndTenantNoRequired(entity.getApplicationId(), entity.getTenantNo());
        return Response.createSuccess(userService.queryListUser(entity));
    }

    @PostMapping("/get")
    @ApiOperation(value = "查询单个用户信息", notes = "查询单个用户信息")
    public Response<UserDTO> get(@RequestBody UserDTO entity) {
        if (entity == null || CuscStringUtils.isEmpty(entity.getUuid())) {
            return Response.createError(ResponseCode.USER_UUID_NOT_NULL.getMsg(),
                    ResponseCode.USER_UUID_NOT_NULL.getCode());
        }
        CommonParamterCheck.appIdAndTenantNoRequired(entity.getApplicationId(), entity.getTenantNo());
        return Response.createSuccess(userService.getUser(entity));
    }

    @PostMapping("/page")
    @ApiOperation(value = "分页查询用户信息(page)", notes = "分页查询用户信息")
    public Response<PageResult<UserDTO>> page(@RequestBody UserDTO entity) {
        CommonParamterCheck.appIdAndTenantNoRequired(entity.getApplicationId(), entity.getTenantNo());
        return Response.createSuccess(userService.pageListUser(entity));
    }

    @PostMapping("/frozen")
    @ApiOperation(value = "冻结用户", notes = "冻结用户")
    public Response frozen(@RequestBody UserDTO entity) {
        if (entity == null || CuscStringUtils.isEmpty(entity.getUuid())) {
            return Response.createError(ResponseCode.USER_UUID_NOT_NULL.getMsg(),
                    ResponseCode.USER_UUID_NOT_NULL.getCode());
        }
        CommonParamterCheck.appIdAndTenantNoRequired(entity.getApplicationId(), entity.getTenantNo());
        return Response.createSuccess(userService.frozen(entity));
    }

    @PostMapping("/unfreeze")
    @ApiOperation(value = "解冻用户", notes = "解冻用户")
    public Response unfreeze(@RequestBody UserDTO entity) {
        if (entity == null || CuscStringUtils.isEmpty(entity.getUuid())) {
            return Response.createError(ResponseCode.USER_UUID_NOT_NULL.getMsg(),
                    ResponseCode.USER_UUID_NOT_NULL.getCode());
        }
        CommonParamterCheck.appIdAndTenantNoRequired(entity.getApplicationId(), entity.getTenantNo());
        return Response.createSuccess(userService.unfreeze(entity));
    }

    @PostMapping("/getByUuid")
    @ApiOperation(value = "通过id查询用户详情", notes = "通过id查询用户详情")
    public Response<UserDTO> getByUuid(@RequestBody UserDTO entity) {
        if (entity == null || CuscStringUtils.isEmpty(entity.getUuid())) {
            return Response.createError(ResponseCode.USER_UUID_NOT_NULL.getMsg(),
                    ResponseCode.USER_UUID_NOT_NULL.getCode());
        }
        CommonParamterCheck.appIdAndTenantNoRequired(entity.getApplicationId(), entity.getTenantNo());
        return Response.createSuccess(userService.getByUuid(entity));
    }

    @PostMapping("/getByUuids")
    @ApiOperation(value = "通过id集合查询用户详情", notes = "通过id集合查询用户详情")
    public Response<List<UserSimpleDTO>> getByUuids(@RequestBody UserDTO entity) {
        if (CollectionUtils.isEmpty(entity.getUuidList())) {
            return Response.createError(ResponseCode.USER_UUID_NOT_NULL.getMsg(),
                    ResponseCode.USER_UUID_NOT_NULL.getCode());
        }
        CommonParamterCheck.appIdAndTenantNoRequired(entity.getApplicationId(), entity.getTenantNo());
        return Response.createSuccess(userService.getByUuids(entity));
    }

    @PostMapping("/queryUserByRoleId")
    @ApiOperation(value = "通过角色id查询用户信息", notes = "通过角色查询用户信息")
    public Response<List<UserDTO>> queryUserByRoleId(@RequestBody UserRoleDTO entity) {
        if (entity.getIsDelete() == null) {
            entity.setIsDelete(CommonDeleteEnum.NORMAL.getCode());
        }
        if (CuscStringUtils.isEmpty(entity.getTenantNo()) || CuscStringUtils.isEmpty(entity.getApplicationId())
                || CuscStringUtils.isEmpty(entity.getRoleId())) {
            return Response.createError(ResponseCode.PARAMETER_NULL.getMsg(), ResponseCode.PARAMETER_NULL.getCode());
        }
        return Response.createSuccess(userService.queryUserByRoleId(entity));
    }

    @PostMapping("/getByPhone")
    @ApiOperation(value = "通过手机号查询用户详情", notes = "通过手机号查询用户详情")
    public Response<UserDTO> getByPhone(@RequestBody UserDTO entity) {
        if (entity == null || CuscStringUtils.isEmpty(entity.getPhone())) {
            return Response.createError(ResponseCode.USER_PHONE_NOT_NULL.getMsg(),
                    ResponseCode.USER_PHONE_NOT_NULL.getCode());
        }
        CommonParamterCheck.appIdAndTenantNoRequired(entity.getApplicationId(), entity.getTenantNo());
        return Response.createSuccess(userService.getUser(entity));
    }

    @PostMapping("/getByUserName")
    @ApiOperation(value = "通过登录名查询用户详情", notes = "通过登录名查询用户详情")
    public Response<UserDTO> getByUserName(@RequestBody UserDTO entity) {
        if (entity == null || CuscStringUtils.isEmpty(entity.getUserName())) {
            return Response.createError(ResponseCode.USER_NAME_NOT_NULL.getMsg(),
                    ResponseCode.USER_NAME_NOT_NULL.getCode());
        }
        CommonParamterCheck.appIdAndTenantNoRequired(entity.getApplicationId(), entity.getTenantNo());
        return Response.createSuccess(userService.getUser(entity));
    }

    @PostMapping("/getByEmail")
    @ApiOperation(value = "通过邮箱查询用户详情", notes = "通过邮箱查询用户详情")
    public Response<UserDTO> getByEmail(@RequestBody UserDTO entity) {
        if (entity == null || CuscStringUtils.isEmpty(entity.getEmail())) {
            return Response.createError(ResponseCode.USER_EMAIL_NOT_NULL.getMsg(),
                    ResponseCode.USER_EMAIL_NOT_NULL.getCode());
        }
        CommonParamterCheck.appIdAndTenantNoRequired(entity.getApplicationId(), entity.getTenantNo());
        return Response.createSuccess(userService.getUser(entity));
    }

    @PostMapping("/changeTenantAdmin")
    @ApiOperation(value = "更换租户管理员账号", notes = "更换租户管理员账号")
    public Response changeTenantAdmin(@RequestBody UserDTO entity) {
        if (entity == null || CuscStringUtils.isEmpty(entity.getUuid()) || CuscStringUtils.isEmpty(
                entity.getTenantNo())) {
            return Response.createError(ResponseCode.USER_UUID_NOT_NULL.getMsg(),
                    ResponseCode.USER_UUID_NOT_NULL.getCode());
        }
        return Response.createSuccess(userService.changeTenantAdmin(entity));
    }

    @PostMapping("/allUserRelUrlToRedis")
    @ApiOperation(value = "将租户下的所有用户对应的url写入redis", notes = "将租户下的所有用户对应的url写入redis")
    public Response allUserRelUrlToRedis(@RequestBody UserDTO entity) {
        userService.allUserRelUrlToRedis(entity);
        return Response.createSuccess();
    }

    @PostMapping("/changePassword")
    @ApiOperation(value = "修改密码", notes = "修改密码")
    public Response changePassword(@RequestBody UserDTO entity) {
        if (entity == null || CuscStringUtils.isEmpty(entity.getUuid()) || CuscStringUtils.isEmpty(entity.getPassword()) || CuscStringUtils.isEmpty(entity.getNewPassword())) {
            return Response.createError(ResponseCode.PARAMETER_NULL.getMsg(),
                    ResponseCode.PARAMETER_NULL.getCode());
        }
        CommonParamterCheck.appIdAndTenantNoRequired(entity.getApplicationId(), entity.getTenantNo());
        return Response.createSuccess(userService.changePassword(entity));
    }

    @PostMapping("/resetPassword")
    @ApiOperation(value = "重置密码", notes = "重置密码")
    public Response resetPassword(@RequestBody UserDTO entity) {
        if (entity == null || CuscStringUtils.isEmpty(entity.getUuid())) {
            return Response.createError(ResponseCode.USER_UUID_NOT_NULL.getMsg(),
                    ResponseCode.USER_UUID_NOT_NULL.getCode());
        }
        CommonParamterCheck.appIdAndTenantNoRequired(entity.getApplicationId(), entity.getTenantNo());
        return Response.createSuccess(userService.resetPassword(entity));
    }

    @PostMapping("/getUserOnline")
    @ApiOperation(value = "查询用户是否在线", notes = "查询用户是否在线")
    public Response getUserOnline(@RequestBody UserDTO entity) {
        if (entity == null || CuscStringUtils.isEmpty(entity.getUuid())) {
            return Response.createError(ResponseCode.USER_UUID_NOT_NULL.getMsg(),
                    ResponseCode.USER_UUID_NOT_NULL.getCode());
        }
        CommonParamterCheck.appIdAndTenantNoRequired(entity.getApplicationId(), entity.getTenantNo());
        return Response.createSuccess(userService.getUserOnline(entity));
    }
}
