package com.cusc.nirvana.user.auth.identification.service.impl;

import com.cache.CacheFactory;
import com.cache.exception.CacheException;
import com.cusc.nirvana.common.result.Response;
import com.cusc.nirvana.user.auth.common.constants.RedisConstant;
import com.cusc.nirvana.user.auth.common.constants.ResponseCode;
import com.cusc.nirvana.user.auth.common.constants.UserTypeEnum;
import com.cusc.nirvana.user.auth.common.service.AppConfigService;
import com.cusc.nirvana.user.auth.identification.dto.CaptchaVerificationReq;
import com.cusc.nirvana.user.auth.identification.dto.MobileLoginReq;
import com.cusc.nirvana.user.auth.identification.dto.Oauth2Token;
import com.cusc.nirvana.user.auth.identification.dto.SmsSendConfig;
import com.cusc.nirvana.user.auth.identification.dto.UserNameLoginReq;
import com.cusc.nirvana.user.auth.identification.service.ICaptchaService;
import com.cusc.nirvana.user.auth.identification.service.ILoginService;
import com.cusc.nirvana.user.auth.identification.service.IRandomIdService;
import com.cusc.nirvana.user.auth.identification.service.ISmsService;
import com.cusc.nirvana.user.auth.identification.service.ITokenService;
import com.cusc.nirvana.user.ciam.dto.CiamUserDTO;
import com.cusc.nirvana.user.ciam.service.ICiamUserService;
import com.cusc.nirvana.user.eiam.constants.CommonStatusEnum;
import com.cusc.nirvana.user.eiam.dto.ApplicationDTO;
import com.cusc.nirvana.user.eiam.dto.EiamUrlDTO;
import com.cusc.nirvana.user.eiam.dto.UserDTO;
import com.cusc.nirvana.user.eiam.service.IUrlService;
import com.cusc.nirvana.user.eiam.service.IUserService;
import com.cusc.nirvana.user.exception.CuscUserException;
import com.cusc.nirvana.user.util.CuscRandomUtils;
import com.cusc.nirvana.user.util.CuscStringUtils;
import com.cusc.nirvana.user.util.crypt.CryptKeyUtil;
import com.cusc.nirvana.user.util.crypt.Sm4Util;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * Description: 登录service实现类
 * <br />
 * CreateDate 2021-11-02 20:25:49
 *
 * @author yuyi
 **/
@Service
@Slf4j
public class LoginServiceImpl implements ILoginService {

    @Autowired
    private CacheFactory cacheFactory;

    @Autowired
    private IUserService userClient;

    @Autowired
    private IUrlService eiamUrlClient;

    @Autowired
    private ITokenService tokenService;

    @Autowired
    private ISmsService smsService;

    @Autowired
    private ICaptchaService captchaService;

    @Autowired
    private AppConfigService appConfigService;

    @Autowired
    private IRandomIdService randomIdService;

    @Autowired
    private ICiamUserService ciamUserClient;


    /**
     * Description: 手机号登录
     * <br />
     * CreateDate 2021-11-04 19:53:41
     *
     * @author yuyi
     **/
    @Override
    public Response<Oauth2Token> mobileLogin(MobileLoginReq bean) {
        //校验短信验证码
        Response ret = checkSmsCaptcha(bean);
        if (!ret.isSuccess()) {
            return ret;
        }
        //校验手机号等信息
        Response<UserDTO> retUser = checkUserByPhone(bean);
        if (!retUser.isSuccess()) {
            return Response.createError(retUser.getMsg(), retUser.getCode());
        }

        //将用户对应的url写入redis 异步
        eiamUrlClient.userRelUrlToRedis(retUser.getData().getUuid(), bean.getTenantNo(), bean.getApplicationId());

        //创建token
        return tokenService.createOauth2TokenByMobile(bean, retUser.getData());
    }

    /**
     * Description: C端用户手机号登录
     * <br />
     * CreateDate 2022-04-15 19:53:41
     *
     * @author huzl
     **/
    @Override
    public Response<Oauth2Token> ciamMobileLogin(MobileLoginReq bean) {
        //校验短信验证码
        Response ret = checkSmsCaptcha(bean);
        if (!ret.isSuccess()) {
            return ret;
        }
        //检查手机号是否存在
        Response<CiamUserDTO> userResp = checkCiamUserByPhone(bean);
        CiamUserDTO ciamUser = userResp.getData();

        //创建token
        UserDTO userDTO = new UserDTO();
        userDTO.setApplicationId(bean.getApplicationId());
        userDTO.setTenantNo(bean.getTenantNo());
        userDTO.setPhone(ciamUser.getPhoneNum());
        userDTO.setUuid(ciamUser.getUuid());
        return tokenService.createOauth2TokenByMobile(bean, userDTO);
    }


    /**
     * Description: C端用户手机号登录小鹏
     * <br />
     * CreateDate 2022-04-15 19:53:41
     *
     * @author huzl
     **/
    @Override
    public Response<Oauth2Token> ciamMobileLoginXP(MobileLoginReq bean) {
        //校验短信验证码
//        Response ret = checkSmsCaptcha(bean);
//        if (!ret.isSuccess()) {
//            return ret;
//        }

        //检查手机号是否存在
        UserDTO userDTO = new UserDTO();
        CiamUserDTO ciamUser = new CiamUserDTO();
        ciamUser.setPhoneNum(bean.getPhone());
        ciamUser.setTenantNo(bean.getTenantNo());
        CiamUserDTO selectCiamUserDTO = ciamUserClient.getUserByPhoneTenantNo(ciamUser);
        if(null!=selectCiamUserDTO){
            userDTO.setPhone(ciamUser.getPhoneNum());
            userDTO.setUuid(ciamUser.getUuid());
        }else{
            CiamUserDTO insertBean = new CiamUserDTO();
            insertBean.setPhoneNum(bean.getPhone());
            insertBean.setTenantNo(bean.getTenantNo());
            CiamUserDTO insertReturnBean = ciamUserClient.addOrGet(insertBean);
            userDTO.setPhone(insertReturnBean.getPhoneNum());
            userDTO.setUuid(insertReturnBean.getUuid());
        }
        userDTO.setApplicationId(bean.getApplicationId());
        userDTO.setTenantNo(bean.getTenantNo());
        return tokenService.createOauth2TokenByMobile(bean, userDTO);
    }

    @Override
    public Response<Boolean> checkSmsCaptcha(MobileLoginReq bean) {
        String smsCaptcha = null;
        try {
            smsCaptcha =
                    cacheFactory.getExpireStringService()
                            .getValue(RedisConstant.SMS_CAPTCHA_KEY + bean.getPhone() + "_" + bean.getTenantNo() + "_"
                                            + bean.getApplicationId(),
                                    String.class);

            if (CuscStringUtils.isEmpty(smsCaptcha) || !smsCaptcha.equals(bean.getCaptcha())) {
                return Response.createError(ResponseCode.SMS_CAPTCHA_INVALID.getMsg(),
                        ResponseCode.SMS_CAPTCHA_INVALID.getCode() + "");
            }
            //验证成功之后清理验证码
            cacheFactory.getExpireStringService()
                    .delete(RedisConstant.SMS_CAPTCHA_KEY + bean.getPhone() + "_" + bean.getTenantNo() + "_"
                            + bean.getApplicationId());
        } catch (CacheException e) {
            log.error("checkSmsCaptcha 获取reids失败 ：", e);
            Response.createError(ResponseCode.SMS_GET_CAPTCHA_FAIL.getMsg(),
                    ResponseCode.SMS_GET_CAPTCHA_FAIL.getCode() + "");
        }

        return Response.createSuccess(true);
    }

    @Override
    public Response<Oauth2Token> userNameLogin(UserNameLoginReq bean) {
        //获取应用配置
        ApplicationDTO appBean = appConfigService.getAppConfigByCode(bean.getApplicationId());

        //检查账号是否锁定
        checkUserNameLock(bean);

        //通过requestId 解密密码
        String respPwd = bean.getPassword();
        if (CuscStringUtils.isNotEmpty(bean.getRequestId())) {
            String secretKey = randomIdService.getByRequestId(bean.getRequestId(), bean.getApplicationId());
            if (CuscStringUtils.isEmpty(secretKey)) {
                return Response.createError(ResponseCode.REQUEST_ID_SECRET_KEY_INVALID.getMsg(),
                        ResponseCode.REQUEST_ID_SECRET_KEY_INVALID.getCode());
            }
            //删除随机数
            randomIdService.delRequestIdRedis(bean.getRequestId(), bean.getApplicationId());
            respPwd = Sm4Util.decryptEcbPadding(secretKey, respPwd);
        }

        //通过用户名查询用户信息
        Response<UserDTO> retUser = checkUserByUserName(bean);
        if (!retUser.isSuccess()) {
            //增加错误次数
            checkPwdFailCount(bean, appBean);
            return Response.createError(retUser.getMsg(), retUser.getCode());
        }

        //将输入的密码进行加密
        respPwd = Sm4Util.encryptEcbPadding(retUser.getData().getUuid().substring(16), respPwd);

        //判断密码是否正确
        if (!respPwd.equals(retUser.getData().getPassword())) {
            //增加错误次数
            checkPwdFailCount(bean, appBean);
            return Response.createError(ResponseCode.LOGIN_USER_NAME_PASSWORD_INVALID.getMsg(),
                    ResponseCode.LOGIN_USER_NAME_PASSWORD_INVALID.getCode());
        }

        //将用户对应的url写入redis 异步
        EiamUrlDTO urlDTO = new EiamUrlDTO();
        urlDTO.setUserId(retUser.getData().getUuid());
        urlDTO.setApplicationId(bean.getApplicationId());
        urlDTO.setTenantNo(bean.getTenantNo());
        eiamUrlClient.userRelUrlToRedis(retUser.getData().getUuid(), bean.getTenantNo(), bean.getApplicationId());

        //创建token
        return tokenService.createOauth2TokenByUserName(bean, retUser.getData());
    }

    @Override
    public Response<Boolean> sendSmsCaptcha(MobileLoginReq bean) {
        //获取应用配置
        ApplicationDTO appBean = appConfigService.getAppConfigByCode(bean.getApplicationId());
        SmsSendConfig smsConfig = bean.getSmsSendConfig();
        //短信配置为空，从应用配置中取
        smsService.convertToSmsConfig(appBean, smsConfig);
        smsConfig.setAppId(bean.getApplicationId());
        smsConfig.setTenantNo(bean.getTenantNo());

        //判断是否需要验证图形验证码
        if (bean.isCheckCaptchaImg()) {
            CaptchaVerificationReq cv = new CaptchaVerificationReq();
            cv.setRequestId(bean.getRequestId());
            cv.setCaptchaValue(bean.getCaptchaImage());
            cv.setApplicationId(bean.getApplicationId());
            boolean checkCaptchaImg = captchaService.verificationCaptcha(cv);
            if (!checkCaptchaImg) {
                return Response.createError(ResponseCode.CAPTCHA_IMAGGE_CHECK_FAIL.getMsg(),
                        ResponseCode.CAPTCHA_IMAGGE_CHECK_FAIL.getCode());
            }
        }

        String smsCaptcha;
        try {
            //判断当前验证码是否达到发送间隔
            if (cacheFactory.getExpireStringService().containsKey(
                    RedisConstant.SMS_CAPTCHA_SEND_INTERVAL_KEY + bean.getPhone() + "_" + bean.getTenantNo() + "_"
                            + bean.getApplicationId())) {
                return Response.createError(ResponseCode.SMS_CAPTCHA_INTERVAL_FAIL.getMsg(),
                        ResponseCode.SMS_CAPTCHA_INTERVAL_FAIL.getCode());
            }
            smsConfig.setTotalLimitKey(RedisConstant.SMS_CAPTCHA_SEND_TOTAL_KEY);
            smsService.checkSmsSendLimit(bean.getPhone(), smsConfig);

            Response ret;
            //检查手机号是否正确
            if (bean.getUserType() == null || UserTypeEnum.EIAM.getCode() == bean.getUserType().intValue()) {
                //EIAM用户
                ret = checkUserByPhone(bean);
            } else {
                //CIAM用户
                ret = checkCiamUserByPhone(bean);
            }
            if (!ret.isSuccess()) {
                return ret;
            }

            //创建随机验证
            smsCaptcha = CuscRandomUtils.randomNumeric(6);
            int captchaExpire = RedisConstant.SMS_CAPTCHA_EXPIRE;
            if (bean.getCaptchaExpire() != null) {
                captchaExpire = bean.getCaptchaExpire();
            }
            //放到redis
            cacheFactory.getExpireStringService()
                    .setExpireValue(RedisConstant.SMS_CAPTCHA_KEY + bean.getPhone() + "_" + bean.getTenantNo() + "_"
                                    + bean.getApplicationId(),
                            smsCaptcha,
                            captchaExpire);
        } catch (CacheException e) {
            log.error("sendSmsCaptcha 存放reids失败：", e);
            return Response.createError(ResponseCode.SMS_CREATE_CAPTCHA_FAIL.getMsg(),
                    ResponseCode.SMS_CREATE_CAPTCHA_FAIL.getCode());
        }

        smsConfig.setIntervalLimitKey(RedisConstant.SMS_CAPTCHA_SEND_INTERVAL_KEY);
        //发送短信
        smsService.sendSms(bean.getPhone(), smsCaptcha, smsConfig);

        return Response.createSuccess(true);
    }

    //-----------------私有方法区--------------------------------

    /**
     * Description: 通过手机号检查用户-eiam
     * <br />
     * CreateDate 2021-11-04 20:26:42
     *
     * @author yuyi
     **/
    private Response<UserDTO> checkUserByPhone(MobileLoginReq bean) {
        //检查手机号是否存在
        UserDTO user = new UserDTO();
        user.setPhone(bean.getPhone());
        user.setTenantNo(bean.getTenantNo());
        user.setApplicationId(bean.getApplicationId());
        UserDTO userResp = userClient.getUser(user);
        if (userResp == null || CuscStringUtils.isEmpty(userResp.getPhone())) {
            return Response.createError(ResponseCode.LOGIN_NAME_INVALID.getMsg(),
                    ResponseCode.LOGIN_NAME_INVALID.getCode());
        }
        user = userResp;
        //检查状态是否正确
        if (CommonStatusEnum.ENABLE.getCode() != user.getStatus()) {
            return Response.createError(ResponseCode.LOGIN_NAME_STOP.getMsg(),
                    ResponseCode.LOGIN_NAME_STOP.getCode());
        }
        return Response.createSuccess(user);
    }

    /**
     * Description: 通过手机号检查用户-ciam
     * <br />
     * CreateDate 2021-11-04 20:26:42
     *
     * @author yuyi
     **/
    private Response<CiamUserDTO> checkCiamUserByPhone(MobileLoginReq bean) {
        CiamUserDTO ciamUser = new CiamUserDTO();
        ciamUser.setPhoneNum(bean.getPhone());
        ciamUser.setTenantNo(bean.getTenantNo());
        //ciamUser.setStatus(CommonStatusEnum.ENABLE.getCode());
        CiamUserDTO userResp = ciamUserClient.getUserByPhoneTenantNo(ciamUser);
        if (userResp == null || CuscStringUtils.isEmpty(userResp.getPhoneNum())) {
            return Response.createError(ResponseCode.LOGIN_PHONE_INVALID.getMsg(),
                    ResponseCode.LOGIN_PHONE_INVALID.getCode());
        }
        ciamUser = userResp;
        //检查状态是否正确
        if (CommonStatusEnum.ENABLE.getCode() != ciamUser.getStatus()) {
            return Response.createError(ResponseCode.LOGIN_NAME_STOP.getMsg(),
                    ResponseCode.LOGIN_NAME_STOP.getCode());
        }
        return Response.createSuccess(ciamUser);
    }

    /**
     * Description: 通过手机号检查用户
     * <br />
     * CreateDate 2021-11-04 20:26:42
     *
     * @author yuyi
     **/
    private Response<UserDTO> checkUserByUserName(UserNameLoginReq bean) {
        //检查手机号是否存在
        UserDTO user = new UserDTO();
        user.setUserName(bean.getUserName());
        user.setTenantNo(bean.getTenantNo());
        user.setApplicationId(bean.getApplicationId());
        UserDTO userResp = userClient.getUser(user);
        if (userResp == null) {
            return Response.createError(ResponseCode.LOGIN_NAME_INVALID.getMsg(),
                    ResponseCode.LOGIN_NAME_INVALID.getCode());
        }
        user = userResp;
        //检查状态是否正确
        if (CommonStatusEnum.ENABLE.getCode() != user.getStatus()) {
            return Response.createError(ResponseCode.LOGIN_NAME_STOP.getMsg(),
                    ResponseCode.LOGIN_NAME_STOP.getCode());
        }
        return Response.createSuccess(user);
    }

    /**
     * Description: 保存密码失败次数
     * <br />
     * CreateDate 2022-02-17 11:03:42
     *
     * @author yuyi
     **/
    private void checkPwdFailCount(UserNameLoginReq bean, ApplicationDTO appDTO) {
        //密码错误锁定次数、时间、期限为空或小于0时，不生效
        if (appDTO.getPwsErrorLockNum() == null || appDTO.getPwsErrorLockNum() <= 0
                || appDTO.getPwsErrorLockTerm() == null || appDTO.getPwsErrorLockTerm() <= 0
                || appDTO.getPwsErrorLockTime() == null || appDTO.getPwsErrorLockTime() <= 0) {
            return;
        }
        try {
            Integer failCount = cacheFactory.getExpireStringService().getValue(
                    RedisConstant.USERNAME_PASSWORD_FAIL_COUNT_KEY + bean.getUserName() + "_" + bean.getTenantNo()
                            + "_" + bean.getApplicationId(), Integer.class);
            int expireTime;
            if (failCount == null || failCount == 0) {
                failCount = 1;
                expireTime = appDTO.getPwsErrorLockTerm();
            } else {
                failCount++;
                expireTime = cacheFactory.getExpireStringService().getKeyExpireTime(
                        RedisConstant.USERNAME_PASSWORD_FAIL_COUNT_KEY + bean.getUserName() + "_"
                                + bean.getTenantNo()
                                + "_" + bean.getApplicationId()).intValue();
            }
            cacheFactory.getExpireStringService()
                    .setExpireValue(
                            RedisConstant.USERNAME_PASSWORD_FAIL_COUNT_KEY + bean.getUserName() + "_"
                                    + bean.getTenantNo()
                                    + "_" + bean.getApplicationId(),
                            failCount, expireTime);

            if (failCount >= appDTO.getPwsErrorLockNum()) {
                //锁定账号
                cacheFactory.getExpireStringService()
                        .setExpireValue(
                                RedisConstant.USERNAME_LOCK_KEY + bean.getUserName() + "_" + bean.getTenantNo()
                                        + "_" + bean.getApplicationId(),
                                failCount, appDTO.getPwsErrorLockTime());
            }
        } catch (CacheException e) {
            log.error("保存密码失败次数至redis异常：", e);
        }
    }

    /**
     * Description: 检查账号是否锁定
     * <br />
     * CreateDate 2022-02-17 11:03:42
     *
     * @author yuyi
     **/
    private void checkUserNameLock(UserNameLoginReq bean) {
        boolean isLock = false;
        try {
            isLock = cacheFactory.getExpireStringService().containsKey(
                    RedisConstant.USERNAME_LOCK_KEY + bean.getUserName() + "_" + bean.getTenantNo()
                            + "_" + bean.getApplicationId());
        } catch (CacheException e) {
            log.error("检查密码失败次数访问redis异常：", e);
        }
        if (isLock) {
            throw new CuscUserException(ResponseCode.USER_NAME_PWD_FAIL_LOCK.getCode(),
                    ResponseCode.USER_NAME_PWD_FAIL_LOCK.getMsg());
        }
    }
}
