package com.cusc.nirvana.user.auth.authentication.plug.interceptor;

import com.cusc.nirvana.common.result.Response;
import com.cusc.nirvana.user.auth.authentication.dto.AccessVerifyReq;
import com.cusc.nirvana.user.auth.authentication.plug.constants.AuthConstant;
import com.cusc.nirvana.user.auth.authentication.plug.util.ThreadLocalUtil;
import com.cusc.nirvana.user.auth.client.AccessAuthClient;
import com.cusc.nirvana.user.auth.common.dto.AccessTokenHashDTO;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.util.StringUtils;
import org.springframework.web.servlet.HandlerInterceptor;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.*;

/**
 * Description: 鉴权拦截器-开发拦截模式
 * <br />
 * 注：开发拦截模式是所有url放行（传了token会自动获取用户id透传），否则用户id使用配置中的固定用户
 * CreateDate 2021-11-05 16:00
 *
 * @author yuyi
 **/
public class DevAuthInterceptor implements HandlerInterceptor {

    private static final Logger LOGGER = LoggerFactory.getLogger(DevAuthInterceptor.class);

    @Value("${spring.application.name}")
    private String serverName;

    @Value("${auth.userId: 1}")
    private String userId;

    @Value("${auth.appId: 1}")
    private String appId;

    @Value("${auth.tenantNo: 1}")
    private String tenantNo;

    @Autowired
    private AccessAuthClient accessAuthClient;

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) {

        String requestUri = request.getRequestURI();

        //从请求头获取token
        String token = request.getHeader("Authorization");

        AccessVerifyReq accessVerifyReq = new AccessVerifyReq();
        accessVerifyReq.setServerName(serverName);
        accessVerifyReq.setUrl(requestUri);
        //检查请求的url是否是白名单
        Response<Boolean> isWhiteListResp = accessAuthClient.isWhiteList(accessVerifyReq);

        //token获取的用户id、应用id等信息
        Response<AccessTokenHashDTO> authResp;
        if (isWhiteListResp != null && isWhiteListResp.isSuccess() && isWhiteListResp.getData()) {
            LOGGER.info("DevAuthInterceptor serverName:{}, url:{}, token:{}. white list", serverName, requestUri,
                    token);
        } else {
            LOGGER.info("DevAuthInterceptor url:{},token:{}", requestUri, token);
        }

        if (!StringUtils.isEmpty(token)) {
            token = token.replaceAll("bearer ", "");
            accessVerifyReq.setToken(token);
            authResp = accessAuthClient.getUserByToken(accessVerifyReq);
            if (authResp.isSuccess() && authResp.getData() != null) {
                ThreadLocalUtil.set(AuthConstant.USER_ID_NAME, authResp.getData().getUserId());
                ThreadLocalUtil.set(AuthConstant.APP_ID_NAME, authResp.getData().getAppId());
                ThreadLocalUtil.set(AuthConstant.TENANT_NO_NAME, authResp.getData().getTenantNo());
            }
        } else {
            ThreadLocalUtil.set(AuthConstant.USER_ID_NAME, userId);
            ThreadLocalUtil.set(AuthConstant.APP_ID_NAME, appId);
            ThreadLocalUtil.set(AuthConstant.TENANT_NO_NAME, tenantNo);
        }
        LOGGER.info("DevAuthInterceptor url:{},userId:{},appId:{},tenantNo:{}", requestUri, userId, appId, tenantNo);
        return true;
    }

    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler,
                                Exception ex) {
        ThreadLocalUtil.removeByKey(AuthConstant.USER_ID_NAME);
    }

    private void returnJson(HttpServletResponse response, String json) {
        response.setCharacterEncoding("utf-8");
        response.setContentType("application/json; charset=utf-8");
        PrintWriter writer = null;
        try {
            writer = response.getWriter();
            writer.write(json);
        } catch (IOException e) {
            LOGGER.error("AuthInterceptor 响应json 异常： {}", e);
        } finally {
            if (writer != null) {
                writer.close();
            }
        }
    }
}
