package com.cusc.nirvana.web.handler;

import com.cusc.nirvana.common.result.Response;
import com.cusc.nirvana.web.exception.AppGlobalException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.annotation.Order;
import org.springframework.http.HttpStatus;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.validation.BindException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;
import java.util.Set;

/**
 * @author jeff.chen
 * @des: 统一异常处理
 * @E-mail chenjf159@chinaunicom.cn
 */
@Order(1)
@ControllerAdvice
public class AppGlobalExceptionHandler {

    private Logger logger = LoggerFactory.getLogger(AppGlobalExceptionHandler.class);

    @ExceptionHandler({IllegalArgumentException.class, MethodArgumentNotValidException.class, AppGlobalException.class})
    @ResponseBody
    @ResponseStatus(HttpStatus.OK)
    public Response methodArgumentNotValidExceptionHandle(Exception ex) {
        String message = "";
        if (ex instanceof IllegalArgumentException) {
            logger.error("IllegalArgumentException:", ex);
            // 参数校验不通过
            IllegalArgumentException illegalArgumentException = ((IllegalArgumentException) ex);
            message = illegalArgumentException.getMessage();
        } else if (ex instanceof MethodArgumentNotValidException) {
            logger.error("MethodArgumentNotValidException:", ex);

            // 参数校验不通过
            MethodArgumentNotValidException methodArgumentNotValidException = ((MethodArgumentNotValidException) ex);
            message = methodArgumentNotValidException.getBindingResult().getAllErrors().get(0).getDefaultMessage();
        } else if (ex instanceof AppGlobalException) {
            logger.error("AppGlobalException:", ex);
            // 自定义异常
            AppGlobalException appGlobalException = (AppGlobalException) ex;
            Response response = appGlobalException.getRes();
            if (response.getData() instanceof Throwable) {
                response.setData(null);
            }

            return response;
        } else {
            message = "未知系统异常200";
        }

        return Response.createError(message);
    }

    @ExceptionHandler(Exception.class)
    @ResponseBody
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    public Response validExceptionHandle(Exception ex) {
        String message;
        if (ex instanceof BindException) {
            BindException bingException = (BindException) ex;
            message = bingException.getAllErrors().get(0).getDefaultMessage();
        } else if (ex instanceof HttpMessageNotReadableException) {
            HttpMessageNotReadableException notReadableException = (HttpMessageNotReadableException) ex;
            message = notReadableException.getMessage();
        } else if (ex instanceof ConstraintViolationException) {
            //参数校验异常，controller层直接参数校验（单个字段校验）
            ConstraintViolationException validationException = (ConstraintViolationException) ex;
            Set<ConstraintViolation<?>> violationErrors = validationException.getConstraintViolations();
            ConstraintViolation<?> error = violationErrors.iterator().next();
            message = error.getMessage();
        } else if (ex instanceof HttpRequestMethodNotSupportedException) {
            HttpRequestMethodNotSupportedException methodNotSupportedException = (HttpRequestMethodNotSupportedException) ex;
            message = methodNotSupportedException.getMessage();
        } else {
            message = "系统繁忙，请稍后重试！";
        }

        logger.error("{}:", message, ex);
        return Response.createError(message);
    }

}
