package com.cusc.nirvana.log.format;

import ch.qos.logback.classic.pattern.ClassicConverter;
import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.CoreConstants;
import ch.qos.logback.core.util.CachingDateFormatter;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.TimeZone;

/**
 * 解决日志时间格式化时，DateConverter Thread.State: BLOCKED (on object monitor)
 *
 * @author jeff.chen
 * @file LogDateConverter
 * @E-mail chenjf159@chinaunicom.cn
 */
public class LogDateConverter extends ClassicConverter {

    ThreadLocal<CachingDateFormatter> cache = ThreadLocal.withInitial(() -> initCachingDateFormatter());

    @Override
    public String convert(ILoggingEvent event) {
        long timestamp = event.getTimeStamp();
        if (Objects.nonNull(cache.get())) {
            return cache.get().format(timestamp);
        } else {
            return new SimpleDateFormat(CoreConstants.ISO8601_PATTERN).format(new Date(timestamp));
        }
    }

    @Override
    public void stop() {
        super.stop();
        cache.remove();
    }

    public CachingDateFormatter initCachingDateFormatter() {

        CachingDateFormatter cachingDateFormatter;

        String datePattern = getFirstOption();
        if (datePattern == null) {
            datePattern = CoreConstants.ISO8601_PATTERN;
        }

        if (datePattern.equals(CoreConstants.ISO8601_STR)) {
            datePattern = CoreConstants.ISO8601_PATTERN;
        }

        try {
            cachingDateFormatter = new CachingDateFormatter(datePattern);
            // maximumCacheValidity =
            // CachedDateFormat.getMaximumCacheValidity(pattern);
        } catch (IllegalArgumentException e) {
            addWarn("Could not instantiate SimpleDateFormat with pattern " + datePattern, e);
            // default to the ISO8601 format
            cachingDateFormatter = new CachingDateFormatter(CoreConstants.ISO8601_PATTERN);
        }

        List<String> optionList = getOptionList();

        // if the option list contains a TZ option, then set it.
        if (optionList != null && optionList.size() > 1) {
            TimeZone tz = TimeZone.getTimeZone((String) optionList.get(1));
            cachingDateFormatter.setTimeZone(tz);
        }

        return cachingDateFormatter;
    }

}
