package com.cusc.nirvana.log.aop;

import ch.qos.logback.core.CoreConstants;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.cusc.nirvana.log.annotation.OperateLog;
import com.cusc.nirvana.log.context.LogRecordContext;
import com.cusc.nirvana.log.dto.LogDTO;
import com.cusc.nirvana.log.function.CustomFunctionRegistrar;
import org.apache.commons.lang3.StringUtils;
import org.apache.skywalking.apm.toolkit.trace.TraceContext;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.DefaultParameterNameDiscoverer;
import org.springframework.expression.Expression;
import org.springframework.expression.spel.standard.SpelExpressionParser;
import org.springframework.expression.spel.support.StandardEvaluationContext;
import org.springframework.stereotype.Component;
import org.springframework.util.StopWatch;

import java.lang.reflect.Method;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

@Aspect
@Component
public class OperateLogAspect {
    private final Logger operatorLog = LoggerFactory.getLogger("OPERATOR_LOG");
    private final Logger log = LoggerFactory.getLogger(OperateLogAspect.class);

    private final SpelExpressionParser parser = new SpelExpressionParser();

    private final DefaultParameterNameDiscoverer discoverer = new DefaultParameterNameDiscoverer();

    @Around("@annotation(com.cusc.nirvana.log.annotation.OperateLog)")
    public Object doAround(ProceedingJoinPoint pjp) throws Throwable{
        Object result;
        List<LogDTO> logS = new ArrayList<>();
        StopWatch stopWatch = new StopWatch();
        try {
            logS = resolveExpress(pjp);
            stopWatch.start();
            result = pjp.proceed();
            stopWatch.stop();
            logS.forEach(logDTO -> {
                logDTO.setSuccess(true);
            });
        } catch (Throwable throwable) {
            stopWatch.stop();
            // 方法执行异常，自定义上下文未写入，但是仍然需要写入其他变量
            logS = resolveExpress(pjp);
            logS.forEach(logDTO -> {
                logDTO.setSuccess(false);
                logDTO.setException(throwable.getMessage());
            });
            throw throwable;
        } finally {
            logS.forEach(logDTO -> {
                try {
                    //记录本地日志
                    MDC.put("bizId", logDTO.getBizId());
                    MDC.put("operator", logDTO.getOperator());
                    MDC.put("operateType", logDTO.getOperateType().getValue());
                    MDC.put("operateDate", dateFormat(logDTO.getOperateDate()));
                    MDC.put("operateResult", logDTO.getSuccess() ? "true" : "false");
                    MDC.put("operateExp", logDTO.getException());
                    MDC.put("traceId", TraceContext.traceId());
                    operatorLog.info(logDTO.getMsg());
                } catch (Throwable throwable) {
                    log.error("OperateLogAspect doAround send log error", throwable);
                }
            });
            // 清除变量上下文
            LogRecordContext.clearContext();
        }
        return result;
    }


    public List<LogDTO> resolveExpress(JoinPoint joinPoint) {
        try {
            List<LogDTO> logDTOList = new ArrayList<>();
            Object[] arguments = joinPoint.getArgs();
            Method method = getMethod(joinPoint);
            OperateLog[] annotations = method.getAnnotationsByType(OperateLog.class);
            for (OperateLog annotation : annotations) {
                LogDTO logDTO = new LogDTO();
                logDTOList.add(logDTO);
                String bizIdSpel = annotation.bizId();
                String operatorSpel = annotation.operator();
                String msgSpel = annotation.msg();
                String bizId = bizIdSpel;
                String operator = operatorSpel;
                String msg = msgSpel;
                try {
                    String[] params = discoverer.getParameterNames(method);
                    StandardEvaluationContext context = LogRecordContext.getContext();
                    CustomFunctionRegistrar.register(context);
                    if (params != null) {
                        for (int len = 0; len < params.length; len++) {
                            context.setVariable(params[len], arguments[len]);
                        }
                    }

                    // bizId 处理：直接传入字符串会抛出异常，写入默认传入的字符串
                    if (StringUtils.isNotBlank(bizIdSpel)) {
                        Expression bizIdExpression = parser.parseExpression(bizIdSpel);
                        bizId = bizIdExpression.getValue(context, String.class);
                    }

                    // operator 处理：直接传入字符串会抛出异常，写入默认传入的字符串
                    if (StringUtils.isNotBlank(operatorSpel)) {
                        Expression operatorExpression = parser.parseExpression(operatorSpel);
                        operator = operatorExpression.getValue(context, String.class);
                    }

                    // msg 处理：写入默认传入的字符串
                    if (StringUtils.isNotBlank(msgSpel)) {
                        Expression msgExpression = parser.parseExpression(msgSpel);
                        Object msgObj = msgExpression.getValue(context, Object.class);
                        msg = msgObj instanceof String ? String.valueOf(msgObj) : JSON.toJSONString(msgObj, SerializerFeature.WriteMapNullValue);
                    }

                } catch (Exception e) {
                    log.error("OperateLogAspect resolveExpress error", e);
                } finally {
                    logDTO.setOperator(operator);
                    logDTO.setBizId(bizId);
                    logDTO.setOperateType(annotation.operatorType());
                    logDTO.setOperateDate(new Date());
                    logDTO.setMsg(msg);
                }
            }
            return logDTOList;

        } catch (Exception e) {
            log.error("OperateLogAspect resolveExpress error", e);
            return new ArrayList<>();
        }
    }


    protected Method getMethod(JoinPoint joinPoint) {
        Method method = null;
        try {
            Signature signature = joinPoint.getSignature();
            MethodSignature ms = (MethodSignature) signature;
            Object target = joinPoint.getTarget();
            method = target.getClass().getMethod(ms.getName(), ms.getParameterTypes());
        } catch (NoSuchMethodException e) {
            log.error("OperateLogAspect getMethod error", e);
        }
        return method;
    }

    private String dateFormat(Date date) {
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat(CoreConstants.ISO8601_PATTERN);
        return simpleDateFormat.format(date);
    }
}
