package com.cusc.nirvana.common.http.client;

import com.cusc.nirvana.common.Callback;
import com.cusc.nirvana.common.loader.EnhancedServiceLoader;
import okhttp3.*;
import okhttp3.internal.Util;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * @author jeff.chen
 * @file HttpBase
 * @E-mail chenjf159@chinaunicom.cn
 */
public class HttpBase {

    private static Logger logger = LoggerFactory.getLogger(HttpBase.class);

    /**
     * 常量
     */
    public static final String CONTENT_TYPE = "content-type";

    public final static String HTTP_CONTENT_TYPE_JSON = "application/json;charset=UTF-8";

    public final static String CONNECTION = "Connection";

    public final static String KEEP_ALIVE = "Keep-Alive";

    public final static String CLOSE = "close";

    public final static String HTTPS = "https";

    private static final OkHttpClient[] OK_HTTP_CLIENTS = new OkHttpClient[3];

    private static final int HTTP_INDEX = 0;

    private static final int HTTPS_SSL_INDEX = 1;

    private static final int HTTPS_TSL_INDEX = 2;

    /**
     * ms
     */
    private static final int DEFAULT_TIMEOUT = 10_000;

    private static final int DEFAULT_MAX_IDLE_CONNECTIONS = 5;

    private static final int DEFAULT_KEEP_ALIVE_DURATION = 5;

    private static final Object LOCK = new Object();

    protected static final byte[] DEFAULT_EMPTY = new byte[0];

    protected static final List<HttpHook> GLOBAL_HOOKS = new ArrayList<>();

    private static final AtomicBoolean LOADED_INVOKERS = new AtomicBoolean(false);

    private static final AbstractHttpInvokerHandler HEADER_HANDLER = new AbstractHttpInvokerHandler() {

        @Override
        public Response invoke(Request request, Callback<Response> callback) throws Throwable {
            return getNext().invoke(request, callback);
        }

    };

    private static final AbstractHttpInvokerHandler TAIL_HANDLER = new AbstractHttpInvokerHandler() {

        @Override
        public Response invoke(Request request, Callback<Response> callback) throws Throwable {
            return callback.call();
        }

    };

    protected static OkHttpClient client(String url) {
        if (isBlank(url)) {
            throw new IllegalArgumentException("cusc HttpBase url is blank");
        }

        if (url.toLowerCase().contains(HTTPS)) {
            return sslClient();
        } else {
            return client();
        }
    }

    /**
     * request
     */
    protected static Request getRequest(String url, Map<String, String> headers) {
        Request.Builder builder = new Request.Builder().url(url);

        HttpHelper.wrapTracing(builder, headers);

        return builder.get().build();
    }

    protected static Request postRequest(String url, Map<String, String> headers, String data, String contentType) {
        MediaType mediaType = MediaType.parse(contentType);

        Request.Builder builder = new Request.Builder().url(url).addHeader(CONTENT_TYPE, contentType);

        HttpHelper.wrapTracing(builder, headers);

        RequestBody body = isBlank(data) ? RequestBody.create(mediaType, DEFAULT_EMPTY)
                : RequestBody.create(mediaType, data);

        return builder.post(body).build();
    }

    protected static Request postRequest(String url, Map<String, String> headers, byte[] data, String contentType) {
        MediaType mediaType = MediaType.parse(contentType);

        Request.Builder builder = new Request.Builder().url(url).addHeader(CONTENT_TYPE, contentType);

        HttpHelper.wrapTracing(builder, headers);

        RequestBody body = RequestBody.create(mediaType, ((null == data) ? DEFAULT_EMPTY : data));

        return builder.post(body).build();
    }

    protected static Request XRequest(String url, HttpMethod httpMethod, Map<String, String> headers, byte[] data, String contentType) {
        Request.Builder builder = new Request.Builder().url(url).addHeader(CONTENT_TYPE, contentType);

        HttpHelper.wrapTracing(builder, headers);

        switch (httpMethod) {
            case GET:
            case HEAD:
                return builder.method(httpMethod.method, null).build();
            case PUT:
            case POST:
            case PATCH:
            case DELETE:
                MediaType mediaType = MediaType.parse(contentType);
                RequestBody body = (null == data) ? Util.EMPTY_REQUEST : RequestBody.create(mediaType, data);
                return builder.method(httpMethod.method, body).build();
            default:
                throw new IllegalStateException("cusc HttpBase X-Request");
        }
    }

    protected static Request XRequest(String url, HttpMethod httpMethod, Map<String, String> headers, RequestBody body, String contentType) {
        Request.Builder builder = new Request.Builder().url(url).addHeader(CONTENT_TYPE, contentType);

        HttpHelper.wrapTracing(builder, headers);

        return builder.method(httpMethod.method, body).build();
    }

    /**
     * tool func
     */

    public static boolean isBlank(final CharSequence cs) {
        int strLen;
        if (cs == null || (strLen = cs.length()) == 0) {
            return true;
        }
        for (int i = 0; i < strLen; i++) {
            if (!Character.isWhitespace(cs.charAt(i))) {
                return false;
            }
        }
        return true;
    }

    /**
     * http client
     */
    private static OkHttpClient client() {
        OkHttpClient client = OK_HTTP_CLIENTS[HTTP_INDEX];
        if (null == client) {
            synchronized (LOCK) {
                client = OK_HTTP_CLIENTS[HTTP_INDEX];
                if (null == client) {
                    ConnectionPool pool = new ConnectionPool(DEFAULT_MAX_IDLE_CONNECTIONS, DEFAULT_KEEP_ALIVE_DURATION, TimeUnit.MINUTES);
                    client = new OkHttpClient().newBuilder()
                            .readTimeout(DEFAULT_TIMEOUT, TimeUnit.MILLISECONDS)
                            .connectTimeout(DEFAULT_TIMEOUT, TimeUnit.MILLISECONDS)
                            .writeTimeout(DEFAULT_TIMEOUT, TimeUnit.MILLISECONDS)
                            .connectionPool(pool)
                            .build();
                    OK_HTTP_CLIENTS[HTTP_INDEX] = client;
                }
            }
        }

        return client;
    }

    private static OkHttpClient sslClient() {
        OkHttpClient sslClient = OK_HTTP_CLIENTS[HTTPS_SSL_INDEX];
        if (null == sslClient) {
            synchronized (LOCK) {
                sslClient = OK_HTTP_CLIENTS[HTTPS_SSL_INDEX];
                if (null == sslClient) {
                    ConnectionPool pool = new ConnectionPool(DEFAULT_MAX_IDLE_CONNECTIONS, DEFAULT_KEEP_ALIVE_DURATION, TimeUnit.MINUTES);
                    OkHttpClient.Builder builder = new OkHttpClient.Builder()
                            .readTimeout(DEFAULT_TIMEOUT, TimeUnit.MILLISECONDS)
                            .connectTimeout(DEFAULT_TIMEOUT, TimeUnit.MILLISECONDS)
                            .writeTimeout(DEFAULT_TIMEOUT, TimeUnit.MILLISECONDS)
                            .connectionPool(pool);

                    try {
                        SSLContext sc = SSLContext.getInstance("SSL");

                        X509TrustManager defaultX509TrustManager = new X509TrustManager() {
                            @Override
                            public void checkClientTrusted(X509Certificate[] x509Certificates, String s) {

                            }

                            @Override
                            public void checkServerTrusted(X509Certificate[] x509Certificates, String s) {

                            }

                            @Override
                            public X509Certificate[] getAcceptedIssuers() {
                                return new X509Certificate[0];
                            }
                        };

                        sc.init(null, new TrustManager[]{defaultX509TrustManager}, new SecureRandom());

                        sslClient = builder.sslSocketFactory(sc.getSocketFactory(), defaultX509TrustManager).build();
                        OK_HTTP_CLIENTS[HTTPS_SSL_INDEX] = sslClient;
                    } catch (NoSuchAlgorithmException e) {
                        e.printStackTrace();
                    } catch (KeyManagementException e) {
                        e.printStackTrace();
                    }
                }
            }
        }

        return sslClient;
    }

//    private static OkHttpClient tslClient() {
//        OkHttpClient tslClient = OK_HTTP_CLIENTS[HTTPS_TSL_INDEX];
//        if (null == tslClient) {
//            synchronized (LOCK) {
//                tslClient = OK_HTTP_CLIENTS[HTTPS_TSL_INDEX];
//                if (null == tslClient) {
//                    OkHttpClient.Builder builder = new OkHttpClient.Builder();
//
//                    try {
//                        HandshakeCertificates clientCertificates = new HandshakeCertificates.Builder()
//                                .addTrustedCertificate(localhostCertificate.certificate())
//                                .build();
//
//                    } catch (NoSuchAlgorithmException e) {
//                        e.printStackTrace();
//                    } catch (KeyManagementException e) {
//                        e.printStackTrace();
//                    }
//                }
//            }
//        }
//
//        return tslClient;
//    }

    public static void addHttpHooks(List<HttpHook> hooks) {
        GLOBAL_HOOKS.addAll(hooks);
    }

    public static void removeHttpHook(HttpHook hook) {
        GLOBAL_HOOKS.remove(hook);
    }

    public static List<HttpHook> getHttpHooks() {
        return Collections.unmodifiableList(GLOBAL_HOOKS);
    }

    public static synchronized void initHttpHookInvokers() {
        if (LOADED_INVOKERS.compareAndSet(false, true)) {
            // load HttpHook
            List<HttpHook> interceptors = EnhancedServiceLoader.loadAll(HttpHook.class);
            HttpBase.addHttpHooks(interceptors);

            // load HttpInvokerHandler
            List<AbstractHttpInvokerHandler> hookInvokers = EnhancedServiceLoader.loadAll(AbstractHttpInvokerHandler.class);

            AbstractHttpInvokerHandler lastInvoker = HEADER_HANDLER;
            for (AbstractHttpInvokerHandler invoker : hookInvokers) {
                lastInvoker.setNext(invoker);
                lastInvoker = invoker;
            }

            lastInvoker.setNext(TAIL_HANDLER);
        }
    }

    public static AbstractHttpInvokerHandler getHeaderHttpHookInvoker() {
        initHttpHookInvokers();
        return HEADER_HANDLER;
    }

}
