package com.cusc.nirvana.common.encrypt.sign;

import com.cusc.nirvana.common.encrypt.Util;
import org.apache.commons.lang3.StringUtils;
import org.bouncycastle.jce.provider.BouncyCastleProvider;

import javax.crypto.KeyGenerator;
import javax.crypto.Mac;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import java.io.UnsupportedEncodingException;
import java.security.*;

/**
 * @author jeff.chen
 * @file HMAC
 * @E-mail chenjf159@chinaunicom.cn
 */
public class HMAC {

    static {
        if (null == Security.getProvider(BouncyCastleProvider.PROVIDER_NAME)) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /**
     * 签名
     *
     * @param content   待签名内容
     * @param key       秘钥
     * @param algorithm 签名算法
     * @return
     */
    public static String sign(String content, String key, Type algorithm) {
        if (StringUtils.isBlank(content)) {
            return null;
        }

        try {
            if (Type.HmacSHA224.equals(algorithm) || Type.HmacSHA384.equals(algorithm)) {
                // 未与在线加密工具对齐信息
                return bouncyCastle(content, key, algorithm);
            } else {
                return jdk(content, key, algorithm);
            }
        } catch (Exception e) {
            throw new RuntimeException("HMAC " + algorithm.algorithm + e.getMessage());
        }
    }

    private static String jdk(String content, String key, Type algorithm) throws NoSuchAlgorithmException, UnsupportedEncodingException, InvalidKeyException {
        Mac mac = Mac.getInstance(algorithm.algorithm);
        mac.init(new SecretKeySpec(key.getBytes("UTF-8"), algorithm.algorithm));
        byte[] bytes = mac.doFinal(content.getBytes("UTF-8"));
        return Util.parseByte2HexStr(bytes);
    }

    private static String bouncyCastle(String content, String key, Type algorithm) throws NoSuchAlgorithmException, UnsupportedEncodingException, InvalidKeyException, NoSuchProviderException {
        KeyGenerator keyGenerator = KeyGenerator.getInstance(algorithm.algorithm, BouncyCastleProvider.PROVIDER_NAME);
        keyGenerator.init(new SecureRandom(key.getBytes("UTF-8")));
        SecretKey secretKey = keyGenerator.generateKey();
        byte[] secret = secretKey.getEncoded();

        SecretKey secretKey0 = new SecretKeySpec(secret, algorithm.algorithm);
        Mac mac = Mac.getInstance(secretKey0.getAlgorithm());
        mac.init(secretKey0);

        byte[] bytes = mac.doFinal(content.getBytes("UTF-8"));
        return Util.parseByte2HexStr(bytes);
    }

    public enum Type {
        /**
         * HmacSHA1
         */
        HmacSHA1("HmacSHA1"),
        /**
         * HmacSHA224
         */
        HmacSHA224("HmacSHA224"),
        /**
         * HmacSHA256
         */
        HmacSHA256("HmacSHA256"),
        /**
         * HmacSHA384
         */
        HmacSHA384("HmacSHA384"),
        /**
         * HmacSHA512
         */
        HmacSHA512("HmacSHA512"),
        /**
         * HmacMD5
         */
        HmacMD5("HmacMD5");

        String algorithm;

        Type(String algorithm) {
            this.algorithm = algorithm;
        }

    }

}
