package com.cusc.nirvana.common.encrypt.ecc;

import org.apache.commons.codec.binary.Base64;
import org.bouncycastle.jce.provider.BouncyCastleProvider;

import javax.crypto.Cipher;
import java.security.*;
import java.security.interfaces.ECPrivateKey;
import java.security.interfaces.ECPublicKey;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;

/**
 * (非对称)
 * JDK ECC 目前只有 公私钥生成、签名验签，但没有实现加密解密
 * ECIES    集成加密方案
 * <p>
 * 非对称，需要调用硬件完成加密/解密（ECC算法相当耗费资源，效率低下）
 *
 * @author jeff.chen
 * @file ECC
 * @E-mail chenjf159@chinaunicom.cn
 */
public class ECIES {

    static {
        if (null == Security.getProvider(BouncyCastleProvider.PROVIDER_NAME)) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /**
     * 初始化密钥对
     *
     * @param keySize 192，224，239，256，384，521
     * @return
     */
    public static KeyPair initKeyPair(int keySize) {
        try {
            KeyPairGenerator keyPairGenerator = KeyPairGenerator.getInstance("EC", BouncyCastleProvider.PROVIDER_NAME);
            keyPairGenerator.initialize(keySize);
            KeyPair keyPair = keyPairGenerator.generateKeyPair();

            return keyPair;
        } catch (Exception ex) {
            throw new RuntimeException("[ECIES initKeyPair]:" + ex.getMessage());
        }
    }

    /**
     * 加密
     *
     * @param content   待加密内容
     * @param publicKey 公钥
     * @return
     */
    public static String encrypt(String content, String publicKey) {
        try {
            byte[] decoded = Base64.decodeBase64(publicKey);
            ECPublicKey pubKey = (ECPublicKey) KeyFactory.getInstance("EC").generatePublic(new X509EncodedKeySpec(decoded));

            Cipher cipher = Cipher.getInstance("ECIES", BouncyCastleProvider.PROVIDER_NAME);
            cipher.init(Cipher.ENCRYPT_MODE, pubKey);

            String outStr = Base64.encodeBase64String(cipher.doFinal(content.getBytes("UTF-8")));

            return outStr;
        } catch (Exception ex) {
            throw new RuntimeException("[ECIES encrypt]: " + ex.getMessage());
        }
    }

    /**
     * 解密
     *
     * @param content    待加密内容
     * @param privateKey 私钥
     * @return
     */
    public static String decrypt(String content, String privateKey) {
        try {
            byte[] decoded = Base64.decodeBase64(privateKey);
            ECPrivateKey priKey = (ECPrivateKey) KeyFactory.getInstance("EC").generatePrivate(new PKCS8EncodedKeySpec(decoded));

            Cipher cipher = Cipher.getInstance("ECIES", BouncyCastleProvider.PROVIDER_NAME);
            cipher.init(Cipher.DECRYPT_MODE, priKey);

            byte[] inputByte = Base64.decodeBase64(content.getBytes("UTF-8"));

            String outStr = new String(cipher.doFinal(inputByte));

            return outStr;
        } catch (Exception ex) {
            throw new RuntimeException("[ECIES decrypt]: " + ex.getMessage());
        }
    }

}
