package com.cusc.nirvana.common.encrypt.ecc;

import com.cusc.nirvana.common.encrypt.Util;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringUtils;
import org.bouncycastle.jce.provider.BouncyCastleProvider;

import java.security.*;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;

/**
 * (非对称)
 * 速度快，强度高，签名短
 * 数字签名
 * ECDSA    数字签名算法
 *
 * @author jeff.chen
 * @file ECDSA
 * @@E-mail chenjf159@chinaunicom.cn
 */
public class ECDSA {

    static {
        if (null == Security.getProvider(BouncyCastleProvider.PROVIDER_NAME)) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /**
     * 初始化密钥对
     *
     * @param keySize 112 ~ 571
     * @return
     */
    public static KeyPair initKeyPair(int keySize) {
        try {
            KeyPairGenerator keyPairGenerator = KeyPairGenerator.getInstance("EC", BouncyCastleProvider.PROVIDER_NAME);
            keyPairGenerator.initialize(keySize);
            KeyPair keyPair = keyPairGenerator.generateKeyPair();

            return keyPair;
        } catch (Exception ex) {
            throw new RuntimeException("ECDSA initKeyPair:" + ex.getMessage());
        }
    }

    /**
     * 签名
     *
     * @param content    待签名内容
     * @param privateKey 秘钥
     * @param algorithm  签名算法
     * @return
     */
    public static String sign(String content, String privateKey, Type algorithm) {
        if (StringUtils.isBlank(content)) {
            return null;
        }

        try {
            PKCS8EncodedKeySpec pkcs8KeySpec = new PKCS8EncodedKeySpec(Base64.decodeBase64(privateKey));
            KeyFactory keyFactory = KeyFactory.getInstance("EC");
            PrivateKey priKey = keyFactory.generatePrivate(pkcs8KeySpec);

            Signature signature = Signature.getInstance(algorithm.algorithm);
            signature.initSign(priKey);
            signature.update(content.getBytes("UTF-8"));
            byte[] signed = signature.sign();

            return Util.parseByte2HexStr(signed);
        } catch (Exception ex) {
            throw new RuntimeException("ECDSA sign:" + ex.getMessage());
        }
    }

    /**
     * 验签
     *
     * @param content   待验签内容
     * @param sign      签名
     * @param publicKey 公钥
     * @param algorithm 验签算法
     * @return
     */
    public static boolean verify(String content, String sign, String publicKey, Type algorithm) {
        if (StringUtils.isBlank(content)) {
            return false;
        }

        try {
            X509EncodedKeySpec x509KeySpec = new X509EncodedKeySpec(Base64.decodeBase64(publicKey));
            KeyFactory keyFactory = KeyFactory.getInstance("EC");
            PublicKey pubKey = keyFactory.generatePublic(x509KeySpec);

            Signature signature = Signature.getInstance(algorithm.algorithm);
            signature.initVerify(pubKey);
            signature.update(content.getBytes("UTF-8"));

            return signature.verify(Util.parseHexStr2Byte(sign));
        } catch (Exception ex) {
            throw new RuntimeException("ECDSA verify:" + ex.getMessage());
        }
    }

    public enum Type {

        // JDK/BC
        NONEwithECDSA("NONEwithECDSA"),

        // BC
        RIPEMD160withECDSA("RIPEMD160withECDSA"),

        // JDK/BC
        SHA1withECDSA("SHA1withECDSA"),

        // BC
        SHA224withECDSA("SHA224withECDSA"),

        // JDK/BC
        SHA384withECDSA("SHA384withECDSA"),

        // JDK/BC
        SHA512withECDSA("SHA512withECDSA");

        String algorithm;

        Type(String algorithm) {
            this.algorithm = algorithm;
        }

    }

}
