package com.cusc.nirvana.common.uid;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicLong;

/**
 * @author jeff.chen
 * @file UniqueID
 * @E-mail chenjf159@chinaunicom.cn
 */
public class UniqueID implements UID {

    /**
     * MAC地址 | IP地址
     */
    final String ADDRESS;

    /**
     * 5位 - 获取当前服务进程号
     */
    final String PROCESS_ID;

    /**
     * 序列-计数器
     */
    final AtomicLong sequenceCounter = new AtomicLong(0);

    /**
     * 序列
     * 容量：33,554,432
     * 0x1FFFFFF = (1 << 25) - 1
     */
    static final int SEQUENCE_BIT_25 = 0x1FFFFFF;

    /**
     * 序列-长度
     */
    static final int SEQUENCE_COUNTER_LEN = 5;

    /**
     * 最后一次生成uid时间
     */
    private long lastTimeOfMillis = System.currentTimeMillis();

    /**
     * 时钟回拨-计数器
     */
    final AtomicLong clockCallbackCounter = new AtomicLong(0);

    /**
     * 时钟回拨序列
     * 容量：32,768
     * 0x7FFF = (1 << 15) - 1
     */
    static final int SEQUENCE_BIT_15 = 0x7FFF;

    /**
     * 时钟回拨序列-长度
     */
    static final int CLOCK_CALLBACK_COUNTER_LEN = 3;

    /**
     * uid 固定长度
     */
    static final int FIXED_LEN = 32;

    private UniqueID(Type type) {
        if (Type.MAC.equals(type)) {
            ADDRESS = UtilTools.getLocalMac();
        } else if (Type.IP.equals(type)) {
            ADDRESS = UtilTools.getLocalIP();
        } else {
            throw new IllegalArgumentException("not support@" + type.name());
        }

        PROCESS_ID = String.format("%05d", UtilTools.getProcessID());
    }

    @Override
    public String ID() {
        long currentTimeOfMillis = currentTimeMillis();

        if (currentTimeOfMillis < lastTimeOfMillis) {
            clockCallbackCounter.incrementAndGet();
        } else if (lastTimeOfMillis < currentTimeOfMillis) {
            lastTimeOfMillis = currentTimeOfMillis;
        }

        return ID0(currentTimeOfMillis);
    }

    /**
     * 10+5+9+5+3=32
     * base:32
     * {mac-address-10 | ip-address-10}{process-id-5}{timeOfMillis-9}{sequence-5}{clock-callback-3}
     *
     * @param currentTimeOfMillis
     * @return {address-10}{process-id-5}{timeOfMillis-9}{sequence-5}{clock-callback-3}
     */
    private String ID0(long currentTimeOfMillis) {
        StringBuilder stringBuilder = new StringBuilder(FIXED_LEN);

        stringBuilder.append(ADDRESS);

        stringBuilder.append(PROCESS_ID);

        stringBuilder.append(UtilTools.base10ToBase32String(currentTimeOfMillis));

        long sequence = sequenceCounter.incrementAndGet() & SEQUENCE_BIT_25;
        stringBuilder.append(UtilTools.appendPrefixWithFixedLength(UtilTools.base10ToBase32String(sequence), SEQUENCE_COUNTER_LEN));

        long clockCallback = clockCallbackCounter.get() & SEQUENCE_BIT_15;
        stringBuilder.append(UtilTools.appendPrefixWithFixedLength(UtilTools.base10ToBase32String(clockCallback), CLOCK_CALLBACK_COUNTER_LEN));

        return stringBuilder.toString();
    }

    public static UID getInstance(Type type) {
        return UniqueIDHolder.get(type);
    }

    static class UniqueIDHolder {

        static final ConcurrentMap<Type, UID> CACHE = new ConcurrentHashMap<>();

        static {
            try {
                CACHE.put(Type.IP, new UniqueID(Type.IP));
            } catch (Exception ex) {
                throw ex;
            }

            try {
                CACHE.put(Type.MAC, new UniqueID(Type.MAC));
            } catch (Exception ex) {
                throw ex;
            }
        }

        public static UID get(Type type) {
            return CACHE.get(type);
        }

    }

    public enum Type {
        /**
         * 基准
         */
        MAC,
        IP;
    }

}
