package com.cusc.nirvana.common.tool;

import io.github.resilience4j.ratelimiter.RateLimiter;
import io.github.resilience4j.ratelimiter.RateLimiterConfig;

import java.time.Duration;

import static java.lang.System.nanoTime;
import static java.lang.Thread.currentThread;
import static java.lang.Thread.sleep;
import static java.util.concurrent.locks.LockSupport.parkNanos;

/**
 * @author jeff.chen
 * @file CustomRateLimiter
 * @E-mail chenjf159@chinaunicom.cn
 */
public class CustomRateLimiter {

    private static final int DEFAULT_RATE = 1000;

    RateLimiter rateLimiter;

    /**
     * default: 1000/s 无等待
     *
     * @param name
     */
    public CustomRateLimiter(String name) {
        rateLimiter = of(name, null);
    }

    /**
     * @param name
     * @param rate 每秒速率，无等待
     */
    public CustomRateLimiter(String name, int rate) {
        rateLimiter = of(name, getRateLimiterConfig(rate, 0));
    }

    /**
     * @param name
     * @param rate    每秒速率
     * @param timeout 等待积压时间(秒)
     */
    public CustomRateLimiter(String name, int rate, long timeout) {
        rateLimiter = of(name, getRateLimiterConfig(rate, timeout));
    }

    public long reservePermission() {
        return rateLimiter.reservePermission();
    }

    public void waitForPermission() {
        long nanosToWait = reservePermission();

        if (nanosToWait > 0) {
            waitForPermission(nanosToWait);
        } else if (nanosToWait < 0) {
            try {
                sleep(10);
            } catch (InterruptedException e) {
            }

            waitForPermission();
        }
    }

    public boolean tryPermission() {
        long nanosToWait = reservePermission();

        if (nanosToWait < 0) {
            return false;
        }

        if (nanosToWait > 0) {
            waitForPermission(nanosToWait);
        }

        return true;
    }

    /**
     * @param limitForPeriod 每秒速率，无等待
     * @return
     */
    private RateLimiterConfig getRateLimiterConfig(int limitForPeriod, long timeout) {
        return RateLimiterConfig.custom()
                // limitForPeriod={default: 1000}
                .limitForPeriod(limitForPeriod)
                // limitRefreshPeriod={default:1s}
                .limitRefreshPeriod(Duration.ofSeconds(1L))
                // timeoutDuration={default:0ms}
                .timeoutDuration(Duration.ofSeconds(timeout))
                .build();
    }

    private io.github.resilience4j.ratelimiter.RateLimiter of(String name, RateLimiterConfig rateLimiterConfig) {
        if (null == rateLimiterConfig) {
            return io.github.resilience4j.ratelimiter.RateLimiter.of(name, getRateLimiterConfig(DEFAULT_RATE, 0));
        } else {
            return io.github.resilience4j.ratelimiter.RateLimiter.of(name, rateLimiterConfig);
        }
    }

    private static final long NANO_TIME_START = nanoTime();

    private static boolean waitForPermission(final long nanosToWait) {
        long deadline = currentNanoTime() + nanosToWait;
        boolean wasInterrupted = false;
        while (currentNanoTime() < deadline && !wasInterrupted) {
            long sleepBlockDuration = deadline - currentNanoTime();
            parkNanos(sleepBlockDuration);
            wasInterrupted = Thread.interrupted();
        }

        if (wasInterrupted) {
            currentThread().interrupt();
        }

        return !wasInterrupted;
    }

    private static long currentNanoTime() {
        return nanoTime() - NANO_TIME_START;
    }

}
