package com.cusc.nirvana.common.encrypt.sign;

import com.cusc.nirvana.common.encrypt.Util;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringUtils;
import org.bouncycastle.jce.provider.BouncyCastleProvider;

import java.security.*;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;

/**
 * (非对称)
 * DSA只是一种算法，和RSA不同之处在于它不能用作加密和解密，也不能进行密钥交换，只用于签名
 * 一种更高级的验证方式，用作数字签名。不单单只有公钥、私钥，还有数字签名。私钥加密生成数字签名，公钥验证数据及签名
 * <p>
 * 实现方JDK：
 * SHA1withDSA
 * <p>
 * 实现方BD：
 * SHA224withDSA
 * SHA256withDSA
 * SHA384withDSA
 * SHA512withDSA
 *
 * @author jeff.chen
 * @file DSC
 * @E-mail chenjf159@chinaunicom.cn
 */
public class DSA {

    static {
        if (null == Security.getProvider(BouncyCastleProvider.PROVIDER_NAME)) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /**
     * 初始化密钥对
     *
     * @param keySize 1) ((512 <= keySize <=1024) && (var0 % 64 == 0)); 2) keySize=2048; 3) keySize=3072
     * @return
     */
    public static KeyPair initKeyPair(int keySize) {
        try {
            KeyPairGenerator keyPairGenerator = KeyPairGenerator.getInstance("DSA", BouncyCastleProvider.PROVIDER_NAME);
            keyPairGenerator.initialize(keySize);
            KeyPair keyPair = keyPairGenerator.generateKeyPair();

            return keyPair;
        } catch (Exception ex) {
            throw new RuntimeException("[DSA init keyPair]:" + ex.getMessage());
        }
    }

    /**
     * 签名
     *
     * @param content    待签名内容
     * @param privateKey 秘钥
     * @return
     */
    public static String sign(String content, String privateKey) {
        if (StringUtils.isBlank(content)) {
            return null;
        }

        try {
            PKCS8EncodedKeySpec pkcs8KeySpec = new PKCS8EncodedKeySpec(Base64.decodeBase64(privateKey));
            KeyFactory keyFactory = KeyFactory.getInstance("DSA");
            PrivateKey priKey = keyFactory.generatePrivate(pkcs8KeySpec);

            Signature signature = Signature.getInstance(keyFactory.getAlgorithm());
            signature.initSign(priKey);
            signature.update(content.getBytes("UTF-8"));
            byte[] signed = signature.sign();

            return Util.parseByte2HexStr(signed);
        } catch (Exception ex) {
            throw new RuntimeException("DSA sign: " + ex.getMessage());
        }
    }

    /**
     * 验签
     *
     * @param content   待验签内容
     * @param sign      签名
     * @param publicKey 公钥
     * @return
     */
    public static boolean verify(String content, String sign, String publicKey) {
        if (StringUtils.isBlank(content)) {
            return false;
        }

        try {
            X509EncodedKeySpec x509KeySpec = new X509EncodedKeySpec(Base64.decodeBase64(publicKey));
            KeyFactory keyFactory = KeyFactory.getInstance("DSA");
            PublicKey pubKey = keyFactory.generatePublic(x509KeySpec);

            Signature signature = Signature.getInstance(keyFactory.getAlgorithm());
            signature.initVerify(pubKey);
            signature.update(content.getBytes("UTF-8"));

            return signature.verify(Util.parseHexStr2Byte(sign));
        } catch (Exception ex) {
            throw new RuntimeException("DSA verify: " + ex.getMessage());
        }
    }

}
