package com.cusc.nirvana.common.encrypt;

import org.apache.commons.codec.binary.Base64;

import javax.crypto.Cipher;
import java.io.UnsupportedEncodingException;
import java.security.*;
import java.security.interfaces.RSAPrivateKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;

/**
 * (非对称)
 *
 * @author jeff.chen
 * @file RSA
 * @E-mail chenjf159@chinaunicom.cn
 */
public class RSA {

    public static final String SIGN_ALGORITHMS = "SHA1WithRSA";

    public static final String UTF8 = "UTF-8";

    /**
     * RSA签名(私钥签名)
     *
     * @param content    待签名数据
     * @param privateKey RSA私钥
     * @return 签名值
     */
    public static String sign(String content, String privateKey) {
        return sign(content, privateKey, UTF8);
    }

    /**
     * RSA签名(私钥签名)
     *
     * @param content      待签名数据
     * @param privateKey   RSA私钥
     * @param inputCharset 编码格式
     * @return 签名值
     */
    public static String sign(String content, String privateKey, String inputCharset) {
        try {
            PKCS8EncodedKeySpec priPKCS8 = new PKCS8EncodedKeySpec(Base64.decodeBase64(privateKey));
            KeyFactory keyFactory = KeyFactory.getInstance("RSA");
            PrivateKey priKey = keyFactory.generatePrivate(priPKCS8);

            Signature signature = Signature.getInstance(SIGN_ALGORITHMS);

            signature.initSign(priKey);
            signature.update(content.getBytes(inputCharset));

            byte[] signed = signature.sign();

            return Base64.encodeBase64String(signed);
        } catch (Exception e) {
            throw new RuntimeException("RSA sign:" + e.getMessage());
        }
    }

    /**
     * RSA验签名检查(公钥验签)
     *
     * @param content   待签名数据
     * @param sign      签名值
     * @param publicKey RSA公钥
     * @return 布尔值
     */
    public static boolean verify(String content, String sign, String publicKey) {
        return verify(content, sign, publicKey, UTF8);
    }

    /**
     * RSA验签名检查(公钥验签)
     *
     * @param content      待签名数据
     * @param sign         签名值
     * @param publicKey    RSA公钥
     * @param inputCharset 编码格式
     * @return 布尔值
     */
    public static boolean verify(String content, String sign, String publicKey, String inputCharset) {
        try {
            byte[] decoded = Base64.decodeBase64(publicKey);
            RSAPublicKey pubKey = (RSAPublicKey) KeyFactory.getInstance("RSA").generatePublic(new X509EncodedKeySpec(decoded));

            Signature signature = Signature.getInstance(SIGN_ALGORITHMS);

            signature.initVerify(pubKey);
            signature.update(content.getBytes(inputCharset));

            return signature.verify(Base64.decodeBase64(sign));
        } catch (Exception e) {
            throw new RuntimeException("RSA verify:" + e.getMessage());
        }
    }

    /**
     * RSA公钥加密
     *
     * @param content
     * @param publicKey
     * @return
     * @throws Exception
     */
    public static String encryptByPublicKey(String content, String publicKey) {
        return encryptByPublicKey(content, publicKey, UTF8);
    }

    /**
     * RSA公钥加密
     *
     * @param content
     * @param publicKey
     * @param inputCharset
     * @return
     * @throws Exception
     */
    public static String encryptByPublicKey(String content, String publicKey, String inputCharset) {
        try {
            byte[] decoded = Base64.decodeBase64(publicKey);
            RSAPublicKey pubKey = (RSAPublicKey) KeyFactory.getInstance("RSA").generatePublic(new X509EncodedKeySpec(decoded));

            // RSA加密
            Cipher cipher = Cipher.getInstance("RSA");
            cipher.init(Cipher.ENCRYPT_MODE, pubKey);

            String outStr = Base64.encodeBase64String(cipher.doFinal(content.getBytes(inputCharset)));

            return outStr;
        } catch (Exception ex) {
            throw new RuntimeException("[RSA encrypt1] " + ex.getMessage());
        }
    }

    /**
     * RSA私钥加密
     *
     * @param content
     * @param privateKey
     * @return
     * @throws Exception
     */
    public static String encryptByPrivateKey(String content, String privateKey) {
        return encryptByPrivateKey(content, privateKey, UTF8);
    }

    /**
     * RSA私钥加密
     *
     * @param content
     * @param privateKey
     * @param inputCharset
     * @return
     * @throws Exception
     */
    public static String encryptByPrivateKey(String content, String privateKey, String inputCharset) {
        try {
            byte[] decoded = Base64.decodeBase64(privateKey);

            RSAPrivateKey priKey = (RSAPrivateKey) KeyFactory.getInstance("RSA").generatePrivate(new PKCS8EncodedKeySpec(decoded));

            Cipher cipher = Cipher.getInstance("RSA");
            cipher.init(Cipher.ENCRYPT_MODE, priKey);

            String outStr = Base64.encodeBase64String(cipher.doFinal(content.getBytes(inputCharset)));

            return outStr;
        } catch (Exception ex) {
            throw new RuntimeException("[RSA decrypt2] " + ex.getMessage());
        }
    }

    /**
     * RSA私钥解密
     *
     * @param content
     * @param privateKey
     * @return
     * @throws Exception
     */
    public static String decryptByPrivateKey(String content, String privateKey) {
        return decryptByPrivateKey(content, privateKey, UTF8);
    }

    /**
     * RSA私钥解密
     *
     * @param content
     * @param privateKey
     * @param inputCharset
     * @return
     * @throws Exception
     */
    public static String decryptByPrivateKey(String content, String privateKey, String inputCharset) {
        try {
            byte[] decoded = Base64.decodeBase64(privateKey);

            RSAPrivateKey priKey = (RSAPrivateKey) KeyFactory.getInstance("RSA").generatePrivate(new PKCS8EncodedKeySpec(decoded));

            Cipher cipher = Cipher.getInstance("RSA");
            cipher.init(Cipher.DECRYPT_MODE, priKey);

            byte[] inputByte = Base64.decodeBase64(content.getBytes(inputCharset));

            String outStr = new String(cipher.doFinal(inputByte));

            return outStr;
        } catch (Exception ex) {
            throw new RuntimeException("[RSA decrypt1] " + ex.getMessage());
        }
    }

    /**
     * RSA公钥解密
     *
     * @param content
     * @param publicKey
     * @return
     * @throws Exception
     */
    public static String decryptByPublicKey(String content, String publicKey) {
        return decryptByPublicKey(content, publicKey, UTF8);
    }

    /**
     * RSA公钥解密
     *
     * @param content
     * @param publicKey
     * @param inputCharset
     * @return
     * @throws Exception
     */
    public static String decryptByPublicKey(String content, String publicKey, String inputCharset) {
        try {
            byte[] decoded = Base64.decodeBase64(publicKey);
            RSAPublicKey pubKey = (RSAPublicKey) KeyFactory.getInstance("RSA").generatePublic(new X509EncodedKeySpec(decoded));

            // RSA加密
            Cipher cipher = Cipher.getInstance("RSA");
            cipher.init(Cipher.DECRYPT_MODE, pubKey);

            byte[] inputByte = Base64.decodeBase64(content.getBytes(inputCharset));

            String outStr = new String(cipher.doFinal(inputByte));

            return outStr;
        } catch (Exception ex) {
            throw new RuntimeException("[RSA encrypt2] " + ex.getMessage());
        }
    }

    /**
     * 生产RSA钥匙对
     *
     * @param keySize 1024 | 2048
     * @throws NoSuchAlgorithmException
     * @throws UnsupportedEncodingException
     */
    public static KeyPair generateKeyPair(int keySize) throws NoSuchAlgorithmException {
        // RSA加密算法
        KeyPairGenerator keyPairGenerator = KeyPairGenerator.getInstance("RSA");

        keyPairGenerator.initialize(keySize);
        KeyPair keyPair = keyPairGenerator.generateKeyPair();

        return keyPair;
    }

}
