package com.cusc.nirvana.common.encrypt;

import org.apache.commons.codec.binary.Base64;

import javax.crypto.KeyAgreement;
import javax.crypto.SecretKey;
import javax.crypto.interfaces.DHPublicKey;
import javax.crypto.spec.DHParameterSpec;
import java.io.UnsupportedEncodingException;
import java.security.*;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;

/**
 * (非对称)
 * 密钥交换
 * <p>
 * 注意
 * 钥所用的算法不被支持，这个是由于JDK8 update 161之后，DH的密钥长度至少为512位，但AES算法密钥不能达到这样的长度，长度不一致所以导致报错。
 * 解决的方法：
 * 将 -Djdk.crypto.KeyAgreement.legacyKDF=true 写入JVM系统变量中
 * <p>
 * https://www.oracle.com/technetwork/java/javase/8u161-relnotes-4021379.html
 *
 * @author jeff.chen
 * @file DH
 * @E-mail chenjf159@chinaunicom.cn
 */
public class DH {

    /**
     * 1.初始化发送者密钥
     *
     * @param keySize 1024 | 2048
     * @return 公私钥对
     * @throws NoSuchAlgorithmException
     * @throws UnsupportedEncodingException
     */
    public static KeyPair initSenderKeyPair(int keySize) throws NoSuchAlgorithmException {
        KeyPairGenerator keyPairGenerator = KeyPairGenerator.getInstance("DH");
        keyPairGenerator.initialize(keySize);
        KeyPair keyPair = keyPairGenerator.generateKeyPair();

        return keyPair;
    }

    /**
     * 2.初始化接收者密钥
     *
     * @param senderPublicKey
     * @return 公私钥对
     * @throws NoSuchAlgorithmException
     * @throws InvalidKeySpecException
     * @throws InvalidAlgorithmParameterException
     * @throws InvalidKeyException
     */
    public static KeyPair initReceiverKeyPair(String senderPublicKey) throws NoSuchAlgorithmException, InvalidKeySpecException, InvalidAlgorithmParameterException, InvalidKeyException {
        KeyFactory keyFactory = KeyFactory.getInstance("DH");
        X509EncodedKeySpec x509EncodedKeySpec = new X509EncodedKeySpec(Base64.decodeBase64(senderPublicKey));
        PublicKey initPublicKey = keyFactory.generatePublic(x509EncodedKeySpec);
        DHParameterSpec dhParameterSpec = ((DHPublicKey) initPublicKey).getParams();

        KeyPairGenerator keyPairGenerator = KeyPairGenerator.getInstance("DH");
        keyPairGenerator.initialize(dhParameterSpec);
        KeyPair keyPair = keyPairGenerator.generateKeyPair();

        return keyPair;
    }

    /**
     * 3.密钥构建
     *
     * @param publicKey  对方公钥
     * @param privateKey 自己私有秘钥
     * @param publicKey
     * @param privateKey
     * @return
     * @throws NoSuchAlgorithmException
     * @throws InvalidKeyException
     */
    public static SecretKey initSecretKey(PublicKey publicKey, PrivateKey privateKey, Type type) throws NoSuchAlgorithmException, InvalidKeyException {
        KeyAgreement keyAgreement = KeyAgreement.getInstance("DH");
        keyAgreement.init(privateKey);
        keyAgreement.doPhase(publicKey, true);

        return keyAgreement.generateSecret(type.algorithm);
    }

    /**
     * 3.密钥构建
     *
     * @param publicKeyStr  对方公钥
     * @param privateKeyStr 自己私有秘钥
     * @return
     * @throws NoSuchAlgorithmException
     * @throws InvalidKeyException
     * @throws InvalidKeySpecException
     */
    public static SecretKey initSecretKey(String publicKeyStr, String privateKeyStr, Type type) throws NoSuchAlgorithmException, InvalidKeyException, InvalidKeySpecException {
        KeyFactory keyFactory = KeyFactory.getInstance("DH");

        X509EncodedKeySpec x509EncodedKeySpec = new X509EncodedKeySpec(Base64.decodeBase64(publicKeyStr));
        PublicKey publicKey = keyFactory.generatePublic(x509EncodedKeySpec);

        PKCS8EncodedKeySpec pKCS8EncodedKeySpec = new PKCS8EncodedKeySpec(Base64.decodeBase64(privateKeyStr));
        PrivateKey privateKey = keyFactory.generatePrivate(pKCS8EncodedKeySpec);

        return initSecretKey(publicKey, privateKey, type);
    }

    public enum Type {
        /**
         * DES
         */
        DES("DES"),
        /**
         * AES
         */
        AES("AES");

        String algorithm;

        Type(String algorithm) {
            this.algorithm = algorithm;
        }

    }

}
