package com.cusc.component.uid.util;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.PostConstruct;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.cache.CacheFactory;
import com.cache.redis.config.CacheConfig;
import com.cusc.component.uid.config.RedisConfiguration;

/**
 * @Description:
 * @author xu_fxiang@163.com
 * @date 2019年12月4日 下午8:38:59
 */

@Component
public class RedisCenter {

	private static Logger logger = LoggerFactory.getLogger(RedisCenter.class);
	private static final String key = "uid-service";

	@Autowired
	private RedisConfiguration redisConfig;

	private static CacheFactory leafCacheFactory;

	private static CacheConfig config;

	public RedisCenter(final RedisConfiguration redisConfiguration) {
		this.redisConfig = redisConfiguration;
	}

	@PostConstruct
	public void initconfig() {
		if (redisConfig.getType() == 1) {
			config = new CacheConfig();
			config.setKey(key);
			config.setHostName(redisConfig.getHostname());
			config.setPort(redisConfig.getPort());
			config.setDatabase(redisConfig.getDatabase());
			config.setPassword(redisConfig.getPassword());
			config.setType(redisConfig.getType());
			config.setTimeout(redisConfig.getTimeout());
			config.setMaxIdle(redisConfig.getMaxidle());
			config.setMinIdle(redisConfig.getMinidle());
			config.setMaxWaitMillis(redisConfig.getMaxwait());
			config.setMaxActive(redisConfig.getMaxactive());
		} else if (redisConfig.getType() == 2) {
			config = new CacheConfig();
			config.setKey(key);
			config.setPassword(redisConfig.getPassword());
			config.setType(redisConfig.getType());
			config.setSentinelMasterName(redisConfig.getSentinelMaster());
			config.setSentinelSet(RedisCenter.getSetNodes(redisConfig.getSentinelNodes()));
			config.setTimeout(redisConfig.getTimeout());
			config.setMaxIdle(redisConfig.getMaxidle());
			config.setMinIdle(redisConfig.getMinidle());
			config.setMaxWaitMillis(redisConfig.getMaxwait());
			config.setMaxActive(redisConfig.getMaxactive());
		} else {
			throw new RuntimeException("type是不支持的集群类别！请联系技术中台增加版本支持！");
		}
		init();
	}

	private static Set<String> getSetNodes(String node) {
		if (StringUtils.isBlank(node)) {
			return Collections.emptySet();
		}

		return new HashSet<>(getStringSet(node));
	}

	private static List<String> getStringSet(String v) {
		if (StringUtils.isBlank(v)) {
			return Collections.emptyList();
		}
		return Arrays.asList(v.trim().split(","));
	}

	private void initStart() {
		if (null == leafCacheFactory) {
			init();
		}
	}

	private void init() {
		if (null != config) {
			if (null != leafCacheFactory) {
				synchronized (leafCacheFactory) {// 热更新
					leafCacheFactory = new CacheFactory(config);
				}
			} else {
				leafCacheFactory = new CacheFactory(config);
			}
			logger.info("T3Cache : Configuration injection succeeded ... host is :{} ", config.getHostName());
		}
	}

	/**
	 * add or update
	 * 
	 * @param key
	 * @param field
	 * @param value
	 * @return true or false
	 */
	public boolean hset(String key, String field, String value) {
		try {
			initStart();
			leafCacheFactory.getInternalService().hset(key, field, value, -1);
			return true;
		} catch (Exception e) {
			logger.error("hset failed,key:{}, field:{}, value:{}, exception:{}", key, field, value, e);
			return false;
		}
	}

	/**
	 * get one
	 * 
	 * @param key  not null
	 * @param item not null
	 * @return
	 */
	public String hget(String key, String item) {
		try {
			initStart();
			return leafCacheFactory.getInternalService().hget(key, item);
		} catch (Exception e) {
			logger.error("hget failed,key:{}, item:{}, exception:{}", key, item, e);
		}
		return null;
	}

	/**
	 * add one
	 * 
	 * @param key
	 * @return
	 */
	public Long incr(String key) {
		try {
			initStart();
			return leafCacheFactory.getInternalService().incr(key, 1, -1);
		} catch (Exception e) {
			logger.error("incr failed,key:{}, exception:{}", key, e);
		}
		return 0L;
	}

	public boolean lock(String lockKey, String requestId, Integer expireTime) {
		try {
			initStart();
			return leafCacheFactory.getInternalService().lock(lockKey, requestId, expireTime);
		} catch (Exception e) {
		}
		return false;
	}

	public boolean unlock(String lockKey, String requestId) {
		try {
			initStart();
			return leafCacheFactory.getInternalService().unLock(lockKey, requestId);
		} catch (Exception e) {
		}
		return false;
	}

	public List<String> queryKey(String keys) {
		try {
			initStart();
			return new ArrayList<String>(leafCacheFactory.getInternalService().hkeys(keys));
		} catch (Exception e) {

		}
		return null;
	}

	public boolean dislock(String key, String value, Integer expire, Integer waitTime, Integer reCount) {
		try {
			initStart();
			return leafCacheFactory.getInternalService().lockAndWait(key, value, expire, waitTime, reCount);
		} catch (Exception e) {
			return false;
		}
	}

	public void restart() {
		initconfig();
	}
}
