package com.cusc.component.uid.service.impl;

import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

import org.apache.commons.lang3.StringUtils;
import org.perf4j.StopWatch;
import org.perf4j.slf4j.Slf4JStopWatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.cusc.component.uid.constant.FormatConstant;
import com.cusc.component.uid.core.LeafAlloc;
import com.cusc.component.uid.core.Result;
import com.cusc.component.uid.core.ResultCode;
import com.cusc.component.uid.core.ResultGenerator;
import com.cusc.component.uid.core.Segment;
import com.cusc.component.uid.core.SegmentBuffer;
import com.cusc.component.uid.core.UidConstants;
import com.cusc.component.uid.dao.IDAllocDao;
import com.cusc.component.uid.service.IDGen;
import com.cusc.component.uid.util.DateUtils;

public class SegmentIDGenImpl implements IDGen {

	private static final Logger logger = LoggerFactory.getLogger(SegmentIDGenImpl.class);

	/**
	 * 最大步长不超过100,0000
	 */
	private static final int MAX_STEP = 1000000;
	/**
	 * 一个Segment维持时间为15分钟
	 */
	private static final long SEGMENT_DURATION = 15 * 60 * 1000L;
	private ExecutorService service = new ThreadPoolExecutor(10, Integer.MAX_VALUE, 60L, TimeUnit.SECONDS,
			new SynchronousQueue<Runnable>(), new UpdateThreadFactory());
	private volatile boolean initOK = false;
	private Map<String, SegmentBuffer> cache = new ConcurrentHashMap<String, SegmentBuffer>();

	private IDAllocDao dao;

	private String aim_key;

	public static class UpdateThreadFactory implements ThreadFactory {

		private static int threadInitNumber = 0;

		private static synchronized int nextThreadNum() {
			return threadInitNumber++;
		}

		@Override
		public Thread newThread(Runnable r) {
			return new Thread(r, "Thread-Segment-Update-" + nextThreadNum());
		}
	}

	@Override
	public boolean init() {
		logger.info("Init ...");
		// 确保加载到kv后才初始化成功
		updateCacheFromDb();
		initOK = true;
		updateCacheFromDbAtEveryMinute();
		return initOK;
	}

	private void updateCacheFromDbAtEveryMinute() {
		ScheduledExecutorService service = Executors.newSingleThreadScheduledExecutor(new ThreadFactory() {
			@Override
			public Thread newThread(Runnable r) {
				Thread t = new Thread(r);
				t.setName("check-idCache-thread");
				t.setDaemon(true);
				return t;
			}
		});
		service.scheduleWithFixedDelay(new Runnable() {
			@Override
			public void run() {
				updateCacheFromDb();
			}
		}, 60, 60, TimeUnit.SECONDS);
	}

	private void updateCacheFromDb() {
		logger.info("update cache from db");
		StopWatch sw = new Slf4JStopWatch();
		try {
			List<String> dbTags = dao.getAllKeys();
			if (dbTags == null || dbTags.isEmpty()) {
				return;
			}
			List<String> cacheTags = new ArrayList<String>(cache.keySet());
			List<String> insertTags = new ArrayList<String>(dbTags);
			List<String> removeTags = new ArrayList<String>(cacheTags);
			// db中新加的tags灌进cache
			insertTags.removeAll(cacheTags);
			for (String tag : insertTags) {
				SegmentBuffer buffer = new SegmentBuffer();
				buffer.setKey(tag);
				Segment segment = buffer.getCurrent();
				segment.setValue(new AtomicLong(0));
				segment.setMax(0);
				segment.setStep(0);
				cache.put(tag, buffer);
				logger.info("Add tag {} from db to IdCache, SegmentBuffer {}", tag, buffer);
			}
			// cache中已失效的tags从cache删除
			removeTags.removeAll(dbTags);
			for (String tag : removeTags) {
				cache.remove(tag);
				logger.info("Remove tag {} from IdCache", tag);
			}
		} catch (Exception e) {
			logger.warn("update cache from db exception", e);
		} finally {
			sw.stop("updateCacheFromDb");
		}
	}

	@Override
	public Result get(final String key) {
		if (!initOK) {
			return ResultGenerator.genFailResult(ResultCode.ID_IDCACHE_INIT_FALSE, "IDCache未初始化成功!");
		}
		if (cache.containsKey(key)) {
			SegmentBuffer buffer = cache.get(key);
			if (!buffer.isInitOk()) {
				synchronized (buffer) {
					if (!buffer.isInitOk()) {
						try {
							updateSegmentFromDb(key, buffer.getCurrent());
							logger.info("Init buffer. Update leafkey {} {} from db", key, buffer.getCurrent());
							buffer.setInitOk(true);
						} catch (Exception e) {
							logger.warn("Init buffer {} exception", buffer.getCurrent(), e);
						}
					}
				}
			}
			return getIdFromSegmentBuffer(cache.get(key));
		}
		return ResultGenerator.genFailResult(ResultCode.ID_KEY_NOT_EXISTS, "key不存在，联系中间件申请!");
	}

	public void updateSegmentFromDb(String key, Segment segment) {
		StopWatch sw = new Slf4JStopWatch();
		SegmentBuffer buffer = segment.getBuffer();
		LeafAlloc leafAlloc;
		if (!buffer.isInitOk()) {
			leafAlloc = dao.updateMaxIdAndGet(key);
			buffer.setStep(leafAlloc.getStep());
			buffer.setMinStep(leafAlloc.getStep());// leafAlloc中的step为DB中的step
		} else if (buffer.getUpdateTimestamp() == 0) {
			leafAlloc = dao.updateMaxIdAndGet(key);
			buffer.setUpdateTimestamp(System.currentTimeMillis());
			buffer.setMinStep(leafAlloc.getStep());// leafAlloc中的step为DB中的step
		} else {
			long duration = System.currentTimeMillis() - buffer.getUpdateTimestamp();
			int nextStep = buffer.getStep();
			if (duration < SEGMENT_DURATION) {
				if (nextStep * 2 > MAX_STEP) {
					// do nothing
				} else {
					nextStep = nextStep * 2;
				}
			} else if (duration < SEGMENT_DURATION * 2) {
				// do nothing with nextStep
			} else {
				nextStep = nextStep / 2 >= buffer.getMinStep() ? nextStep / 2 : nextStep;
			}
			logger.info("leafKey[{}], step[{}], duration[{}mins], nextStep[{}]", key, buffer.getStep(),
					String.format("%.2f", ((double) duration / (1000 * 60))), nextStep);
			LeafAlloc temp = new LeafAlloc();
			temp.setKey(key);
			temp.setStep(nextStep);
			leafAlloc = dao.updateMaxIdByCustomStep(temp);
			buffer.setUpdateTimestamp(System.currentTimeMillis());
			buffer.setStep(nextStep);
			buffer.setMinStep(leafAlloc.getStep());// leafAlloc的step为DB中的step
		}
		// must set value before set max
		long value = leafAlloc.getMaxId() - buffer.getStep();
		segment.getValue().set(value);
		segment.setMax(leafAlloc.getMaxId());
		segment.setStep(buffer.getStep());
		sw.stop("updateSegmentFromDb", key + " " + segment);
	}

	public Result getIdFromSegmentBuffer(final SegmentBuffer buffer) {
		while (true) {
			try {
				buffer.rLock().lock();
				final Segment segment = buffer.getCurrent();
				if (!buffer.isNextReady() && (segment.getIdle() < 0.9 * segment.getStep())
						&& buffer.getThreadRunning().compareAndSet(false, true)) {
					service.execute(new Runnable() {
						@Override
						public void run() {
							Segment next = buffer.getSegments()[buffer.nextPos()];
							boolean updateOk = false;
							try {
								updateSegmentFromDb(buffer.getKey(), next);
								updateOk = true;
								logger.info("update segment {} from db {}", buffer.getKey(), next);
							} catch (Exception e) {
								logger.warn(buffer.getKey() + " updateSegmentFromDb exception", e);
							} finally {
								if (updateOk) {
									buffer.wLock().lock();
									buffer.setNextReady(true);
									buffer.getThreadRunning().set(false);
									buffer.wLock().unlock();
								} else {
									buffer.getThreadRunning().set(false);
								}
							}
						}
					});
				}
				long value = segment.getValue().getAndIncrement();
				if (value < segment.getMax()) {
					return ResultGenerator.genSuccessResult(value);
				}
			} finally {
				buffer.rLock().unlock();
			}
			waitAndSleep(buffer);
			try {
				buffer.wLock().lock();
				final Segment segment = buffer.getCurrent();
				long value = segment.getValue().getAndIncrement();
				if (value < segment.getMax()) {
					return ResultGenerator.genSuccessResult(value);
				}
				if (buffer.isNextReady()) {
					buffer.switchPos();
					buffer.setNextReady(false);
				} else {
					logger.error("Both two segments in {} are not ready!", buffer);
					return ResultGenerator.genFailResult(ResultCode.ID_TWO_SEGMENTS_ARE_NULL,
							"SegmentBuffer中的两个Segment均未从DB中装载");
				}
			} finally {
				buffer.wLock().unlock();
			}
		}
	}

	@SuppressWarnings("static-access")
	private void waitAndSleep(SegmentBuffer buffer) {
		int roll = 0;
		while (buffer.getThreadRunning().get()) {
			roll += 1;
			if (roll > 10000) {
				try {
					Thread.currentThread().sleep(500);
					break;
				} catch (InterruptedException e) {
					logger.warn("Thread {} Interrupted", Thread.currentThread().getName());
					Thread.currentThread().interrupt();
					break;
				}
			}
		}
	}

	public Map<String, SegmentBuffer> getCache() {
		return cache;
	}

	public IDAllocDao getDao() {
		return dao;
	}

	public void setDao(IDAllocDao dao) {
		this.dao = dao;
	}

	@Override
	public Result getContinu(final String key) {
		if (!initOK) {
			return ResultGenerator.genFailResult(ResultCode.ID_IDCACHE_INIT_FALSE, "IDCache未初始化成功!");
		}
		if (cache.containsKey(key)) {
			setAimKey(key);
			Long id = dao.incrKey(aim_key);
			// return id !=
			// null?ResultGenerator.genSuccessResult(id):ResultGenerator.genFailResult(null);
			if (id >= 1) {
				return ResultGenerator.genSuccessResult(id);
			} else {
				return ResultGenerator.genFailResult(ResultCode.FAIL, "获取失败，值小于1!");
			}
		}
		return ResultGenerator.genFailResult(ResultCode.ID_KEY_NOT_EXISTS, "key不存在，联系中间件申请!");
	}

	private void setAimKey(String key) {
		if (null == aim_key || !aim_key.split(":")[2].equals(key)) {// 为空时/未包含key
			StringBuffer keys = new StringBuffer(UidConstants.UID_KEY_REALTIME).append(key);
			this.aim_key = keys.toString();
		}
	}

	@Override
	public Result getDayContinueCode(String domainKey, String prefixCode, String dayFormat, String codeFormat) {
		if (!initOK) {
			return ResultGenerator.genFailResult(ResultCode.ID_IDCACHE_INIT_FALSE, "IDCache未初始化成功!");
		}

		if (cache.containsKey(domainKey)) {
			String dateTime = DateUtils.dateTime();
			if (StringUtils.isNotBlank(dayFormat)) {
				dateTime = DateUtils.dateTimeNow(dayFormat);
			}
			String key = domainKey + ":" + dateTime;
			setAimKey(key);
			Long id = dao.incrKey(aim_key);

			if (id >= 1) {
				String codeInitial = FormatConstant.CODE_FORMAT;
				if (StringUtils.isNotBlank(codeFormat)) {
					codeInitial = codeFormat;
				}

				DecimalFormat df = new DecimalFormat(codeInitial);
				String stringValue = df.format(id);

				StringBuffer sb = new StringBuffer();
				if (StringUtils.isNotBlank(prefixCode)) {
					sb.append(prefixCode);
				}
				sb.append(dateTime);
				sb.append(stringValue);
				return ResultGenerator.genSuccessResult(sb.toString());
			} else {
				return ResultGenerator.genFailResult(ResultCode.FAIL, "获取失败，值小于1!");
			}
		}
		return ResultGenerator.genFailResult(ResultCode.ID_KEY_NOT_EXISTS, "key不存在，联系中间件申请!");
	}
}
