/*
 * Copyright (c) 2019 Baidu.com, Inc. All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 */
package com.baidubce.services.dcc.model;

import com.baidubce.auth.BceCredentials;
import com.baidubce.model.AbstractBceRequest;
import com.baidubce.services.bcc.model.Billing;
import com.fasterxml.jackson.annotation.JsonIgnore;

/**
 * The request class used for creating dcc.
 */
public class CreateDccRequest extends AbstractBceRequest {

    /**
     * An ASCII string whose length is less than 64.
     *
     * The request will be idempotent if clientToken is provided.
     * If the clientToken is not specified by the user, a random String generated by default algorithm will be used.
     * See more detail at
     * <a href = "https://bce.baidu.com/doc/BCC/API.html#.E5.B9.82.E7.AD.89.E6.80.A7">
     *     BCE API doc</a>
     */
    @JsonIgnore
    private String clientToken;

    /**
     * Dcc supports the following two types of cloud disk.
     */
    public static final String CLOUD_DISK_TYPE_HP1 = "hp1";
    public static final String CLOUD_DISK_TYPE_CLOUD_HP1 = "cloud_hp1";

    /**
     * Dcc supports the following two types of auto renew time unit.
     */
    public static final String AUTO_RENEW_TIME_UNIT_MONTH = "month";
    public static final String AUTO_RENEW_TIME_UNIT_YEAR = "year";

    /**
     * dcc flavor used to create dcc.
     */
    private String flavor;

    /**
     * specifying the virtual cpu count of the dcc
     */
    private int cpuCount;

    /**
     * The number of dcc to be created.
     */
    private int purchaseCount;

    /**
     * The type of the dcc: calculation or storage.
     */
    private String type;

    /**
     * specifying which zone the dcc should be created in.
     */
    private String zoneName;

    /**
     * specifying auto renew time length.
     */
    private int autoRenewTime;

    /**
     * specifying auto renew time unit: month or year
     */
    private String autoRenewTimeUnit;

    /**
     * specifying the size of the system cloud disk in Gb.
     */
    private int cloudDiskSize;

    /**
     * specifying the type of the system cloud disk.
     */
    private String cloudDiskStorageType = "hp1";

    /**
     * The detail model to specify the billing info.
     */
    private Billing billing;

    public String getClientToken() {
        return clientToken;
    }

    public void setClientToken(String clientToken) {
        this.clientToken = clientToken;
    }

    /**
     * check whether cloud disk storage type is correct.
     */
    public void checkCloudDiskType() {
        if (cloudDiskSize > 0) {
            if (cloudDiskStorageType == null
                    || (!cloudDiskStorageType.equals(CLOUD_DISK_TYPE_HP1)
                    && !cloudDiskStorageType.equals(CLOUD_DISK_TYPE_CLOUD_HP1))) {
                throw new IllegalArgumentException("cloud disk storage type is wrong.");
            }
        }
    }

    /**
     * check whether auto renew time unit is correct.
     */
    public void checkAutoRenewTime() {
        if (autoRenewTime > 0) {
            if (autoRenewTimeUnit == null
                    || (!autoRenewTimeUnit.equals(AUTO_RENEW_TIME_UNIT_MONTH)
                    && !autoRenewTimeUnit.equals(AUTO_RENEW_TIME_UNIT_YEAR))) {
                throw new IllegalArgumentException("auto renew time unit is wrong");
            }
        }
    }

    public String getFlavor() {
        return flavor;
    }

    public void setFlavor(String flavor) {
        this.flavor = flavor;
    }

    public int getCpuCount() {
        return cpuCount;
    }

    public void setCpuCount(int cpuCount) {
        this.cpuCount = cpuCount;
    }

    public int getPurchaseCount() {
        return purchaseCount;
    }

    public void setPurchaseCount(int purchaseCount) {
        this.purchaseCount = purchaseCount;
    }

    public String getType() {
        return type;
    }

    public void setType(String type) {
        this.type = type;
    }

    public String getZoneName() {
        return zoneName;
    }

    public void setZoneName(String zoneName) {
        this.zoneName = zoneName;
    }

    public int getAutoRenewTime() {
        return autoRenewTime;
    }

    public void setAutoRenewTime(int autoRenewTime) {
        this.autoRenewTime = autoRenewTime;
    }

    public String getAutoRenewTimeUnit() {
        return autoRenewTimeUnit;
    }

    public void setAutoRenewTimeUnit(String autoRenewTimeUnit) {
        this.autoRenewTimeUnit = autoRenewTimeUnit;
    }

    public int getCloudDiskSize() {
        return cloudDiskSize;
    }

    public void setCloudDiskSize(int cloudDiskSize) {
        this.cloudDiskSize = cloudDiskSize;
    }

    public String getCloudDiskStorageType() {
        return cloudDiskStorageType;
    }

    public void setCloudDiskStorageType(String cloudDiskStorageType) {
        this.cloudDiskStorageType = cloudDiskStorageType;
    }

    public Billing getBilling() {
        return billing;
    }

    public void setBilling(Billing billing) {
        this.billing = billing;
    }

    @Override
    public CreateDccRequest withRequestCredentials(BceCredentials credentials) {
        this.setRequestCredentials(credentials);
        return this;
    }
}
