package cn.spatiotemporal.commons.file.service.impl;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.time.LocalDateTime;

import org.apache.commons.io.FileUtils;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Conditional;
import org.springframework.stereotype.Component;
import org.springframework.util.ResourceUtils;
import org.springframework.web.multipart.MultipartFile;

import cn.spatiotemporal.commons.file.service.FileService;
import cn.spatiotemporal.commons.file.support.FtpUploadCondition;
import cn.spatiotemporal.commons.file.util.FtpUtil;
import cn.spatiotemporal.commons.file.vo.FileInfo;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.io.FileUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.IdUtil;
import cn.hutool.core.util.StrUtil;
import lombok.Setter;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;

/**
 * 
 * @ClassName: FtpServiceImpl    
 * @Description: ftp实现
 * @date 2023年9月14日 上午11:30:23    
 *     
 * @author  Q.JI
 * @version  
 * @since   JDK 1.8
 * 
 * <p> 选用ftp配置示例
  uploadservice:
    #（minio/ftp）
    type: ftp   
    # ftp配置
    ftp:
      host: 192.168.59.181
      port: 21
      username: ftp
      password: ftp
      requestDir: upload
      encoding: GBK
 */

@Component("uploadFileServiceFtp")
@Conditional(FtpUploadCondition.class)
@ConfigurationProperties(prefix = "uploadservice.ftp")
@Slf4j
public class FtpServiceImpl implements FileService, InitializingBean {

    /**
     * ftp服务地址
     */
    @Setter
    private String host;

    /**
     * 端口
     */
    @Setter
    private String port;

    /**
     * 用户名
     */
    @Setter
    private String userName;
    
    /**
     * 密钥
     */
    @Setter
    private String password;

    /**
     * 请求路径
     */
    @Setter
    private String requestDir;

    /**
     * 自定义文件服务域名(非必须)
     */
    @Setter
    private String customDomain;
    
    private static String UPLOADED_FOLDER = getLocalStaticPath() + File.separator;

    @Override
    public void afterPropertiesSet() {
        log.info("Ftp Client init...");
        Assert.notBlank(host, "Ftp hot can not be null");
        Assert.notBlank(port, "Ftp port can not be null");
        Assert.notBlank(userName, "Ftp userName can not be null");
        Assert.notBlank(password, "Ftp password can not be null");
        Assert.notBlank(requestDir, "Ftp requestDir can not be null");
    }


    /**
     * 上传文件
     *
     * @param file 表单文件对象
     * @return
     */
    @Override
    @SneakyThrows
    public FileInfo uploadFile(MultipartFile file) { 
        // 生成文件名(日期文件夹)
        String suffix = FileUtil.getSuffix(file.getOriginalFilename());
        String uuid = IdUtil.simpleUUID();
        String location = DateUtil.format(LocalDateTime.now(), "yyyy/MM/dd");
        String fileName = uuid + "." + suffix;
        
//        String outputFile = "";
//        boolean result = File.separatorChar == '\\';
//        if(result) {
//        	outputFile = "c:" + File.separatorChar + "tmp" + File.separator + fileName;
//        }else {
//        	outputFile = File.separatorChar + "tmp" + File.separatorChar + fileName;
//        }
        
		String localLocation = location.replaceAll("\\/", "\\" + File.separator);
		File uploadOutFile = new File(UPLOADED_FOLDER + localLocation);
		if (!uploadOutFile.exists()) {
			uploadOutFile.mkdirs();
		}
		String outputFile = uploadOutFile.getAbsolutePath() + File.separator + fileName;

        InputStream inputStream = file.getInputStream();        
        FileUtils.copyInputStreamToFile(inputStream, new File(outputFile));
        
        // 文件上传
        FtpUtil.ftpPutFile(host, port, userName, password, new String[] {outputFile}, requestDir + "/" + location);
        FileUtils.deleteQuietly(new File(outputFile));

        // 返回文件路径
        String fileUrl;
        if (StrUtil.isBlank(customDomain)) { // 未配置自定义域名
            fileUrl = String.format("ftp://%s:%s/%s/%s", host, port, location, fileName);            
        } else { // 配置自定义文件路径域名
            fileUrl = String.format("%s/%s/s", customDomain, location, fileName);
        }

        FileInfo fileInfo = new FileInfo();
        fileInfo.setName(fileName);
        fileInfo.setUrl("/" + requestDir + "/" + location + "/" + fileName);
        fileInfo.setType(file.getContentType());
        fileInfo.setSize(file.getSize());
        fileInfo.setOriginalName(file.getOriginalFilename());
        return fileInfo;
    }

    /**
     * 删除文件
     *
     * @param filePath 文件路径
     *                 eg: /upload/2023/09/13/4b50e00c14524f8cadb14478f9c1ffbf.jpg
     *                 
     * @return
     */
    @Override
    @SneakyThrows
    public boolean deleteFile(String filePath) {
        Assert.notBlank(filePath, "删除文件路径不能为空");
        
        String fileSignimgPath = filePath.substring(0, filePath.lastIndexOf("/"));
        String overFileSignimgName = filePath.substring(filePath.lastIndexOf("/")+1);
        return FtpUtil.ftpDeleteFile(host, port, userName, password, 
        		fileSignimgPath, new String[]{overFileSignimgName}) == 0 ? true : false; 
    }
    
    
    /**
	 * 
	 * @Title: getLocalStaticPath
	 * @Description: 获得本地static资源路径
	 * @return String
	 */
	private static String getLocalStaticPath() {
		File path = null;
		try {
			path = new File(ResourceUtils.getURL("classpath:").getPath());
		} catch (FileNotFoundException e) {
			log.error("获取系统根目录发生异常", e);
		}
		if (!path.exists()) {
			path = new File("");
		}

		File upload = new File(path.getAbsolutePath(), "static");
		if (!upload.exists()) {
			upload.mkdirs();
		}
		return upload.getAbsolutePath();
	}

}
