package com.inzy.wsmock


import io.netty.bootstrap.ServerBootstrap
import io.netty.channel.ChannelInitializer
import io.netty.channel.ChannelOption
import io.netty.channel.nio.NioEventLoopGroup
import io.netty.channel.socket.SocketChannel
import io.netty.channel.socket.nio.NioServerSocketChannel
import io.netty.handler.codec.http.HttpObjectAggregator
import io.netty.handler.codec.http.HttpServerCodec
import io.netty.handler.codec.http.websocketx.WebSocketServerProtocolHandler
import io.netty.handler.stream.ChunkedWriteHandler
import jakarta.annotation.PostConstruct
import jakarta.annotation.PreDestroy
import org.springframework.beans.factory.annotation.Value
import org.springframework.stereotype.Component

@Component
class NettyWebSocketServer(
    @Value("\${netty.websocket.port:8089}") private val port: Int,
    private val webSocketHandler: WebSocketHandler
) {
    // Netty主从线程组
    private val bossGroup = NioEventLoopGroup(1)
    private val workerGroup = NioEventLoopGroup()

    /**
     * 启动Netty WebSocket服务
     */
    @PostConstruct
    fun start() {
        try {
            val bootstrap = ServerBootstrap()
            bootstrap.group(bossGroup, workerGroup)
                .channel(NioServerSocketChannel::class.java)
                .option(ChannelOption.SO_BACKLOG, 1024)
                .childOption(ChannelOption.SO_KEEPALIVE, true)
                .childHandler(object : ChannelInitializer<SocketChannel>() {
                    override fun initChannel(ch: SocketChannel) {
                        val pipeline = ch.pipeline()
                        // HTTP编解码器
                        pipeline.addLast(HttpServerCodec())
                        // 分块写处理器
                        pipeline.addLast(ChunkedWriteHandler())
                        // HTTP消息聚合器（最大1024*1024字节）
                        pipeline.addLast(HttpObjectAggregator(1024 * 1024))
                        // WebSocket协议处理器（指定WebSocket路径）
                        pipeline.addLast(WebSocketServerProtocolHandler("/gs-guide-websocket"))
                        // 自定义WebSocket消息处理器
                        pipeline.addLast(webSocketHandler)
                    }
                })
            // 绑定端口并启动
            val channelFuture = bootstrap.bind(port).sync()
            println("Netty WebSocket服务启动成功，端口：$port")
            // 阻塞直到服务关闭
            channelFuture.channel().closeFuture().sync()
        } catch (e: Exception) {
            e.printStackTrace()
        } finally {
            // 优雅关闭线程组
            bossGroup.shutdownGracefully()
            workerGroup.shutdownGracefully()
        }
    }

    /**
     * 销毁时关闭Netty服务
     */
    @PreDestroy
    fun stop() {
        bossGroup.shutdownGracefully()
        workerGroup.shutdownGracefully()
        println("Netty WebSocket服务已关闭")
    }
}