package com.inzy.wsmock


import org.springframework.context.annotation.Configuration
import org.springframework.http.MediaType
import org.springframework.messaging.converter.MappingJackson2MessageConverter
import org.springframework.messaging.simp.config.ChannelRegistration
import org.springframework.messaging.simp.config.MessageBrokerRegistry
import org.springframework.web.socket.config.annotation.EnableWebSocketMessageBroker
import org.springframework.web.socket.config.annotation.StompEndpointRegistry
import org.springframework.web.socket.config.annotation.WebSocketMessageBrokerConfigurer

/**
 * Kotlin 版 STOMP WebSocket 核心配置
 * @EnableWebSocketMessageBroker：启用 STOMP 协议 + 消息代理
 */
@Configuration
@EnableWebSocketMessageBroker // 核心注解：启用 STOMP + 消息代理
class StompWebSocketConfig : WebSocketMessageBrokerConfigurer {

    /**
     * 配置消息代理（广播/订阅的前缀）
     */
    override fun configureMessageBroker(config: MessageBrokerRegistry) {
        // 1. 启用内存消息代理（生产环境可替换为 RabbitMQ/ActiveMQ）
        // 2. 配置代理前缀：客户端订阅以 /topic/ 开头的主题（广播）、/queue/ 开头的队列（点对点）
        config.enableSimpleBroker("/topic", "/queue")

        // 3. 配置应用前缀：客户端发送消息的前缀（对应 @MessageMapping）
        // 客户端发送消息需用 /app/xxx，服务端 @MessageMapping("xxx") 接收
        config.setApplicationDestinationPrefixes("/app")

        // 4. 配置用户点对点前缀（可选）：客户端订阅 /user/queue/xxx 接收单人消息
        config.setUserDestinationPrefix("/user")
    }

    /**
     * 注册 STOMP 端点（客户端连接入口）
     */
    override fun registerStompEndpoints(registry: StompEndpointRegistry) {
        // 1. 注册 STOMP 端点：客户端通过 ws://localhost:8088/kotlin-stomp/ws/stomp 连接
        registry.addEndpoint("/gs-guide-websocket")
            .setAllowedOrigins("*") // 允许跨域（生产环境指定具体域名）

        // 2. 可选：支持 SockJS（兼容低版本浏览器，如不支持 WebSocket 的场景）
        registry.addEndpoint("/ws/stomp")
            .setAllowedOrigins("*")
            .withSockJS()
    }

    /**
     * 可选：配置消息通道拦截器（如权限验证、日志）
     */
    override fun configureClientInboundChannel(registration: ChannelRegistration) {
        // 可添加拦截器，比如验证客户端发送的消息是否有权限
        // registration.interceptors(StompAuthInterceptor())
    }
}
