package com.inzy.wsmock.push

import com.alibaba.fastjson2.JSON
import com.alibaba.fastjson2.JSONObject
import com.inzy.wsmock.ChannelManager
import com.inzy.wsmock.RequestParamHandler
import com.inzy.wsmock.bean.MRoutes
import com.inzy.wsmock.bean.V2xStartBean
import com.inzy.wsmock.bean.VObject
import com.inzy.wsmock.utils.FileIoUtil
import io.netty.channel.Channel
import io.netty.handler.codec.http.websocketx.TextWebSocketFrame
import jakarta.annotation.PostConstruct
import jakarta.annotation.PreDestroy
import kotlinx.coroutines.*
import org.slf4j.LoggerFactory
import org.springframework.core.io.ResourceLoader
import org.springframework.stereotype.Component
import java.util.concurrent.atomic.AtomicBoolean

@Component
//@Scope("prototype")
class AvpPushTask(
    // 注入Spring资源加载器（Spring自动装配）
    private val resourceLoader: ResourceLoader,
//    private val pushConfig: PushConfig,
) : PushTask {
    //    private val instanceId = System.currentTimeMillis()  // 实例唯一标识，用于验证多实例
    private val logger = LoggerFactory.getLogger(javaClass)


    // 注入ChannelManager单例
    private val channelManager = ChannelManager.instance

    //车辆位姿，泊车
    private var carDstPark = mutableListOf<String>()

    //车辆位姿，召车
    private var carDstCall = mutableListOf<String>()

    //车辆行驶的位置
    private var carPos = 0

    //AVP状态 park call
    private var status = ""

    //感知物
    private var preDst = mutableListOf<String>()


    //v2x只推一次开关
    private var isV2x = AtomicBoolean(false)

    @PostConstruct
    fun readMockFile() {
//        logger.info("创建 AdasPushTask 实例，ID: $instanceId")
        var classpathResource = resourceLoader.getResource("classpath:avpm/Car_fangzhen_park.txt")
        // 读取文件内容（Kotlin简化写法）
        CoroutineScope(Dispatchers.IO).launch {
            //读取泊车数据
            FileIoUtil.getMockToList(classpathResource.inputStream, carDstPark)
            //读取召车数据
            classpathResource = resourceLoader.getResource("classpath:avpm/Car_fangzhen_call.txt")
            FileIoUtil.getMockToList(classpathResource.inputStream, carDstCall)
            //读取感知物数据
            classpathResource = resourceLoader.getResource("classpath:avpm/PerTarget.txt")
            FileIoUtil.getMockToList(classpathResource.inputStream, preDst)
        }
    }

    @PreDestroy
    fun destroy() {
//        logger.info("销毁 AdasPushTask 实例，ID: $instanceId")
        carDstPark.clear()
        carDstCall.clear()
        preDst.clear()
    }

    override fun pushMsgFromType(type: String, channel: Channel) {
        when (type) {
            RequestParamHandler.PARAM_TYPE_VALUE_1 -> {//车辆位姿
                pushCarPos(channel)
            }

            RequestParamHandler.PARAM_TYPE_VALUE_2 -> {//感知物
                pushPre(channel)
            }

            RequestParamHandler.PARAM_TYPE_VALUE_3 -> {//v2x 预警
                pushV2x(channel)
            }

            else -> {
                val msg = JSONObject()
                msg.put("type", "avp")
                sendMsg(channel, msg.toJSONString())
            }
        }
    }

    /**
     * 推送车辆位姿
     */
    private fun pushCarPos(channel: Channel) {
        CoroutineScope(Dispatchers.Default).launch {
            delay(8000)
            status = "park"
            //开始推送泊车
            carDstPark.forEachIndexed { i, str ->
                val mRoutes = JSON.parseObject(str, MRoutes::class.java)
                mRoutes.status = status
                mRoutes.carPos = i
                sendMsg(channel, JSON.toJSONString(mRoutes))
                delay(300)
            }
            //泊车完成
            val parkTemp=MRoutes().apply {
                status = "park_com"
            }
            sendMsg(channel, JSON.toJSONString(parkTemp))
            delay(3000)
            status = "call"
            //开始推送召车
            carDstCall.forEachIndexed { i, str ->
                val mRoutes = JSON.parseObject(str, MRoutes::class.java)
                mRoutes.status = status
                mRoutes.carPos = i
                sendMsg(channel, JSON.toJSONString(mRoutes))
                delay(200)
            }
            //召车完成
            val callTemp=MRoutes().apply {
                status = "call_com"
            }
            sendMsg(channel, JSON.toJSONString(callTemp))
            //全部完成
            val temp=MRoutes().apply {
                status = ""
            }
            sendMsg(channel, JSON.toJSONString(temp))

            status = ""
        }
    }

    /**感知物**/
    private fun pushPre(channel: Channel) {
        CoroutineScope(Dispatchers.Default).launch {
            if (preDst.isNotEmpty()) {
                delay(2000)
                preDst.forEachIndexed { index, string ->
//                            println("index = ${index}")
                    sendMsg(channel, string)
                    delay(4500)
                }
                pushPre(channel)
            }
        }
    }

    private fun pushV2x(channel: Channel) {
        CoroutineScope(Dispatchers.Default).launch {
            while (isActive) {
                carPos = 8
                if (status.equals("park")) {
                    when (carPos) {
                        8 -> {//紧急制动预警
                            val v2x = V2xStartBean().apply {
                                type = 6
                                objects =
                                    listOf(VObject(""), VObject("f117fdfa-feff-0100-85dc-35850000acb0"))
                            }
                            pushV2xAtomic(channel, v2x)
                        }

                        30 -> {//前向碰撞
                            val v2x = V2xStartBean().apply {
                                type = 1
                                objects =
                                    listOf(VObject(""), VObject("f117fdfa-feff-0100-85dc-35850000acb0"))
                            }
                            pushV2xAtomic(channel, v2x)
                        }

                        60 -> {//弱势交通参与者
                            val v2x = V2xStartBean().apply {
                                type = 12
                                objects =
                                    listOf(VObject(""), VObject("50332456-3030-3030-3530-303334533955"))
                            }
                            pushV2xAtomic(channel, v2x)
                        }
                    }

                } else if (status.equals("call")) {
                    when (carPos) {
                        70 -> {//盲区预警
                            val v2x = V2xStartBean().apply {
                                type = 4
                                objects =
                                    listOf(VObject(""), VObject("2435"))
                            }
                            pushV2xAtomic(channel, v2x)
                        }
                    }
                }
                delay(300)
            }
        }
    }

    private fun pushV2xAtomic(channel: Channel, v2xStartBean: V2xStartBean, change: Boolean = true) {
        if (isV2x.compareAndSet(false, true)) {
//            val msgObj = JSONObject()
//            msgObj["content"] = "定时推送消息 type=2 ${LocalDateTime.now()}"
            val str = JSONObject.toJSONString(v2xStartBean)
            sendMsg(channel, str)
            if (change) {
                isV2x.set(false)
            }
        }
    }


    private fun sendMsg(channel: Channel, msg: String) {
        if (!channel.isActive()) {
//            println("无在线客户端，跳过推送")
            channelManager.removeChannel(channel)
            return
        }
        val clientId = channel.id().asShortText()
        val frame = TextWebSocketFrame(msg)
        channel.writeAndFlush(frame)
            .addListener { future ->
                if (!future.isSuccess) {
                    println("推送消息给客户端[$clientId]失败：${future.cause()?.message}")
                    channelManager.removeChannel(channel) // 推送失败移除失效Channel
                }
            }
    }


}